/*******************************************************************************
 * Copyright (c) 2006 IBM Corporation.
 * Copyright (c) 2007 Novell, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Michael McIntosh (IBM Corporation) - initial API and implementation
 *    Jim Sermersheim (Novell)
 *******************************************************************************/ 

package org.eclipse.higgins.sts.server.token.saml;

import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.cert.X509Certificate;
import java.util.Map;

import org.eclipse.higgins.configuration.api.ISettingDescriptor;
import org.eclipse.higgins.sts.api.IConstants;
import org.eclipse.higgins.sts.api.IRequestSecurityToken;
import org.eclipse.higgins.sts.api.ISTSRequest;
import org.eclipse.higgins.sts.api.ISTSResponse;
import org.eclipse.higgins.sts.common.Fault;

/**
 * Handle RSTs and generate RSTRs as SAML Assertions. 
 * Compatible with www.identityblog.com
 * 
 * @author mikemci at us dot ibm dot com
 */
public class TokenGeneratorHandler
	extends org.eclipse.higgins.sts.server.token.handler.TokenHandler
{
	private final org.eclipse.higgins.sts.utilities.LogHelper log = new org.eclipse.higgins.sts.utilities.LogHelper
		(TokenGeneratorHandler.class.getName());
	
	javax.xml.namespace.QName qnameIdentityClaimType = new javax.xml.namespace.QName
		(null,
		"ClaimType");
	javax.xml.namespace.QName qnameIdentityClaimURI = new javax.xml.namespace.QName
		(null,
		"Uri");
	
    private boolean bConfigured = false;
	
	/**
	 * Protected constructor, must use TokenGeneratorHandlerFactory
	 */
	protected TokenGeneratorHandler()
	{
		this.log.trace("TokenGeneratorHandler::TokenGeneratorHandler");
	}
	
    /* (non-Javadoc)
	 * @see org.eclipse.higgins.sts.IExtension#configure(java.util.Hashtable)
	 */
	public void configure
		(final Map mapGlobalSettings,
		final String strComponentName,
		final Map mapComponentSettings,
		final ISettingDescriptor componentDescriptor,
		final ISettingDescriptor globalDescriptor)
	{
		this.log.trace("TokenGeneratorHandler::initialize");

		this.bConfigured = true;
	}

	/**
	 * Invoked to add each claim to the AttributeStatement
	 * 
	 * @param constants  an IConstants containing the URIs appropriate for the request
	 * @param elemAttributeStatement the AttrobuteStatement element that will contain the added claim
     * @param uriClaim the URI for the claim to be added
	 * @param strClaimName the name for the claim to be added
	 * @param strClaimValue the value for the claim to be added
	 */
  	private void AddIdentityClaim
  		(final org.eclipse.higgins.sts.api.IConstants constants,
  		final boolean bSAML20,
  		final org.apache.axiom.om.OMFactory omFactory,
  		final org.apache.axiom.om.OMElement omAttributeStatement,
  		final org.apache.axiom.om.OMNamespace omIdentityNamespace,
  		final org.apache.axiom.om.OMNamespace omSAMLAssertionNamespace,
  		final org.eclipse.higgins.sts.api.IDisplayToken dtDisplayToken,
  		final org.eclipse.higgins.sts.api.IClaim claim,
  		final java.util.Map mapAttributeClaim)
  	{
  		final org.eclipse.higgins.sts.api.IClaimType claimType = claim.getType();
  		final java.net.URI uriClaimName = claimType.getName();
  		final String strClaimName = uriClaimName.toString();

  		this.log.trace("Adding Identity Claim URI: " + strClaimName);

  		final java.util.Map mapClaim = (java.util.Map)mapAttributeClaim.get(strClaimName);

  		final String strAttributeName = (String)mapClaim.get("ClaimName");
  		final java.net.URI uriAttributeNamespace = (java.net.URI)mapClaim.get("AttributeNamespace");
  		final String strDisplayName = (String)mapClaim.get("DisplayName");

  		this.log.trace("Adding Identity Claim Name: " + strAttributeName);
  		if (null == strAttributeName)
  		{
  			this.log.error("Unsupported claim");
  			return;
  		}

  		final org.apache.axiom.om.OMElement elemAttribute = omFactory.createOMElement
  			("Attribute",
  			omSAMLAssertionNamespace,
  			omAttributeStatement);

  		final org.eclipse.higgins.sts.api.IDisplayClaim dcDisplayClaim = new org.eclipse.higgins.sts.common.DisplayClaim();
  		dtDisplayToken.getDisplayClaimList().add(dcDisplayClaim);
  		dcDisplayClaim.setType(claimType.getName());
  		dcDisplayClaim.setDisplayTag(strDisplayName);

  		if (bSAML20)
  		{
	  		elemAttribute.addAttribute
				("Name",
				strAttributeName,
				null);
			elemAttribute.addAttribute
				("NameFormat",
				uriAttributeNamespace.toString(),
				null);  
			elemAttribute.addAttribute
				("FriendlyName",
				strDisplayName,
				null);			
  		}
  		else
  		{
	  		elemAttribute.addAttribute
				("AttributeName",
				strAttributeName,
				null);
			elemAttribute.addAttribute
				("AttributeNamespace",
				uriAttributeNamespace.toString(),
				null);
  		}

  		try
  		{
 			final java.util.Iterator iterValue = claim.getValues();
  			while (iterValue.hasNext())
  			{
				final String strValue = (String)iterValue.next();
				this.log.trace("Adding Claim Value: " + strValue);

	  			dcDisplayClaim.setDisplayValue(strValue);

		  		final org.apache.axiom.om.OMElement elemAttributeValue = omFactory.createOMElement
					("AttributeValue",
					omSAMLAssertionNamespace,
					elemAttribute);
		  		elemAttributeValue.setText(strValue);
 			}
		}
 		catch (final Exception e)
  		{
    		org.eclipse.higgins.sts.utilities.ExceptionHelper.Log
				(this.log,
				e);	
  		}
  	}
  	
	/**
	 * Invoked to add a PPID claim to the AttributeStatement
	 * 
	 * @param constants  an IConstants containing the URIs appropriate for the request
	 * @param elemAttributeStatement the AttrobuteStatement element that will contain the added claim
     * @param uriClaim the URI for the claim to be added
	 * @param strClaimName the name for the claim to be added
	 * @param strClaimValue the value for the claim to be added
	 */
  	private void AddPrivatePersonalIdentityClaim
  		(final org.eclipse.higgins.sts.api.IConstants constants,
  		final IRequestSecurityToken RST,
  		final boolean bSAML20,
  		final org.apache.axiom.om.OMFactory omFactory,
  		final org.apache.axiom.om.OMElement omAttributeStatement,
  		final org.apache.axiom.om.OMNamespace omIdentityNamespace,
  		final org.apache.axiom.om.OMNamespace omSAMLAssertionNamespace,
  		final org.eclipse.higgins.sts.api.IDisplayToken dtDisplayToken,
  		final org.eclipse.higgins.sts.api.IClaim claim,
  		final java.util.Map mapAttributeClaim)
  	{
  		final org.eclipse.higgins.sts.api.IClaimType claimType = claim.getType();
  		final java.net.URI uriClaimName = claimType.getName();
  		final String strClaimName = uriClaimName.toString();

  		this.log.trace("Adding Identity Claim URI: " + strClaimName);

  		final java.util.Map mapClaim = (java.util.Map)mapAttributeClaim.get(strClaimName);

  		final String strAttributeName = (String)mapClaim.get("ClaimName");
  		final java.net.URI uriAttributeNamespace = (java.net.URI)mapClaim.get("AttributeNamespace");
  		final String strDisplayName = (String)mapClaim.get("DisplayName");

  		this.log.trace("Adding Identity Claim Name: " + strAttributeName);
  		if (null == strAttributeName)
  		{
  			this.log.error("Unsupported claim");
  			return;
  		}
  		final org.apache.axiom.om.OMElement elemAttribute = omFactory.createOMElement
  			("Attribute",
  			omSAMLAssertionNamespace,
  			omAttributeStatement);
  		final org.eclipse.higgins.sts.api.IDisplayClaim dcDisplayClaim = new org.eclipse.higgins.sts.common.DisplayClaim();
  		dtDisplayToken.getDisplayClaimList().add(dcDisplayClaim);
  		dcDisplayClaim.setType(claimType.getName());
  		dcDisplayClaim.setDisplayTag(strDisplayName);
 
  		if (bSAML20)
  		{
	  		elemAttribute.addAttribute
				("Name",
				strAttributeName,
				null);
			elemAttribute.addAttribute
				("NameFormat",
				uriAttributeNamespace.toString(),
				null); 
			elemAttribute.addAttribute
				("FriendlyName",
				strDisplayName,
				null);
  		}
  		else
  		{
	  		elemAttribute.addAttribute
				("AttributeName",
				strAttributeName,
				null);
			elemAttribute.addAttribute
				("AttributeNamespace",
				uriAttributeNamespace.toString(),
				null);
  		}
  		
  		try
  		{
	   		final org.eclipse.higgins.sts.api.IElement elemClientPseudonym = RST.getClientPseudonym();
	   		String strPPID = "BOGUS";
	   		if (null != elemClientPseudonym)
	   		{
		   		final org.apache.axiom.om.OMElement omClientPseudonym = (org.apache.axiom.om.OMElement)elemClientPseudonym.getAs
	   				(org.apache.axiom.om.OMElement.class);
		   		final java.util.Iterator iterPPID = omClientPseudonym.getChildElements();
		   		final org.apache.axiom.om.OMElement omPPID = (org.apache.axiom.om.OMElement)iterPPID.next();
		   		strPPID = omPPID.getText();
	   		}
			this.log.trace("Adding Claim Value: " + strPPID);
  			dcDisplayClaim.setDisplayValue(strPPID);
	  		final org.apache.axiom.om.OMElement elemAttributeValue = omFactory.createOMElement
				("AttributeValue",
				omSAMLAssertionNamespace,
				elemAttribute);
	  		elemAttributeValue.setText(strPPID);
		}
 		catch (final Exception e)
  		{
    		org.eclipse.higgins.sts.utilities.ExceptionHelper.Log
				(this.log,
				e);	
  		}
  	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.sts.IExtension#invoke
     */
	public void invoke
		(final java.util.Map mapGlobalSettings,
		final String strComponentName,
		final java.util.Map mapComponentSettings,
		final java.util.Map mapInvocationSettings,
		final IConstants constants,
		final ISTSRequest request,
		final ISTSResponse response)
	{
		this.log.trace("TokenGeneratorHandler::invoke: " + strComponentName);
		
		if (!this.bConfigured)
		{
			final Fault fault = new Fault
				(constants.getWSTrustNamespace(),
				"wst",
				constants.getRequestFailedFaultCode(),
				"The specified request failed",
				"Issue handler not configured");
			response.setFault(fault);
			return;		
		}
		
		final org.eclipse.higgins.sts.spi.IXMLSecurityExtension xmlSecurity = (org.eclipse.higgins.sts.spi.IXMLSecurityExtension)mapGlobalSettings.get
			("XMLSecurityExtension");
		
		final java.net.URI uriDefaultKeyType = (java.net.URI)mapComponentSettings.get
    		("DefaultKeyType");
		this.log.trace("DefaultKeyType: " + uriDefaultKeyType.toString());
		
		final java.lang.Boolean bIncludeBearerSubjectName = (java.lang.Boolean)mapComponentSettings.get
    		("IncludeBearerSubjectName");
		this.log.trace("IncludeBearerSubjectName: " + bIncludeBearerSubjectName.toString());
		
		final java.net.URI uriTokenIssuer = (java.net.URI)mapComponentSettings.get
    		("TokenIssuer");
		this.log.trace("TokenIssuer: " + uriTokenIssuer.toString());
		if (null == uriTokenIssuer)
		{
			final Fault fault = new Fault
				(constants.getWSTrustNamespace(),
				"wst",
				constants.getRequestFailedFaultCode(),
				"The specified request failed",
				"TokenIssuer not set.");
			response.setFault(fault);
			return;
		}

		final java.net.URI uriSubjectNameIdentifier = (java.net.URI)mapComponentSettings.get
    		("SubjectNameIdentifierAttribute");
		if (null != uriSubjectNameIdentifier)
			this.log.trace("SubjectNameIdentifier: " + uriSubjectNameIdentifier.toString());
		
		final java.net.URI uriSubjectNameIdentifierFormat = (java.net.URI)mapComponentSettings.get
			("SubjectNameIdentifierFormat");
		if (null != uriSubjectNameIdentifierFormat)
			this.log.trace("SubjectNameIdentifierFormat: " + uriSubjectNameIdentifierFormat.toString());

		final java.lang.Boolean bEncryptToken = (java.lang.Boolean)mapComponentSettings.get
    		("EncryptToken");
		this.log.trace("EncryptToken: " + bEncryptToken.toString());
	
		final java.util.Map mapAttributeClaim = (java.util.Map)mapGlobalSettings.get
			("AttributeClaimMap");
		
		if (null == mapAttributeClaim)
		{
			final Fault fault = new Fault
				(constants.getWSTrustNamespace(),
				"wst",
				constants.getRequestFailedFaultCode(),
				"The specified request failed",
				"AttributeClaimMap not set.");
			response.setFault(fault);
	   		return;
		}
		
		// TODO: Properly namespace qualify LocalNames throughout
		final java.util.List listRST = request.getRequestSecurityTokenCollection();
		final IRequestSecurityToken RST = (IRequestSecurityToken)listRST.get(0);
		// TODO: Deal with collections?
		
		
		
		final java.net.URI uriTokenType = RST.getTokenType();
		final org.eclipse.higgins.sts.api.IAppliesTo appliesToRequest = RST.getAppliesTo();
		final org.eclipse.higgins.sts.api.ILifetime ltLifetime = RST.getLifetime();

		java.net.URI uriKeyType = RST.getKeyType();
		if (null == uriKeyType)
		{
			if (null == uriDefaultKeyType) {
				uriKeyType = constants.getNoProofKeyKeyType();
			} else {
				uriKeyType = uriDefaultKeyType;
			}
		}
		String strSubjectNameIdentifier = null;
		boolean bBearerToken = uriKeyType.equals(constants.getNoProofKeyKeyType());
 		
		org.eclipse.higgins.sts.api.IEndpointReference eprAppliesTo = null;
		java.net.URI uriAppliesTo = null;
		this.log.trace("Checking for AppliesTo");
		if (appliesToRequest != null)
		{
			this.log.trace("Found AppliesTo");
			eprAppliesTo = appliesToRequest.getEndpointReference();
			uriAppliesTo = eprAppliesTo.getAddress();
		}
		
		final X509Certificate certificateIssuer = (X509Certificate)mapGlobalSettings.get
			("IssuerCertificate");
		PublicKey publicKeyIssuer = null;
		PrivateKey privateKeyIssuer = null;
		if (uriTokenIssuer.equals(constants.getIssuerSelf()))
		{
			java.security.KeyPair kpSelf = request.getSelfSigningKeyPair();
			this.log.trace("KeyPair Class: " + kpSelf.getClass().getName());
			if (null == kpSelf)
			{
				final Fault fault = new Fault
					(constants.getWSTrustNamespace(),
					"wst",
					constants.getRequestFailedFaultCode(),
					"The specified request failed",
					"Self Signing KeyPair not set.");
				response.setFault(fault);
		   		return;
			}
			publicKeyIssuer = kpSelf.getPublic();
			this.log.trace("PublicKey Class: " + publicKeyIssuer.getClass().getName());
			if (null == publicKeyIssuer)
			{
				final Fault fault = new Fault
					(constants.getWSTrustNamespace(),
					"wst",
					constants.getRequestFailedFaultCode(),
					"The specified request failed",
					"Self Signing PublicKey not set.");
				response.setFault(fault);
		   		return;
			}
			privateKeyIssuer = kpSelf.getPrivate();
			this.log.trace("PrivateKey Class: " + privateKeyIssuer.getClass().getName());
			if (null == privateKeyIssuer)
			{
				final Fault fault = new Fault
					(constants.getWSTrustNamespace(),
					"wst",
					constants.getRequestFailedFaultCode(),
					"The specified request failed",
					"Self Signing PrivateKey not set.");
				response.setFault(fault);
		   		return;
			}	
		}
		else
		{
			publicKeyIssuer = certificateIssuer.getPublicKey();
			privateKeyIssuer = (PrivateKey)mapGlobalSettings.get
				("IssuerPrivateKey");
			if (null == certificateIssuer)
			{
				final Fault fault = new Fault
					(constants.getWSTrustNamespace(),
					"wst",
					constants.getRequestFailedFaultCode(),
					"The specified request failed",
					"IssuerCertificate not set.");
				response.setFault(fault);
		   		return;
			}
			if (null == privateKeyIssuer)
			{
				final Fault fault = new Fault
					(constants.getWSTrustNamespace(),
					"wst",
					constants.getRequestFailedFaultCode(),
					"The specified request failed",
					"IssuePrivateKey not set.");
				response.setFault(fault);
		   		return;
			}
		}
	
		final java.net.URI uriResponseID = org.eclipse.higgins.sts.utilities.UUIDHelper.getAsURI();
		final java.net.URI uriAssertionID = org.eclipse.higgins.sts.utilities.UUIDHelper.getAsURI();
		final java.util.Date dateNow = new java.util.Date();
		final java.util.Date dateLater = new java.util.Date(dateNow.getTime() + (1000*60*60*24*7));
		final String strNow = org.eclipse.higgins.sts.utilities.DateHelper.getIso8601Date(dateNow);
		final String strLater = org.eclipse.higgins.sts.utilities.DateHelper.getIso8601Date(dateLater);
		final org.apache.axiom.om.OMFactory omFactory = org.apache.axiom.om.OMAbstractFactory.getOMFactory();
		
		final org.apache.axiom.om.OMNamespace omIdentityNamespace = omFactory.createOMNamespace
			(constants.getIdentityNamespace().toString(),
			"ic");
		final org.apache.axiom.om.OMNamespace omWSTrustNamespace = omFactory.createOMNamespace
			(constants.getWSTrustNamespace().toString(),
			"wst");
		final org.apache.axiom.om.OMNamespace omDSNamespace = omFactory.createOMNamespace
			(constants.getXMLSignatureNamespace().toString(),
			"ds");

		final org.apache.axiom.om.OMElement omRequestedAttachedReference = omFactory.createOMElement
			("RequestedAttachedReference",
			omWSTrustNamespace);
		final org.apache.axiom.om.OMElement omRequestedUnattachedReference = omFactory.createOMElement
			("RequestedUnattachedReference",
			omWSTrustNamespace);
		
		final org.eclipse.higgins.sts.api.IDisplayToken dtRequestedDisplayToken = new org.eclipse.higgins.sts.common.DisplayToken();
		/*
		final org.apache.axiom.om.OMElement omRequestedDisplayToken = omFactory.createOMElement
			("RequestedDisplayToken",
			omIdentityNamespace);
		final org.apache.axiom.om.OMElement omDisplayToken = omFactory.createOMElement
			("DisplayToken",
			omIdentityNamespace,
			omRequestedDisplayToken);
		*/
		final org.apache.axiom.om.OMNamespace omWSSNamespace = omFactory.createOMNamespace
			(constants.getWSSecurityNamespace().toString(),
			"wsse");
		final org.apache.axiom.om.OMElement omSecurityTokenReference1 = omFactory.createOMElement
			("SecurityTokenReference",
			omWSSNamespace,
			omRequestedAttachedReference);
		final org.apache.axiom.om.OMElement omSecurityTokenReference2 = omFactory.createOMElement
			("SecurityTokenReference",
			omWSSNamespace,
			omRequestedUnattachedReference);
		final org.apache.axiom.om.OMElement omKeyIdentifier1 = omFactory.createOMElement
			("KeyIdentifier",
			omWSSNamespace,
			omSecurityTokenReference1);
		final org.apache.axiom.om.OMElement omKeyIdentifier2 = omFactory.createOMElement
			("KeyIdentifier",
			omWSSNamespace,
			omSecurityTokenReference2);
		// Revisit this KeyReference for WSS 1.1 / SAML 2.0
		omKeyIdentifier1.addAttribute
			("ValueType", 
			"http://docs.oasis-open.org/wss/oasis-wss-saml-token-profile-1.0#SAMLAssertionID",
			null);
		omKeyIdentifier1.setText(uriAssertionID.toASCIIString());
		omKeyIdentifier2.addAttribute
			("ValueType", 
			"http://docs.oasis-open.org/wss/oasis-wss-saml-token-profile-1.0#SAMLAssertionID",
			null);
		omKeyIdentifier2.setText(uriAssertionID.toASCIIString());
		org.apache.axiom.om.OMElement omRequestedSecurityToken = omFactory.createOMElement
			("RequestedSecurityToken",
			omWSTrustNamespace);
		
		java.net.URI uriSAMLAssertionNamespace = null;
		java.net.URI uriSAMLProtocolNamespace = null;
		// Check to see whether this is SAML 2.0 or 1.1
		boolean bSAML20 = uriTokenType.equals(constants.getSAML20TokenType())
			|| uriTokenType.equals(constants.getSAML20AssertionNamespace())
			|| uriTokenType.equals(constants.getSAML20ProtocolNamespace());
		// Check to see whether this is a request for an Assertion or a Response
		boolean bResponseToken = uriTokenType.equals(constants.getSAML20ProtocolNamespace());
		if (bSAML20)
		{
			uriSAMLAssertionNamespace = constants.getSAML20AssertionNamespace();
			uriSAMLProtocolNamespace = constants.getSAML20ProtocolNamespace();
		}
		else
		{
			uriSAMLAssertionNamespace = constants.getSAML10AssertionNamespace();
			uriSAMLProtocolNamespace = constants.getSAML10ProtocolNamespace();
		}
		final org.apache.axiom.om.OMNamespace omSAMLAssertionNamespace = omFactory.createOMNamespace
			(uriSAMLAssertionNamespace.toString(),
			"saml");
		org.apache.axiom.om.OMNamespace omSAMLProtocolNamespace = omFactory.createOMNamespace
			(uriSAMLProtocolNamespace.toString(),
			"samlp");
		org.apache.axiom.om.OMElement omResponse = null;
		org.apache.axiom.om.OMElement omAssertion = null;
		if (bResponseToken)
		{
			omResponse = omFactory.createOMElement
				("Response",
				omSAMLProtocolNamespace,
				omRequestedSecurityToken);
			omResponse.declareNamespace(omSAMLProtocolNamespace);
			omResponse.declareNamespace(omSAMLAssertionNamespace);			
			omAssertion = omFactory.createOMElement
				("Assertion",
				omSAMLAssertionNamespace,
				omResponse);
		}
		else
		{
			omAssertion = omFactory.createOMElement
				("Assertion",
				omSAMLAssertionNamespace,
				omRequestedSecurityToken);
		}
		if (bSAML20)
		{
			if (bResponseToken)
			{
				omResponse.addAttribute
					("Version",
					"2.0",
					null);
				omResponse.addAttribute
					("ID",
					uriResponseID.toASCIIString(),
					null);
				final org.apache.axiom.om.OMElement omResponseIssuer = omFactory.createOMElement
					("Issuer",
					omSAMLAssertionNamespace,
					omResponse);
				omResponseIssuer.setText
					(uriTokenIssuer.toString());
				final org.apache.axiom.om.OMElement omStatus =  omFactory.createOMElement
					("Status",
					omSAMLProtocolNamespace,
					omResponse);
				final org.apache.axiom.om.OMElement omStatusCode =  omFactory.createOMElement
					("StatusCode",
					omSAMLProtocolNamespace,
					omStatus);
				omStatusCode.addAttribute
					("Value",
					constants.getSAML20StatusCodeSuccess().toString(),
					null);
			}
			omAssertion.addAttribute
				("Version",
				"2.0",
				null);
			omAssertion.addAttribute
				("ID",
				uriAssertionID.toASCIIString(),
				null);
			final org.apache.axiom.om.OMElement omAssertionIssuer = omFactory.createOMElement
				("Issuer",
				omSAMLAssertionNamespace,
				omAssertion);
			omAssertionIssuer.setText
				(uriTokenIssuer.toString());
		}
		else
		{
			omAssertion.addAttribute
				("MajorVersion",
				"1",
				null);
			omAssertion.addAttribute
				("MinorVersion",
				"1",
				null);
			omAssertion.addAttribute
				("Issuer",
				uriTokenIssuer.toString(),
				null);
			omAssertion.addAttribute
				("AssertionID",
				uriAssertionID.toASCIIString(),
				null);
		}
		
		if (bResponseToken)
		{
			omResponse.addAttribute
				("IssueInstant",
				strNow.toString(),
				null);
			if (null != uriAppliesTo)
			{
				omResponse.addAttribute
					("Destination",
					uriAppliesTo.toString(),
					null);
			}
		}
		omAssertion.addAttribute
			("IssueInstant",
			strNow.toString(),
			null);
		
		final org.apache.axiom.om.OMElement omConditions = omFactory.createOMElement
			("Conditions",
			omSAMLAssertionNamespace,
			omAssertion);

		omConditions.addAttribute("NotBefore", strNow.toString(), null);
		omConditions.addAttribute("NotOnOrAfter", strLater.toString(), null);
		
		if (null != uriAppliesTo)
		{
			org.apache.axiom.om.OMElement omAudienceRestriction = null;
			
			if (bSAML20)
			{
				omAudienceRestriction = omFactory.createOMElement
					("AudienceRestriction",
					omSAMLAssertionNamespace,
					omConditions);
			}
			else
			{
				omAudienceRestriction = omFactory.createOMElement
					("AudienceRestrictionCondition",
					omSAMLAssertionNamespace,
					omConditions);
			}
			
			final org.apache.axiom.om.OMElement omAudience = omFactory.createOMElement
				("Audience",
				omSAMLAssertionNamespace,
				omAudienceRestriction);
			omAudience.setText(uriAppliesTo.toString());
		}
		
		// TODO: Deal with complete content of request (rather than assuming all SAML per Kim's Blog)
		final org.apache.axiom.om.OMElement omAttributeStatement = omFactory.createOMElement
			("AttributeStatement",
			omSAMLAssertionNamespace,
			omAssertion);
		final org.apache.axiom.om.OMElement omSubject = omFactory.createOMElement
			("Subject",
			omSAMLAssertionNamespace,
			omAttributeStatement);
		
		final org.eclipse.higgins.sts.api.IDigitalIdentity digitalIdentity = RST.getDigitalIdentity();
		if (null == digitalIdentity)
	   	{
			final Fault fault = new Fault
				(constants.getWSTrustNamespace(),
				"wst",
				constants.getRequestFailedFaultCode(),
				"The specified request failed",
				"Node was not found");
			response.setFault(fault);
			return;
	   	}
		
		if (digitalIdentity.getIncludePPIClaim())
		{
			final org.eclipse.higgins.sts.api.IClaimType claimType = new org.eclipse.higgins.sts.common.ClaimType();
			claimType.setName(constants.getIdentityClaimPrivatePersonalIdentifier());
			final org.eclipse.higgins.sts.api.IClaim claim = new org.eclipse.higgins.sts.common.Claim();
			claim.setType(claimType);
	  		this.AddPrivatePersonalIdentityClaim
	  			(constants,
	  			RST,
	  			bSAML20,
	  	  		omFactory,
	  	  		omAttributeStatement,
	  	  		omIdentityNamespace,
	  	  		omSAMLAssertionNamespace,
	  			dtRequestedDisplayToken,
	  			claim,
	  			mapAttributeClaim);
		}
		
		final java.util.List listClaims = digitalIdentity.getClaims();

		for (int i = 0; i < listClaims.size(); ++i)
		{
			final org.eclipse.higgins.sts.api.IClaim claim = (org.eclipse.higgins.sts.api.IClaim)listClaims.get(i);
	  		this.AddIdentityClaim
	  			(constants,
	  			//RST,
	  			bSAML20,
	  	  		omFactory,
	  	  		omAttributeStatement,
	  	  		omIdentityNamespace,
	  	  		omSAMLAssertionNamespace,
	  	  		dtRequestedDisplayToken,
	  			claim,
	  			mapAttributeClaim);
		}
		
		if ((!bBearerToken) || (bIncludeBearerSubjectName.booleanValue()))
		{
			final org.eclipse.higgins.sts.api.IClaimType claimType = new org.eclipse.higgins.sts.common.ClaimType();
			claimType.setName(uriSubjectNameIdentifier);
			final org.eclipse.higgins.sts.api.IClaim claim = digitalIdentity.getClaim(uriSubjectNameIdentifier);
			if (null != claim)
			{
				final java.util.Iterator iterClaimValue = claim.getValues();
				while (iterClaimValue.hasNext())
				{
					strSubjectNameIdentifier = (String)iterClaimValue.next();
					break;
	  			}
	  		}
			if (null == strSubjectNameIdentifier)
			{
				this.log.error("Node does not contain Subject NameIdentifier Claim: " + uriSubjectNameIdentifier.toString());
				final Fault fault = new Fault
					(constants.getWSTrustNamespace(),
					"wst",
					constants.getRequestFailedFaultCode(),
					"The specified request failed",
					"Node does not contain Subject NameIdentifier Claim: " + uriSubjectNameIdentifier.toString());
				response.setFault(fault);
				return;	
			}
			
			org.apache.axiom.om.OMElement omNameIdentifier = null;
			if (bSAML20)
			{
				omNameIdentifier = omFactory.createOMElement
					("NameID",
					omSAMLAssertionNamespace,
					omSubject);		
			}
			else
			{
				omNameIdentifier = omFactory.createOMElement
					("NameIdentifier",
					omSAMLAssertionNamespace,
					omSubject);
			}
			
			omNameIdentifier.addAttribute
				("Format",
				uriSubjectNameIdentifierFormat.toString(),
				null);
			omNameIdentifier.setText
				(strSubjectNameIdentifier);
		}
		final org.apache.axiom.om.OMElement omSubjectConfirmation = omFactory.createOMElement
	   		("SubjectConfirmation",
	   		omSAMLAssertionNamespace,
			omSubject);
		
		org.apache.axiom.om.OMElement omConfirmation = null;
		if (bSAML20)
		{
			if (bBearerToken)
		   	{
				omSubjectConfirmation.addAttribute("Method", constants.getSAML10ConfirmationMethodBearer().toString(), null);
		   	}
			else
			{
				omSubjectConfirmation.addAttribute("Method", constants.getSAML20ConfirmationMethodHolderOfKey().toString(), null);
			}

			org.apache.axiom.om.OMElement omSubjectConfirmationData = omFactory.createOMElement
		   		("SubjectConfirmationData",
		   		omSAMLAssertionNamespace,
				omSubjectConfirmation);
			omConfirmation = omSubjectConfirmationData;
			if (null != uriAppliesTo)
			{
				if (bSAML20)
				{
					omSubjectConfirmationData.addAttribute
						("Recipient",
						uriAppliesTo.toString(),
						null);
				}
			}
		}
		else
		{
			org.apache.axiom.om.OMElement omConfirmationMethod = omFactory.createOMElement
		   		("ConfirmationMethod",
		   		omSAMLAssertionNamespace,
				omSubjectConfirmation);
		   	
			if (bBearerToken)
		   	{
				omConfirmationMethod.setText(constants.getSAML10ConfirmationMethodBearer().toString());
		   	}
		   	else
		   	{
		   		omConfirmationMethod.setText(constants.getSAML10ConfirmationMethodHolderOfKey().toString());
		   	}
			omConfirmation = omSubjectConfirmation;
		}
		if (!bBearerToken)
		{
			try
			{
			   	final org.apache.axiom.om.OMElement omKeyInfo = omFactory.createOMElement
		   			("KeyInfo",
		   			omDSNamespace,
		   			omConfirmation);
			   	final org.apache.axiom.om.OMElement omKeyValue = omFactory.createOMElement
					("KeyValue",
					omDSNamespace,
					omKeyInfo);
			   	final org.apache.axiom.om.OMElement omRSAKeyValue = omFactory.createOMElement
					("RSAKeyValue",
					omDSNamespace,
					omKeyValue);
			    final java.security.interfaces.RSAPublicKey rsaPublicKey = (java.security.interfaces.RSAPublicKey)publicKeyIssuer;
			    final java.math.BigInteger biModulus = rsaPublicKey.getModulus();
			   	String strModulus = new String
			   		(org.apache.commons.codec.binary.Base64.encodeBase64
			   			(biModulus.toByteArray()));
			   	strModulus = org.eclipse.higgins.sts.utilities.XMLHelper.stripNewLinesFromString
			   		(strModulus);
			    final java.math.BigInteger biExponent = rsaPublicKey.getPublicExponent();
			   	String strExponent = new String
		   			(org.apache.commons.codec.binary.Base64.encodeBase64
		   				(biExponent.toByteArray()));
			   	strExponent = org.eclipse.higgins.sts.utilities.XMLHelper.stripNewLinesFromString
			   		(strExponent);
			   	final org.apache.axiom.om.OMElement omModulus = omFactory.createOMElement
					("Modulus",
					omDSNamespace,
					omRSAKeyValue);
			   	omModulus.setText
			   		(strModulus);
			   	final org.apache.axiom.om.OMElement omExponent = omFactory.createOMElement
					("Exponent",
					omDSNamespace,
					omRSAKeyValue);
			   	omExponent.setText
			   		(strExponent);
	   		}
		    catch (final Exception e)
		    {
	    		org.eclipse.higgins.sts.utilities.ExceptionHelper.Log
					(this.log,
					e);
				final Fault fault = new Fault
					(constants.getWSTrustNamespace(),
					"wst",
					constants.getRequestFailedFaultCode(),
					"The specified request failed",
					"XML Signature operation failed.");
				response.setFault(fault);
	    		return;	
		    }
		}
		
		org.eclipse.higgins.sts.api.IElement elemSignedRequestedSecurityToken = null;
		try
		{
			org.eclipse.higgins.sts.api.IElement elemRequestedSecurityToken = new org.eclipse.higgins.sts.common.Element();
			elemRequestedSecurityToken.set(omRequestedSecurityToken);
			elemSignedRequestedSecurityToken = xmlSecurity.SignEnveloped
	    		("saml:Assertion",
	    		elemRequestedSecurityToken,
	    		uriAssertionID.toASCIIString(),
	    		privateKeyIssuer,
	    		publicKeyIssuer,
	    		constants);
		}
	    catch (final Exception e)
	    {
    		org.eclipse.higgins.sts.utilities.ExceptionHelper.Log
				(this.log,
				e);
			final Fault fault = new Fault
				(constants.getWSTrustNamespace(),
				"wst",
				constants.getRequestFailedFaultCode(),
				"The specified request failed",
				"XML Signature operation failed.");
			response.setFault(fault);
    		return;	
	    }
	    
		final java.util.List listRSTR = response.getRequestSecurityTokenResponseCollection();
		if (0 == listRSTR.size())
		{
			listRSTR.add(new org.eclipse.higgins.sts.common.RequestSecurityTokenResponse());
		}
		final org.eclipse.higgins.sts.api.IRequestSecurityTokenResponse RSTR = (org.eclipse.higgins.sts.common.RequestSecurityTokenResponse)listRSTR.get(0);
		try
		{
			RSTR.setTokenType
				(uriTokenType);
			RSTR.setLifetime
				(ltLifetime);
			RSTR.setRequestedSecurityToken
				(elemSignedRequestedSecurityToken);
			RSTR.setRequestedAttachedReference
				(org.eclipse.higgins.sts.utilities.XMLHelper.toElement(omRequestedAttachedReference));
			RSTR.setRequestedUnattachedReference
				(org.eclipse.higgins.sts.utilities.XMLHelper.toElement(omRequestedUnattachedReference));
			RSTR.setRequestedDisplayToken
				(dtRequestedDisplayToken);
		}
		catch (final Exception e)
		{
    		org.eclipse.higgins.sts.utilities.ExceptionHelper.Log
    			(this.log,
    			e);
			final Fault fault = new Fault
				(constants.getWSTrustNamespace(),
				"wst",
				constants.getRequestFailedFaultCode(),
				"The specified request failed",
				"Failed to set RequestSecurityToken elements.");
			response.setFault(fault);
			return;
		}
	}
	
	public ISettingDescriptor getComponentDescriptor() {
		return null;
	}
}