/*******************************************************************************
 * Copyright (c) 2006 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Michael McIntosh (IBM Corporation) - initial API and implementation
 *******************************************************************************/ 

package org.eclipse.higgins.sts.xmlsecurity.apache;

import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.cert.X509Certificate;
import java.util.Map;

import org.eclipse.higgins.configuration.api.ISettingDescriptor;
import org.eclipse.higgins.sts.api.IConstants;
import org.eclipse.higgins.sts.spi.IXMLSecurityExtension;
import org.w3c.dom.Element;

/**
 * @author mikemci
 *
 */
public class XMLSecurityApacheExtension
	implements IXMLSecurityExtension
{
	private static final org.eclipse.higgins.sts.utilities.LogHelper log = new org.eclipse.higgins.sts.utilities.LogHelper
		(XMLSecurityApacheExtension.class.getName());
	
	private String strSignatureAlgorithm = org.apache.xml.security.signature.XMLSignature.ALGO_ID_SIGNATURE_RSA_SHA1;
	
	private String strSignatureCanonicalizationAlgorithm = org.apache.xml.security.transforms.Transforms.TRANSFORM_C14N_EXCL_OMIT_COMMENTS;
	
	private String strReferenceCanonicalizationAlgorithm = org.apache.xml.security.transforms.Transforms.TRANSFORM_C14N_EXCL_OMIT_COMMENTS;

	private String strReferenceDigestAlgorithm = org.apache.xml.security.utils.Constants.ALGO_ID_DIGEST_SHA1;
	
	private String strEncryptionAlgorithm = org.apache.xml.security.encryption.XMLCipher.AES_256;
	
	private String strEncryptionAlgorithmProvider = null;
	
	private String strEncryptionDigestAlgorithm = org.apache.xml.security.utils.Constants.ALGO_ID_DIGEST_SHA1;
	
	private String strEncryptionKeyGeneratorAlgorithm = "AES";
	
	private String strEncryptionKeyGeneratorAlgorithmProvider = null;
	
	private int nEncryptionKeyGeneratorSize = 256;
	
	private String strEncryptionKeyWrapAlgorithm = org.apache.xml.security.encryption.XMLCipher.RSA_OAEP;
	
	private String strEncryptionKeyWrapAlgorithmProvider = null;
	
	public org.eclipse.higgins.sts.api.IElement SignEnveloping
		(final org.eclipse.higgins.sts.api.IElement elemObject,
		final String strObjectId,
		final X509Certificate x509Certificate,
		final PrivateKey privateKey) throws Exception
	{
	    final javax.xml.parsers.DocumentBuilderFactory documentBuilderFactory = javax.xml.parsers.DocumentBuilderFactory.newInstance();
	    documentBuilderFactory.setNamespaceAware(true);
	    final javax.xml.parsers.DocumentBuilder documentBuilder = documentBuilderFactory.newDocumentBuilder();
	    final org.w3c.dom.Document docSignedManagedCard = documentBuilder.newDocument();

		final org.apache.xml.security.signature.XMLSignature xmlSignature = new org.apache.xml.security.signature.XMLSignature
			(docSignedManagedCard,
			"",
			strSignatureAlgorithm,
			strSignatureCanonicalizationAlgorithm);
	    docSignedManagedCard.appendChild(xmlSignature.getElement());
	    final org.apache.xml.security.signature.ObjectContainer objectContainer = new org.apache.xml.security.signature.ObjectContainer
	    	(docSignedManagedCard);
	    final org.w3c.dom.Element domObject = (org.w3c.dom.Element)elemObject.getAs(org.w3c.dom.Element.class);
	    objectContainer.appendChild
	    	(docSignedManagedCard.importNode
	    		(domObject,
	    		true));
	    objectContainer.setId
	    	(strObjectId);
	    xmlSignature.appendObject
	    	(objectContainer);
	    final org.apache.xml.security.transforms.Transforms transforms = new org.apache.xml.security.transforms.Transforms
	    	(docSignedManagedCard);
	    transforms.addTransform
	    	(strReferenceCanonicalizationAlgorithm);
	    xmlSignature.addDocument
	    	("#" + strObjectId,
	    	transforms,
	    	strReferenceDigestAlgorithm);
	    xmlSignature.addKeyInfo
	    	(x509Certificate);
	    xmlSignature.sign
	    	(privateKey);
	    final org.eclipse.higgins.sts.api.IElement elemResult = new org.eclipse.higgins.sts.common.Element();
	    elemResult.set
	    	(org.eclipse.higgins.sts.utilities.XMLHelper.reparseElement
	    		(xmlSignature.getElement()));
	    return elemResult;
	}

	public String SHA1Digest
		(final byte [] byteData) throws Exception
	{
		java.security.MessageDigest md = java.security.MessageDigest.getInstance("SHA-1");
		md.reset();
		final byte [] byteDigest = md.digest(byteData);
		return org.apache.xml.security.utils.Base64.encode(byteDigest);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.sts.IXMLSecurityExtension#DecryptElement(org.w3c.dom.Element, java.security.PrivateKey)
	 */
	public org.eclipse.higgins.sts.api.IElement DecryptElement
		(final org.eclipse.higgins.sts.api.IElement elemEncryptedData,
		final PrivateKey privateKey) throws Exception
	{
		XMLSecurityApacheExtension.log.trace("DecryptElement");
		
		final org.w3c.dom.Element domEncryptedData = (org.w3c.dom.Element)elemEncryptedData.getAs(org.w3c.dom.Element.class);
		final org.w3c.dom.NodeList nlEncryptedKey = domEncryptedData.getElementsByTagNameNS
			(org.apache.xml.security.utils.EncryptionConstants.EncryptionSpecNS,
			org.apache.xml.security.utils.EncryptionConstants._TAG_ENCRYPTEDKEY);
		if (null == nlEncryptedKey)
		{
			XMLSecurityApacheExtension.log.trace("No EncryptedKey found (getElementsByTagName returned null)");
		}
		else if (0 == nlEncryptedKey.getLength())
		{
			XMLSecurityApacheExtension.log.trace("No EncryptedKey found (0 == getLength())");
		}
		else
		{
			final org.w3c.dom.Document domDocument = domEncryptedData.getOwnerDocument();
			final org.w3c.dom.Element elemEncryptedKey = (org.w3c.dom.Element)nlEncryptedKey.item(0);
			final org.w3c.dom.NodeList nlKeyEncryptionMethod = elemEncryptedKey.getElementsByTagNameNS
				(org.apache.xml.security.utils.EncryptionConstants.EncryptionSpecNS,
				org.apache.xml.security.utils.EncryptionConstants._TAG_ENCRYPTIONMETHOD);
			String strKeyEncryptionMethod = null;
			if (null == nlKeyEncryptionMethod)
			{
				XMLSecurityApacheExtension.log.trace("No EncryptionMethod found");
				strKeyEncryptionMethod = strEncryptionKeyWrapAlgorithm;
			}
			else if (0 == nlKeyEncryptionMethod.getLength())
			{
				XMLSecurityApacheExtension.log.trace("No EncryptionMethod found");
				strKeyEncryptionMethod = strEncryptionKeyWrapAlgorithm;
			}
			else
			{
				final org.w3c.dom.Element elemKeyEncryptionMethod = (org.w3c.dom.Element)nlKeyEncryptionMethod.item(0);
				strKeyEncryptionMethod = elemKeyEncryptionMethod.getAttribute(org.apache.xml.security.utils.EncryptionConstants._ATT_ALGORITHM);
			}
			org.apache.xml.security.encryption.XMLCipher keyCipher = null;
			if (null == strEncryptionKeyWrapAlgorithmProvider)
				keyCipher = org.apache.xml.security.encryption.XMLCipher.getInstance
					(strKeyEncryptionMethod);
			else
				keyCipher = org.apache.xml.security.encryption.XMLCipher.getProviderInstance
					(strKeyEncryptionMethod,
					strEncryptionKeyWrapAlgorithmProvider);
			keyCipher.init
				(org.apache.xml.security.encryption.XMLCipher.UNWRAP_MODE,
				privateKey);
			final org.apache.xml.security.encryption.EncryptedKey encryptedKey = keyCipher.loadEncryptedKey
				(domDocument,
				elemEncryptedKey);
			final java.security.Key keySecret = keyCipher.decryptKey
				(encryptedKey,
				strEncryptionKeyWrapAlgorithm);
			final javax.crypto.SecretKey secretKey = (javax.crypto.SecretKey)keySecret;
			final byte [] encodedSecretKey = secretKey.getEncoded();
			final javax.crypto.spec.SecretKeySpec secretKeySpec = new javax.crypto.spec.SecretKeySpec
				(encodedSecretKey,
				strEncryptionKeyGeneratorAlgorithm);
			org.apache.xml.security.encryption.XMLCipher xmlCipher = null;
			if (null == strEncryptionAlgorithmProvider)
				xmlCipher = org.apache.xml.security.encryption.XMLCipher.getInstance
					(strEncryptionAlgorithm);
			else
				xmlCipher = org.apache.xml.security.encryption.XMLCipher.getProviderInstance
					(strEncryptionAlgorithm,
					strEncryptionAlgorithmProvider);
	        xmlCipher.init
	        	(org.apache.xml.security.encryption.XMLCipher.DECRYPT_MODE,
	        	secretKeySpec);
			final org.w3c.dom.Document domResultDocument = xmlCipher.doFinal
				(domDocument,
				domEncryptedData,
				false);
			final org.eclipse.higgins.sts.api.IElement elemResult = new org.eclipse.higgins.sts.common.Element();
			elemResult.set
				(domResultDocument.getDocumentElement());
	        return elemResult;
		}
		return null;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.sts.IXMLSecurityExtension#EncryptElement(java.lang.String, org.apache.axiom.om.OMElement, java.lang.String)
	 */
	public org.eclipse.higgins.sts.api.IElement EncryptElement
		(final String strTagNameOfElementToEncrypt,
		final org.eclipse.higgins.sts.api.IElement elemParent,
		final String strCertificate)
		throws Exception
	{
		final org.w3c.dom.Element domParent = (org.w3c.dom.Element)elemParent.getAs
			(org.w3c.dom.Element.class);
		final org.w3c.dom.Document domDocument = domParent.getOwnerDocument();
		final org.w3c.dom.NodeList nlToEncrypts = domParent.getElementsByTagName
			(strTagNameOfElementToEncrypt);
		if (null == nlToEncrypts)
		{
			final String strErrorMessage = "org.w3c.dom.Element.getElementsByTagNameNS unexpectedly returned null";
			XMLSecurityApacheExtension.log.error
				(strErrorMessage);
			return null;	
		}
		if (1 != nlToEncrypts.getLength())
		{
			final String strErrorMessage = "org.w3c.dom.Element.getElementsByTagNameNS unexpectedly returned "
				+ nlToEncrypts.getLength()
				+ " nodes";
			XMLSecurityApacheExtension.log.error
				(strErrorMessage);
			return null;	
		}
		final org.w3c.dom.Element domToEncrypt = (org.w3c.dom.Element)nlToEncrypts.item(0);
		final org.w3c.dom.Element domTempParent = domDocument.createElement
			("Parent");
		final org.w3c.dom.Element domTempToEncrypt = (org.w3c.dom.Element)domTempParent.appendChild
			(domToEncrypt.cloneNode(true));
		
		final java.security.cert.X509Certificate certificate = org.eclipse.higgins.sts.utilities.CertificateHelper.fromString(strCertificate);
		
		javax.crypto.KeyGenerator keyGenerator = null;
		if (null == strEncryptionKeyGeneratorAlgorithmProvider)
			keyGenerator = javax.crypto.KeyGenerator.getInstance
				(strEncryptionKeyGeneratorAlgorithm);
		else
			keyGenerator = javax.crypto.KeyGenerator.getInstance
				(strEncryptionKeyGeneratorAlgorithm,
				strEncryptionKeyGeneratorAlgorithmProvider);
		keyGenerator.init
			(nEncryptionKeyGeneratorSize);
		final javax.crypto.SecretKey secretKey = keyGenerator.generateKey();
		final java.security.PublicKey publicKeyRP = certificate.getPublicKey();
		org.apache.xml.security.encryption.XMLCipher keyCipher = null;
		if (null == strEncryptionKeyWrapAlgorithmProvider)
			keyCipher = org.apache.xml.security.encryption.XMLCipher.getInstance
				(strEncryptionKeyWrapAlgorithm);
		else
			keyCipher = org.apache.xml.security.encryption.XMLCipher.getProviderInstance
				(strEncryptionKeyWrapAlgorithm,
				strEncryptionKeyWrapAlgorithmProvider);
		keyCipher.init
			(org.apache.xml.security.encryption.XMLCipher.WRAP_MODE,
			publicKeyRP);
	    final org.apache.xml.security.keys.KeyInfo keyInfoKey = new org.apache.xml.security.keys.KeyInfo
    		(domDocument);
		final java.security.MessageDigest mdSha1 = java.security.MessageDigest.getInstance
			("SHA-1");
		final byte [] byteThumbPrint = mdSha1.digest(certificate.getEncoded());
		final org.w3c.dom.Document domParentDocument = domParent.getOwnerDocument();
		final org.w3c.dom.Element domSTR = domParentDocument.createElementNS
			("http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-secext-1.0.xsd",
			"SecurityTokenReference");
		final org.w3c.dom.Element domKeyIdentifier = domParentDocument.createElementNS
			("http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-secext-1.0.xsd",
			"KeyIdentifier");
		domKeyIdentifier.setAttribute
			("ValueType",
			"http://docs.oasis-open.org/wss/oasis-wss-soap-message-security-1.1#ThumbprintSHA1");
		domKeyIdentifier.setAttribute
			("EncodingType",
			"http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-soap-message-security-1.0#Base64Binary");
		String strThumbprint = org.apache.xml.security.utils.Base64.encode
			(byteThumbPrint);
		org.eclipse.higgins.sts.utilities.XMLHelper.setTextContent
			(domKeyIdentifier,
			strThumbprint);
		domSTR.appendChild(domKeyIdentifier);
	    keyInfoKey.addUnknownElement(domSTR);
		final org.apache.xml.security.encryption.EncryptedKey encryptedKey = keyCipher.encryptKey
			(domDocument,
			secretKey);
		encryptedKey.setKeyInfo(keyInfoKey);
		final org.apache.xml.security.encryption.EncryptionMethod encryptionMethod = encryptedKey.getEncryptionMethod();
		final org.w3c.dom.Element elemDigestMethod = domDocument.createElementNS
			(org.apache.xml.security.utils.Constants.SignatureSpecNS,
			"DigestMethod");
		elemDigestMethod.setAttribute
			("Algorithm",
			strEncryptionDigestAlgorithm);
		encryptionMethod.addEncryptionMethodInformation(elemDigestMethod);
		org.apache.xml.security.encryption.XMLCipher xmlCipher = null;
		if (null == strEncryptionAlgorithmProvider)
			xmlCipher = org.apache.xml.security.encryption.XMLCipher.getInstance
				(strEncryptionAlgorithm);
		else
			xmlCipher = org.apache.xml.security.encryption.XMLCipher.getProviderInstance
				(strEncryptionAlgorithm,
				strEncryptionAlgorithmProvider);
	    xmlCipher.init
	    	(org.apache.xml.security.encryption.XMLCipher.ENCRYPT_MODE,
	    	secretKey);
	    final org.apache.xml.security.encryption.EncryptedData encryptedData = xmlCipher.getEncryptedData();
	    final org.apache.xml.security.keys.KeyInfo keyInfoEncryption = new org.apache.xml.security.keys.KeyInfo
	    	(domDocument);
	    keyInfoEncryption.add
	    	(encryptedKey);
	    encryptedData.setKeyInfo
	    	(keyInfoEncryption);
	    xmlCipher.doFinal
	    	(domDocument,
	    	domTempToEncrypt,
	    	false);
	    final org.w3c.dom.NodeList nlEncryptedData = domTempParent.getElementsByTagNameNS
	    	(org.apache.xml.security.utils.EncryptionConstants.EncryptionSpecNS,
	    	"EncryptedData");
	    if (1 != nlEncryptedData.getLength())
	    {
			throw new Exception("One EncryptedData Not Found!");
		}
	    org.w3c.dom.Element domEncryptedData = (org.w3c.dom.Element)nlEncryptedData.item(0);
	    org.eclipse.higgins.sts.utilities.XMLHelper.stripNewLinesFromElement((org.w3c.dom.Element)nlEncryptedData.item(0));
		domParent.replaceChild(domEncryptedData, domToEncrypt);
		org.eclipse.higgins.sts.api.IElement elemResult = new org.eclipse.higgins.sts.common.Element();
		elemResult.set(domParent);
		return elemResult;
	}
	
	public org.eclipse.higgins.sts.api.IElement SignDetached
		(final String [] strElementIdsToSign,
		final org.eclipse.higgins.sts.api.IElement elemParent,
		final java.security.PrivateKey privateKey,
		final java.security.PublicKey publicKey,
		final String strIDOfToken,	
		final org.eclipse.higgins.sts.api.IConstants constants)
		throws Exception
	{
		if (0 == strElementIdsToSign.length)
		{
			throw new Exception("Sign Failed: Invalid Parameter: strElementIdsToSign is length 0");
		}
		if (null == elemParent)
		{
			throw new Exception("Sign Failed: Invalid Parameter: Parent is null");
		}
		if (null == privateKey)
		{
			throw new Exception("Sign Failed: Invalid Parameter: PrivateKey is null");
		}
		if (null == strIDOfToken)
		{
			throw new Exception("Sign Failed: Invalid Parameter: IDOfToken is null");
		}
		if (null == constants)
		{
			throw new Exception("Sign Failed: Invalid Parameter: Constants is null");
		}
		org.apache.xml.security.utils.Constants.setSignatureSpecNSprefix("ds");
		
		final org.w3c.dom.Element domParent = (org.w3c.dom.Element)elemParent.getAs(org.w3c.dom.Element.class);
		final org.w3c.dom.Document domDocument = domParent.getOwnerDocument();
		final org.w3c.dom.Element domDocumentElement = domDocument.getDocumentElement();
		
		log.trace("SignDetached: " + org.eclipse.higgins.sts.utilities.XMLHelper.toString(domDocumentElement));
		
		final org.apache.xml.security.signature.XMLSignature signature = new org.apache.xml.security.signature.XMLSignature
			(domDocument,
			"",
			strSignatureAlgorithm,
			strSignatureCanonicalizationAlgorithm);
		
		final org.apache.xml.security.signature.SignedInfo signedInfo = signature.getSignedInfo();
		final IDResolver resolver = new IDResolver
			(domDocument);
		signedInfo.addResourceResolver
			(resolver);
		for (int i = 0; i < strElementIdsToSign.length; ++i)
		{
			String strReferenceId = "#" + strElementIdsToSign[i];
			final org.apache.xml.security.transforms.Transforms transforms = new org.apache.xml.security.transforms.Transforms
				(domDocument);
			transforms.addTransform
				(strReferenceCanonicalizationAlgorithm);
			signature.addDocument
				(strReferenceId,
				transforms);
		}
		org.apache.xml.security.keys.KeyInfo keyInfo = signature.getKeyInfo();
		
		try
		{
			signature.sign(privateKey);
		}
		catch (Exception e)
		{
			org.eclipse.higgins.sts.utilities.ExceptionHelper.Log(log, e);
			return null;
		}
		final org.w3c.dom.Element domKeyInfo = keyInfo.getElement();
		final org.w3c.dom.Element domSTR = domDocument.createElementNS
			(constants.getWSSecurityNamespace().toString(),
			"SecurityTokenReference");
		final org.w3c.dom.Element domSTRReference = domDocument.createElementNS
			(constants.getWSSecurityNamespace().toString(),
			"Reference");
		domSTRReference.setAttribute("ValueType", "http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-x509-token-profile-1.0#X509v3");
		domSTRReference.setAttribute("URI", "#" + strIDOfToken);
		domSTR.appendChild(domSTRReference);
		domKeyInfo.appendChild(domSTR);
		org.w3c.dom.Element domSignature = signature.getElement();
		org.eclipse.higgins.sts.api.IElement elemResult = new org.eclipse.higgins.sts.common.Element();
		elemResult.set(domSignature);
		return elemResult;
	}


	/* (non-Javadoc)
	 * @see org.eclipse.higgins.sts.IXMLSecurityExtension#SignEnveloped(java.lang.String, org.apache.axiom.om.OMElement, java.lang.String, java.security.PrivateKey, java.security.PublicKey, org.eclipse.higgins.sts.IConstants)
	 */
	public org.eclipse.higgins.sts.api.IElement SignEnveloped
		(final String strTagNameOfElementsToSign,
		final org.eclipse.higgins.sts.api.IElement elemParent,
		final String strReferenceIdentifier,
		final PrivateKey privateKey,
		final PublicKey publicKey,
		final IConstants constants)
		throws Exception
	{
		if (null == strTagNameOfElementsToSign)
		{
			throw new Exception("Sign Failed: Invalid Parameter: TagNameOfElementsToSign is null");
		}
		if (null == elemParent)
		{
			throw new Exception("Sign Failed: Invalid Parameter: Parent is null");
		}
		if (null == strReferenceIdentifier)
		{
			throw new Exception("Sign Failed: Invalid Parameter: ReferenceIdentifier is null");
		}
		if (null == privateKey)
		{
			throw new Exception("Sign Failed: Invalid Parameter: PrivateKey is null");
		}
		if (null == publicKey)
		{
			throw new Exception("Sign Failed: Invalid Parameter: PublicKey is null");
		}
		if (null == constants)
		{
			throw new Exception("Sign Failed: Invalid Parameter: Constants is null");
		}
		org.apache.xml.security.utils.Constants.setSignatureSpecNSprefix("ds");
		org.w3c.dom.Element domParent = (org.w3c.dom.Element)elemParent.getAs(org.w3c.dom.Element.class);
		domParent = org.eclipse.higgins.sts.utilities.XMLHelper.reparseElement
			(domParent);
		final org.w3c.dom.Document domDocument = domParent.getOwnerDocument();
		final org.w3c.dom.NodeList nlToSigns = domParent.getElementsByTagName
			(strTagNameOfElementsToSign);
		if (null == nlToSigns)
		{
			final String strErrorMessage = "org.w3c.dom.Element.getElementsByTagNameNS unexpectedly returned null";
			XMLSecurityApacheExtension.log.error
				(strErrorMessage);
			throw new Exception("Sign Failed: " + strErrorMessage);
		}
		if (1 != nlToSigns.getLength())
		{
			final String strErrorMessage = "org.w3c.dom.Element.getElementsByTagNameNS unexpectedly returned "
				+ nlToSigns.getLength()
				+ " nodes";
			XMLSecurityApacheExtension.log.error
				(strErrorMessage);
			throw new Exception("Sign Failed: " + strErrorMessage);			
		}
		final org.w3c.dom.Element domToSign = (org.w3c.dom.Element)nlToSigns.item(0);
		final org.apache.xml.security.signature.XMLSignature signature = new org.apache.xml.security.signature.XMLSignature
    		(domDocument,
    		"",
    		strSignatureAlgorithm,
    		strSignatureCanonicalizationAlgorithm);
		domToSign.appendChild(signature.getElement());
		final org.apache.xml.security.signature.SignedInfo signedInfo = signature.getSignedInfo();
		final IDResolver resolver = new IDResolver
			(domDocument);
		signedInfo.addResourceResolver(resolver);
		final org.apache.xml.security.transforms.Transforms transforms = new org.apache.xml.security.transforms.Transforms
			(domDocument);
		transforms.addTransform
			(org.apache.xml.security.transforms.Transforms.TRANSFORM_ENVELOPED_SIGNATURE);
		transforms.addTransform
			(strReferenceCanonicalizationAlgorithm);
		signature.addDocument
			("#" + strReferenceIdentifier,
			transforms,
			strReferenceDigestAlgorithm);
		signature.addKeyInfo
			(publicKey);
		signature.sign(privateKey);
		final org.w3c.dom.Element elemSignature = signature.getElement();
		final org.w3c.dom.NodeList nodeListSignatureValue = elemSignature.getElementsByTagNameNS
			(constants.getXMLSignatureNamespace().toString(),
			"SignatureValue");
		final org.w3c.dom.Element elemSignatureValue = (org.w3c.dom.Element)nodeListSignatureValue.item(0);
		org.eclipse.higgins.sts.utilities.XMLHelper.stripWhiteSpaceFromElement(elemSignatureValue);
		domParent = org.eclipse.higgins.sts.utilities.XMLHelper.reparseElement
			(domParent);
		org.eclipse.higgins.sts.api.IElement elemResult = new org.eclipse.higgins.sts.common.Element();
		elemResult.set(domParent);
		return elemResult;
	}
	
	public boolean VerifyEnveloped
		(final org.eclipse.higgins.sts.api.IElement elemSignedElement,
		final IConstants constants)
		throws Exception
	{
		if (null == elemSignedElement)
		{
			throw new Exception("Verify Failed: Invalid Parameter: SignedElement is null");
		}
//		if (null == constants)
//		{
//			throw new Exception("Verify Failed: Invalid Parameter: Constants is null");
//		}
		final org.w3c.dom.Element domSignedElement = (org.w3c.dom.Element)elemSignedElement.getAs(org.w3c.dom.Element.class);
		final org.w3c.dom.Document domParent = domSignedElement.getOwnerDocument();
        final Element elemContext = org.apache.xml.security.utils.XMLUtils.createDSctx
        	(domParent,
        	"ds",
        	org.apache.xml.security.utils.Constants.SignatureSpecNS);
        final Element elemSignature = (Element) org.apache.xpath.XPathAPI.selectSingleNode
        	(domParent,
            "//ds:Signature[1]",
            elemContext);
        final org.apache.xml.security.signature.XMLSignature signature = new org.apache.xml.security.signature.XMLSignature
        	(elemSignature,
        	null);
        signature.addResourceResolver(new IDResolver(domParent));
        final org.apache.xml.security.keys.KeyInfo keyInfo = signature.getKeyInfo();
        boolean bResult = false;
        if (keyInfo != null)
        {
           if (keyInfo.containsX509Data())
           {
              System.out.println("Found a X509Data element in the KeyInfo");
           }
           final java.security.cert.X509Certificate cert = signature.getKeyInfo().getX509Certificate();
           if (cert != null)
           {
        	   bResult = signature.checkSignatureValue(cert);
           } 
           else
           {
              System.out.println("Did not find an X509Data element in the KeyInfo");
              final PublicKey publicKey = keyInfo.getPublicKey();
              if (publicKey != null)
              {
            	  bResult = signature.checkSignatureValue(publicKey);
              } 
              else
              {
                 System.out.println("Did not find a public key, so I can't check the signature");
              }
           }
        }
        else
        {
        	System.out.println("Did not find a KeyInfo");
        }
        return bResult;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.sts.IXMLSecurityExtension#configure(java.lang.String, java.util.Map)
	 */
	public void configure
		(final Map mapGlobalSettings,
		final String strComponentName,
		final Map mapComponentSettings,
		final ISettingDescriptor componentDescriptor,
		final ISettingDescriptor globalDescriptor)
	{
		System.err.println("XMLSecurityApacheExtension:configure");
		log.trace("XMLSecurityApacheExtension:configure");
		
		org.apache.xml.security.Init.init();
		
		if (null == mapComponentSettings) return;
	    
		strSignatureAlgorithm = (String)mapComponentSettings.get
			("SignatureAlgorithm");
		if (null == strSignatureAlgorithm)
			strSignatureAlgorithm = org.apache.xml.security.signature.XMLSignature.ALGO_ID_SIGNATURE_RSA_SHA1;
		log.trace("SignatureAlgorithm: " + strSignatureAlgorithm);
		
		strSignatureCanonicalizationAlgorithm = (String)mapComponentSettings.get
			("SignatureCanonicalizationAlgorithm");
		if (null == strSignatureCanonicalizationAlgorithm)
			strSignatureCanonicalizationAlgorithm = org.apache.xml.security.transforms.Transforms.TRANSFORM_C14N_EXCL_OMIT_COMMENTS;
		log.trace("SignatureCanonicalizationAlgorithm: " + strSignatureCanonicalizationAlgorithm);
		
		strReferenceCanonicalizationAlgorithm = (String)mapComponentSettings.get
			("ReferenceCanonicalizationAlgorithm");
		if (null == strReferenceCanonicalizationAlgorithm)
			strReferenceCanonicalizationAlgorithm = org.apache.xml.security.transforms.Transforms.TRANSFORM_C14N_EXCL_OMIT_COMMENTS;
		log.trace("ReferenceCanonicalizationAlgorithm: " + strReferenceCanonicalizationAlgorithm);

		strReferenceDigestAlgorithm = (String)mapComponentSettings.get
			("ReferenceDigestAlgorithm");
		if (null == strReferenceDigestAlgorithm)
			strReferenceDigestAlgorithm = org.apache.xml.security.utils.Constants.ALGO_ID_DIGEST_SHA1;
		log.trace("ReferenceDigestAlgorithm: " + strReferenceDigestAlgorithm);
		
		strEncryptionAlgorithm = (String)mapComponentSettings.get
			("EncryptionAlgorithm");
		if (null == strEncryptionAlgorithm)
			strEncryptionAlgorithm = org.apache.xml.security.encryption.XMLCipher.AES_256;
		log.trace("EncryptionAlgorithm: " + strEncryptionAlgorithm);
		
		strEncryptionAlgorithmProvider = (String)mapComponentSettings.get
			("EncryptionAlgorithmProvider"); // "IBMJCE"
		log.trace("EncryptionAlgorithmProvider: " + strEncryptionAlgorithmProvider);

		strEncryptionDigestAlgorithm = (String)mapComponentSettings.get
			("EncryptionDigestAlgorithm");
		if (null == strEncryptionDigestAlgorithm)
			strEncryptionDigestAlgorithm = org.apache.xml.security.utils.Constants.ALGO_ID_DIGEST_SHA1;
		log.trace("EncryptionDigestAlgorithm: " + strEncryptionDigestAlgorithm);
	
		strEncryptionKeyGeneratorAlgorithm = (String)mapComponentSettings.get
			("EncryptionKeyGeneratorAlgorithm");
		if (null == strEncryptionKeyGeneratorAlgorithm)
			strEncryptionKeyGeneratorAlgorithm = "AES";
		log.trace("EncryptionKeyGeneratorAlgorithm: " + strEncryptionKeyGeneratorAlgorithm);
		
		Integer intEncryptionKeyGeneratorSize = (Integer)mapComponentSettings.get
			("EncryptionKeyGeneratorSize");
		if (null == intEncryptionKeyGeneratorSize)
			nEncryptionKeyGeneratorSize = 256;
		else
			nEncryptionKeyGeneratorSize = intEncryptionKeyGeneratorSize.intValue();
		log.trace("EncryptionKeyGeneratorSize: " + nEncryptionKeyGeneratorSize);
		
		strEncryptionKeyGeneratorAlgorithmProvider = (String)mapComponentSettings.get
			("EncryptionKeyGeneratorAlgorithmProvider"); // "IBMJCE"
		log.trace("EncryptionKeyGeneratorAlgorithmProvider: " + strEncryptionKeyGeneratorAlgorithmProvider);

		strEncryptionKeyWrapAlgorithm = (String)mapComponentSettings.get
			("EncryptionKeyWrapAlgorithm");
		if (null == strEncryptionKeyWrapAlgorithm)
			strEncryptionKeyWrapAlgorithm = org.apache.xml.security.encryption.XMLCipher.RSA_OAEP;
		log.trace("EncryptionKeyWrapAlgorithm: " + strEncryptionKeyWrapAlgorithm);
		
		strEncryptionKeyWrapAlgorithmProvider = (String)mapComponentSettings.get
			("EncryptionKeyWrapAlgorithmProvider"); // "IBMJCE"
		log.trace("EncryptionKeyWrapAlgorithmProvider: " + strEncryptionKeyWrapAlgorithmProvider);
			
	}
	
	public XMLSecurityApacheExtension()
	{
		System.err.println("XMLSecurityApacheExtension:XMLSecurityApacheExtension");
		log.trace("XMLSecurityApacheExtension:XMLSecurityApacheExtension");		
	}
	
	public ISettingDescriptor getComponentDescriptor() {
		return null;
	}

}
