/** 
 * Copyright (c) 2007 Parity Communications, Inc.  
 * All rights reserved. This program and the accompanying materials  
 * are made available under the terms of the Eclipse Public License v1.0  
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 *  
 * Contributors: 
 *		Yuriy Pilipenko - API and implementation 
 * 
 */
package org.eclipse.higgins.userprofile.idas;

import java.net.URI;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.icard.CUID;
import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IAttributeValue;
import org.eclipse.higgins.idas.api.IComplexAttrValue;
import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.ISimpleAttrValue;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.userprofile.ICardUsageManager;
import org.eclipse.higgins.userprofile.UserProfileException;
import org.eclipse.higgins.userprofile.entity.CardUsage;
import org.eclipse.higgins.userprofile.entity.WebForm;
import org.eclipse.higgins.userprofile.idas.util.IdasCtxUris;
import org.eclipse.higgins.userprofile.idas.util.IdasUtils;

/**
 * @author PilipenkoYN
 *
 */
public class CardUsageManager implements ICardUsageManager {
	private static Log log = LogFactory.getLog(CardUsageManager.class);
	private IEntity dsUser_ = null;
	private CardUrlRelationBinding cuRelation_ = null;
	private CardHistoryBinding history_ = null;
	private CUID cuid_ = null;
	
	private CardHistoryBinding createHistory(CUID cuid) throws Exception {
		if (cuid_ == null || !cuid_.equals(cuid)) {
			cuid_ = cuid;
			history_ = new CardHistoryBinding(dsUser_, cuid);
		}
		return history_;
	}
	
	public CardUsageManager() {

	}
	
	public CardUsageManager(IEntity dsUserProfile) throws Exception {
		dsUser_ = dsUserProfile;
		cuRelation_ = new CardUrlRelationBinding(dsUser_);
	}
	

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.icard.userprofile.ICardUsageManager#addCardHistory(org.eclipse.higgins.icard.CUID, java.net.URI)
	 */
	public CardUsage addCardHistory(CUID cuid, URI url) throws UserProfileException {
		try {
			if (cuid == null || url == null)
				throw new IllegalArgumentException("Both cuid and site must be set!");
			
			createHistory(cuid);
			CardUsage result = history_.saveHistoryEvent(url);
			cuRelation_.updateUsageSummary(result);
			//cuRelation_.finish();
			//finish in close() method!!!
			return result;
		} catch (Exception e) {
			log.error(e);
			throw new UserProfileException(e);
		}
	}
	
	public CardUsage addCardHistory(CUID cuid, WebForm form) throws UserProfileException {
		try {
			if (cuid == null || form == null)
				throw new IllegalArgumentException("Both cuid and form must be set!");
			
			createHistory(cuid);
			CardUsage result = history_.saveHistoryEvent(form);
			cuRelation_.updateUsageSummary(result);
			//cuRelation_.finish();
			//finish in close() method!!!
			return result;
		} catch (Exception e) {
			log.error(e);
			throw new UserProfileException(e);
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.icard.userprofile.ICardUsageManager#clearCardHistory(org.eclipse.higgins.icard.CUID)
	 */
	public void clearCardHistory(CUID cuid) throws UserProfileException {
		try {
			if (cuid == null)
				throw new IllegalArgumentException("Card ID can not be null");

			createHistory(cuid);
			history_.clearHistory();
			cuRelation_.setCuid(cuid);
			cuRelation_.clearLastUsedDate();
		} catch (Exception e) {
			log.error(e);
			throw new UserProfileException(e);
		}
	}
	
	public void clearCardHistory(CUID cuid, URI site) throws UserProfileException {
		try {
			if (cuid == null)
				throw new IllegalArgumentException("Card ID can not be null");
			if (site == null)
				throw new IllegalArgumentException("Site url can not be null");

			//createHistory(cuid);
			//history_.clearHistory();
			cuRelation_.setCuid(cuid);
			cuRelation_.clearLastUsedDate(new WebForm(site, null, null, null));
		} catch (Exception e) {
			log.error(e);
			throw new UserProfileException(e);
		}		
	}
	
	public void clearCardHistory(CUID cuid, WebForm form) throws UserProfileException {
		try {
			if (cuid == null)
				throw new IllegalArgumentException("Card ID can not be null");
			if (form == null)
				throw new IllegalArgumentException("Web form can not be null");

			//createHistory(cuid);
			//history_.clearHistory();
			cuRelation_.setCuid(cuid);
			cuRelation_.clearLastUsedDate(form);
		} catch (Exception e) {
			log.error(e);
			throw new UserProfileException(e);
		}		
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.icard.userprofile.ICardUsageManager#close()
	 */
	public void close() throws UserProfileException {
		try {
			try {
				cuRelation_.finish(); 
				dsUser_.getContext().applyUpdates();
			} catch (Exception e) {
				log.error(e);
				e.printStackTrace();
				throw new UserProfileException(e);
			}
		} finally {
			try {
				dsUser_.getContext().close();
			} catch (IdASException e) {
				log.error(e);
				e.printStackTrace();
			}			
		}
	}
		
	/* (non-Javadoc)
	 * @see org.eclipse.higgins.icard.userprofile.ICardUsageManager#setUsedAlways(org.eclipse.higgins.icard.CUID, java.net.URI)
	 */
	public CardUsage setUsedAlways(CUID cuid, URI url) throws UserProfileException {
		try {
			if (cuid == null || url == null)
				throw new IllegalArgumentException("Both cuid and site must be set!");
				
			CardUsage cardUsage = new CardUsage(cuid, url, null, Boolean.TRUE);
			cardUsage = cuRelation_.updateUsageSummary(cardUsage);
			return cardUsage;
		} catch (Exception e) {
			log.error(e);
			e.printStackTrace();
			throw new UserProfileException(e);
		}
	}
	
	public CardUsage setUsedAlways(CUID cuid, WebForm form) throws UserProfileException {
		try {
			if (cuid == null || form == null)
				throw new IllegalArgumentException("Both cuid and form must be set!");
				
			CardUsage cardUsage = new CardUsage(cuid, form, null, Boolean.TRUE);
			cardUsage = cuRelation_.updateUsageSummary(cardUsage);
			return cardUsage;
		} catch (Exception e) {
			log.error(e);
			e.printStackTrace();
			throw new UserProfileException(e);
		}
	}
	
	
	/* (non-Javadoc)
	 * @see org.eclipse.higgins.icard.userprofile.ICardUsageManager#deleteUsedAlways(org.eclipse.higgins.icard.CUID, java.net.URI)
	 */
	public CardUsage deleteUsedAlways(CUID cuid, URI url) throws UserProfileException {
		try {
			if (cuid == null || url == null)
				throw new IllegalArgumentException("Both cuid and site must be set!");
			
			CardUsage cardUsage = new CardUsage(cuid, url, null, Boolean.FALSE);
			cardUsage = cuRelation_.updateUsageSummary(cardUsage);
			return cardUsage;
		} catch (Exception e) {
			log.error(e);
			throw new UserProfileException(e);
		}
	}
	
	public CardUsage deleteUsedAlways(CUID cuid, WebForm form) throws UserProfileException {
		try {
			if (cuid == null || form == null)
				throw new IllegalArgumentException("Both cuid and form must be set!");
			
			CardUsage cardUsage = new CardUsage(cuid, form, null, Boolean.FALSE);
			cardUsage = cuRelation_.updateUsageSummary(cardUsage);
			return cardUsage;
		} catch (Exception e) {
			log.error(e);
			throw new UserProfileException(e);
		}
	}
	
	public void setOptionalClaims(CUID cuid, URI site, Set optionalClaims) throws UserProfileException {
		try {
			if (cuid == null || site == null)
				throw new IllegalArgumentException("Both cuid and site must be set!");
			
			cuRelation_.setCuid(cuid);
			cuRelation_.setOptionalClaims(new WebForm(site, null, null, null), optionalClaims);
		} catch (Exception e) {
			log.error(e);
			throw new UserProfileException(e);
		}		
	}
	
	public void setOptionalClaims(CUID cuid, WebForm form, Set optionalClaims) throws UserProfileException {
		try {
			if (cuid == null || form == null)
				throw new IllegalArgumentException("Both cuid and form must be set!");
			
			cuRelation_.setCuid(cuid);
			cuRelation_.setOptionalClaims(form, optionalClaims);
		} catch (Exception e) {
			log.error(e);
			throw new UserProfileException(e);
		}		
	}
	
	public Set getOptionalClaims(CUID cuid, URI site) throws UserProfileException {
		try {
			if (cuid == null || site == null)
				throw new IllegalArgumentException("Both cuid and site must be set!");
			
			cuRelation_.setCuid(cuid);
			return cuRelation_.getOptionalClaims(new WebForm(site, null, null, null));
		} catch (Exception e) {
			log.error(e);
			throw new UserProfileException(e);
		}
	}
	
	public Set getOptionalClaims(CUID cuid, WebForm form) throws UserProfileException {
		try {
			if (cuid == null || form == null)
				throw new IllegalArgumentException("Both cuid and form must be set!");
			
			cuRelation_.setCuid(cuid);
			return cuRelation_.getOptionalClaims(form);
		} catch (Exception e) {
			log.error(e);
			throw new UserProfileException(e);
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.higgins.icard.userprofile.ICardUsageManager#clearUsedAlwaysUrls(org.eclipse.higgins.icard.CUID)
	 */
	public void clearUsedAlwaysURLs(CUID cuid) throws UserProfileException {
		try {
			if (cuid == null)
				throw new IllegalArgumentException("Card ID can not be null");

			cuRelation_.setCuid(cuid);
			cuRelation_.clearUsedAlwaysURLs();
			//cuRelation_.clearLastUsed(); 
		} catch (Exception e) {
			log.error(e);
			throw new UserProfileException(e);
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.icard.userprofile.ICardUsageManager#getCardHistory(org.eclipse.higgins.icard.CUID)
	 */
	public List getCardHistory(CUID cuid) throws UserProfileException {
		try {
			if (cuid == null)
				throw new IllegalArgumentException("Parameter \"cuid\" is null");
			
			createHistory(cuid);
			return history_.getCardHistory();
		} catch (Exception e) {
			log.error(e);
			throw new UserProfileException(e);
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.icard.userprofile.ICardUsageManager#getCardHistory(org.eclipse.higgins.icard.CUID, int, int, java.lang.String, boolean)
	 */
	public List getCardHistory(CUID cuid, int startFrom, int size, String orderBy, boolean desc)
			throws UserProfileException {
		try {
			if (cuid == null)
				throw new IllegalArgumentException("Parameter \"cuid\" is null");
			
			createHistory(cuid);
			return history_.getCardHistory(startFrom, size, orderBy, desc);
		} catch (Exception e) {
			log.error(e);
			throw new UserProfileException(e);
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.icard.userprofile.ICardUsageManager#getCardUsageSummary(org.eclipse.higgins.icard.CUID)
	 */
	public List getCardUsageSummary(CUID cuid) throws UserProfileException {
		try {
			if (cuid == null)
				throw new IllegalArgumentException("Parameter \"cardCredentialInfo\" is null");

			cuRelation_.setCuid(cuid);
			return cuRelation_.getCardUsageSummary();
		} catch (Exception e) {
			log.error(e);
			e.printStackTrace();
			throw new UserProfileException(e);
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.icard.userprofile.ICardUsageManager#getCardUsageSummary(org.eclipse.higgins.icard.CUID, int, int, java.lang.String, boolean)
	 */
	public List getCardUsageSummary(CUID cuid, int startFrom, int size, String orderBy, boolean desc)
			throws UserProfileException {
		try {
			if (cuid == null)
				throw new IllegalArgumentException("Parameter \"cardCredentialInfo\" is null");
			log.debug(" *** CardUsageManager.getCardUsageSummary"); 
			log.debug(" *** " + cuid + ", " + startFrom + ", " + size + ", " + orderBy + ", " + desc);
			log.debug(" *** user " + dsUser_.getEntityID());

			cuRelation_.setCuid(cuid);
			return cuRelation_.getCardUsageSummary(startFrom, size, orderBy, desc);
		} catch (Exception e) {
			log.error(e);
			e.printStackTrace();
			throw new UserProfileException(e);
		}

	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.icard.userprofile.ICardUsageManager#getHistoryTotalSize(org.eclipse.higgins.icard.CUID)
	 */
	public int getHistoryTotalSize(CUID cuid) throws UserProfileException {
		try {
			if (cuid == null)
				throw new IllegalArgumentException("Parameter \"cuid\" is null");
			
			createHistory(cuid);
			return history_.getTotalSize();
		} catch (Exception e) {
			log.error(e);
			throw new UserProfileException(e);
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.icard.userprofile.ICardUsageManager#getSummaryTotalSize(org.eclipse.higgins.icard.CUID)
	 */
	public int getSummaryTotalSize(CUID cuid) throws UserProfileException {
		try {
			if (cuid == null)
				throw new IllegalArgumentException("Parameter \"cuid\" is null");

			cuRelation_.setCuid(cuid);
			return cuRelation_.getTotalSize();
		} catch (Exception e) {
			log.error(e);
			e.printStackTrace();
			throw new UserProfileException(e);
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.icard.userprofile.ICardUsageManager#getUsedAlwaysCard(java.net.URI)
	 */
	public CUID getUsedAlwaysCard(URI url) throws UserProfileException {
		try {
			if (url == null)
				throw new IllegalArgumentException("Parameter \"url\" is null");

			return cuRelation_.getUsedAlwaysCard(new WebForm(url, null, null, null));
		} catch (Exception e) {
			log.error(e);
			e.printStackTrace();
			throw new UserProfileException(e);
		}
	}
	
	public CUID getUsedAlwaysCard(WebForm form) throws UserProfileException {
		try {
			if (form == null)
				throw new IllegalArgumentException("Parameter \"form\" is null");

			return cuRelation_.getUsedAlwaysCard(form);
		} catch (Exception e) {
			log.error(e);
			e.printStackTrace();
			throw new UserProfileException(e);
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.icard.userprofile.ICardUsageManager#getUsedAlwaysUrls(org.eclipse.higgins.icard.CUID)
	 */
	public List getUsedAlwaysURLs(CUID cuid) throws UserProfileException {
		try {
			if (cuid == null)
				throw new IllegalArgumentException("Parameter \"cuid\" is null");

			cuRelation_.setCuid(cuid);
			return cuRelation_.getUsedAlwaysUrls();
		} catch (Exception e) {
			log.error(e);
			e.printStackTrace();
			throw new UserProfileException(e);
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.icard.userprofile.ICardUsageManager#getUsedCards(java.net.URI)
	 */
	public List getUsedCards(URI url) throws UserProfileException {
		try {
			if (url == null)
				throw new IllegalArgumentException("Parameter \"url\" is null");

			return cuRelation_.getUsedCards(new WebForm(url, null, null, null));
		} catch (Exception e) {
			log.error(e);
			e.printStackTrace();
			throw new UserProfileException(e);
		}
	}
	
	public List getUsedCards(WebForm form) throws UserProfileException {
		try {
			if (form == null)
				throw new IllegalArgumentException("Parameter \"form\" is null");

			return cuRelation_.getUsedCards(form);
		} catch (Exception e) {
			log.error(e);
			e.printStackTrace();
			throw new UserProfileException(e);
		}
	}

	public Map getClaimValuesMRU() throws UserProfileException {
		try {
			Map res = new HashMap();
			IAttribute attCV = dsUser_.getAttribute(IdasCtxUris.UP_claimValuesMRU);
			Iterator itValsCV = attCV.getValues();
			while (itValsCV.hasNext()) {
				IAttributeValue valCV = (IAttributeValue) itValsCV.next();
				if (!valCV.isSimple()) {
					String claimType = (String) IdasUtils.getSimpleValueData((IComplexAttrValue)valCV,
							IdasCtxUris.CU_claimType);
					IAttribute attMRUValues = ((IComplexAttrValue)valCV).getAttribute(IdasCtxUris.CU_claimValue);
					Iterator itMRUValues = attMRUValues.getValues();
					Set resValues = new HashSet();
					while (itMRUValues.hasNext()) {
						ISimpleAttrValue val = (ISimpleAttrValue) itMRUValues.next();
						String sVal = (String) val.getData();
						resValues.add(sVal);
					}
					res.put(claimType, resValues);
				} else 
					throw new UserProfileException("Attribute " + (attCV.getAttrID() != null ? attCV.getAttrID().toString() : "")
							+ " contains simple value");
			}
			return res;
		} catch (Exception e) {
			log.error(e);
			e.printStackTrace();
			throw new UserProfileException(e);
		}
	}

	public Set getClaimValuesMRU(String claimType) throws UserProfileException {
		try {
			if (claimType == null)
				throw new IllegalArgumentException("Parameter \"claimType\" is null");

			Set resValues = new HashSet();
			IAttribute attCV = dsUser_.getAttribute(IdasCtxUris.UP_claimValuesMRU);
			Iterator itValsCV = attCV.getValues();
			while (itValsCV.hasNext()) {
				IAttributeValue valCV = (IAttributeValue) itValsCV.next();
				if (!valCV.isSimple()) {
					if ( claimType.equals((String) IdasUtils.getSimpleValueData((IComplexAttrValue)valCV,
							IdasCtxUris.CU_claimType)) ) {
						IAttribute attMRUValues = ((IComplexAttrValue)valCV).getAttribute(IdasCtxUris.CU_claimValue);
						Iterator itMRUValues = attMRUValues.getValues();
						while (itMRUValues.hasNext()) {
							ISimpleAttrValue val = (ISimpleAttrValue) itMRUValues.next();
							String sVal = (String) val.getData();
							resValues.add(sVal);
						}
						break;
					}
				} else 
					throw new UserProfileException("Attribute " + (attCV.getAttrID() != null ? attCV.getAttrID().toString() : "")
							+ " contains simple value");
			}
			return resValues;
		} catch (Exception e) {
			log.error(e);
			e.printStackTrace();
			throw new UserProfileException(e);
		}
	}

	public void setClaimValuesMRU(String claimType, Set claimValues) throws UserProfileException {
		try {
			if (claimType == null)
				throw new IllegalArgumentException("Parameter \"claimType\" is null");
			
			IAttribute attCV = dsUser_.getAttribute(IdasCtxUris.UP_claimValuesMRU);
			Iterator itValsCV = attCV.getValues();
			while (itValsCV.hasNext()) {
				IAttributeValue valCV = (IAttributeValue) itValsCV.next();
				if (!valCV.isSimple()) {
					if ( claimType.equals((String) IdasUtils.getSimpleValueData((IComplexAttrValue)valCV,
							IdasCtxUris.CU_claimType)) ) {
						IAttribute attMRUValues = ((IComplexAttrValue) valCV).getAttribute(IdasCtxUris.CU_claimValue);
						attMRUValues.remove();
						if (claimValues != null && claimValues.size() > 0) {
							Iterator itMRUValues = claimValues.iterator();
							while (itMRUValues.hasNext()) {
								attMRUValues.addSimpleValue(null, (String) itMRUValues.next());
							}
						} else
							valCV.remove();//delete whole record with this claim type
						return;
					}
				} else 
					throw new UserProfileException("Attribute " + (attCV.getAttrID() != null ? attCV.getAttrID().toString() : "")
							+ " contains simple value");
			}
			
			IComplexAttrValue valCV = attCV.addComplexValue(IdasCtxUris.CU_ClaimValuesMRU);
			valCV.getAttribute(IdasCtxUris.CU_claimType).addSimpleValue(null, claimType);
			if (claimValues != null) {
				IAttribute attMRUValues = valCV.getAttribute(IdasCtxUris.CU_claimValue);
				Iterator itMRUValues = claimValues.iterator();
				while (itMRUValues.hasNext()) {
					attMRUValues.addSimpleValue(null, (String) itMRUValues.next());
				}
			}
			
		} catch (Exception e) {
			log.error(e);
			e.printStackTrace();
			throw new UserProfileException(e);
		}
	}

}
