/** 
 * Copyright (c) 2007 Parity Communications, Inc.  
 * All rights reserved. This program and the accompanying materials  
 * are made available under the terms of the Eclipse Public License v1.0  
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 *  
 * Contributors: 
 *		Yuriy Pilipenko - API and implementation 
 * 
 */
package org.eclipse.higgins.userprofile.idas;

import java.net.URI;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import org.eclipse.higgins.icard.CUID;
import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IAttributeValue;
import org.eclipse.higgins.idas.api.IComplexAttrValue;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.userprofile.UserProfileException;
import org.eclipse.higgins.userprofile.entity.CardUsage;
import org.eclipse.higgins.userprofile.entity.WebForm;
import org.eclipse.higgins.userprofile.idas.util.IdasCtxUris;
import org.eclipse.higgins.userprofile.idas.util.IdasUtils;

/**
 * @author PilipenkoYN
 *
 */
class CardHistoryBinding {
	private IContext ctx_;
	private IAttribute attCardUsage_;	
	private List cardHistory_ = new ArrayList();
	private CUID cuid_;	

	public CardHistoryBinding(IEntity dsUserProfile, CUID cuid) throws Exception {
		cuid_ = cuid;
		ctx_ = dsUserProfile.getContext();
		IAttribute attCCI = dsUserProfile.getAttribute(IdasCtxUris.UP_cardInformation);
		Iterator valsCCI = attCCI.getValues();
		IComplexAttrValue valCCI = null;
		while (valsCCI.hasNext()) {
			IAttributeValue val = (IAttributeValue) valsCCI.next();
			if (!val.isSimple()) {
				String cuidFound = (String) IdasUtils.getSimpleValueData((IComplexAttrValue) val, IdasCtxUris.CC_cuid);
				if (cuidFound.equals(cuid.toString())) {
					valCCI = (IComplexAttrValue) val;
					break;
				}
			} else {
				throw new UserProfileException("Complex value expected, but found " + attCCI.getModel().getType());
			}
		}
		if (valCCI == null) {
			valCCI = attCCI.addComplexValue(null);
			valCCI.getSingleValuedAttribute(IdasCtxUris.CC_cuid).addSimpleValue(null, cuid.toString());
		}
		attCardUsage_ = valCCI.getAttribute(IdasCtxUris.CCI_cardUsage);
		
		Iterator valsCardUsage = attCardUsage_.getValues();
		while (valsCardUsage.hasNext()) {
			IAttributeValue val = (IAttributeValue) valsCardUsage.next();
			if (!val.isSimple()) {
				CardUsage cu = new CardUsage();
				cu.setCuid(cuid);
				cu.setForm(new WebForm((URI.create((String)IdasUtils.getSimpleValueData((IComplexAttrValue)val, IdasCtxUris.CCI_CU_site))),
						(String)IdasUtils.getSimpleValueData((IComplexAttrValue)val, IdasCtxUris.CU_formName),
						(String)IdasUtils.getSimpleValueData((IComplexAttrValue)val, IdasCtxUris.CU_formId),
						(String)IdasUtils.getSimpleValueData((IComplexAttrValue)val, IdasCtxUris.CU_formAction)));
				cu.setDate((Date) IdasUtils.getSimpleValueData((IComplexAttrValue) val, IdasCtxUris.CCI_CU_date));
				cardHistory_.add(cu);
			} else {
				throw new UserProfileException("Complex value expected, but found " + attCardUsage_.getModel().getType());
			}
		}
	}
	
	static class CardUsageComparator implements Comparator {
		private String orderBy = null;
		private boolean desc = false;

		public CardUsageComparator(String orderBy, boolean desc) {
			super();
			this.orderBy = orderBy;
			this.desc = desc;
		}

		private void checkClassCast(Object cu1, Object cu2) {
			if (cu1.getClass() != CardUsage.class | cu2.getClass() != CardUsage.class) 
				throw new ClassCastException("Both arguments must be of type '" + 
						CardUsage.class.getName() + "'");
		}

		public int compare(Object arg0, Object arg1) {
			checkClassCast(arg0, arg1);
			
			CardUsage cu1, cu2;
			if (!desc) {
				cu1 = (CardUsage) arg0; 
				cu2 = (CardUsage) arg1;
			} else {
				cu2 = (CardUsage) arg0; 
				cu1 = (CardUsage) arg1;
			}

			if (orderBy == CardUsage.BY_SITE) 
				return (cu1.getForm().getUrl() == null && cu2.getForm().getUrl() == null) ? 0 : 
					cu1.getForm().getUrl() == null ? -1 : cu2.getForm().getUrl() == null ? 1 : 
						cu1.getForm().getUrl().toString().compareTo(cu2.getForm().getUrl().toString());
			else if (orderBy == CardUsage.BY_DATE)
				return (cu1.getDate() == null && cu2.getDate() == null) ? 0 : 
					cu1.getDate() == null ? -1 : cu2.getDate() == null ? 1 : 
						cu1.getDate().compareTo(cu2.getDate());
			else if (orderBy == CardUsage.BY_USE) {
				boolean b1 = cu1.getUseAlways() == null ? false : cu1.getUseAlways().booleanValue(); 
				boolean b2 = cu2.getUseAlways() == null ? false : cu2.getUseAlways().booleanValue();
				return (b1 == b2 ? 0 : (b1 ? 1 : -1));
			} else
				return 0;
		}
	}
	
	public CardUsage saveHistoryEvent(URI uri) throws Exception {
		CardUsage cardUsage = new CardUsage();
		cardUsage.setCuid(cuid_);
		cardUsage.setSite(uri);
		cardUsage.setDate(new Date());

		IComplexAttrValue valCU = attCardUsage_.addComplexValue(null);
		valCU.getSingleValuedAttribute(IdasCtxUris.CCI_CU_site).addSimpleValue(null, cardUsage.getSite().toString());
		valCU.getSingleValuedAttribute(IdasCtxUris.CCI_CU_date).addSimpleValue(null, cardUsage.getDate());

		return cardUsage;
	}
	
	public CardUsage saveHistoryEvent(WebForm form) throws Exception {
		CardUsage cardUsage = new CardUsage();
		cardUsage.setCuid(cuid_);
		cardUsage.setForm(form);
		cardUsage.setDate(new Date());

		IComplexAttrValue valCU = attCardUsage_.addComplexValue(null);
		valCU.getSingleValuedAttribute(IdasCtxUris.CCI_CU_site).addSimpleValue(null, cardUsage.getForm().getUrl().toString());
		valCU.getSingleValuedAttribute(IdasCtxUris.CCI_CU_date).addSimpleValue(null, cardUsage.getDate());
		valCU.getSingleValuedAttribute(IdasCtxUris.CU_formName).addSimpleValue(null, cardUsage.getForm().getFormName());
		valCU.getSingleValuedAttribute(IdasCtxUris.CU_formId).addSimpleValue(null, cardUsage.getForm().getFormId());
		valCU.getSingleValuedAttribute(IdasCtxUris.CU_formAction).addSimpleValue(null, cardUsage.getForm().getFormAction());

		return cardUsage;
	}

	public void clearHistory() throws IdASException {
		attCardUsage_.remove();
	}
	
	public List getCardHistory() {
		return cardHistory_;
	}

	public List getCardHistory(int startFrom, int size, String orderBy, boolean desc) throws Exception {
		int fullSize = cardHistory_.size();
		int to = startFrom + size;
		if (to > fullSize) 
			to = fullSize;
		List historyList = getCardHistory();
		Collections.sort(historyList, new CardUsageComparator(orderBy, desc));
		return historyList.subList(startFrom, to);
	}
	
	public int getTotalSize() {
		return cardHistory_.size();
	}
	
	public void finish() throws IdASException {
		ctx_.applyUpdates();
		ctx_.close();
	}
	
}
