/**
 * Copyright (c) 2007-2008 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *		Yuriy Pilipenko - API and implementation
 *
 */
package org.eclipse.higgins.userprofile.idas;

import java.io.File;
import java.net.URI;
import java.nio.ByteBuffer;
import java.security.MessageDigest;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.NameCallback;
import javax.security.auth.callback.PasswordCallback;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.cache.CacheProviderFactory;
import org.eclipse.higgins.cache.api.ICache;
import org.eclipse.higgins.cache.api.key.UserCacheKey;
import org.eclipse.higgins.cache.nocache.NOCache;
import org.eclipse.higgins.icard.CUID;
import org.eclipse.higgins.icard.common.ProviderConfiguration;
import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IAttributeValue;
import org.eclipse.higgins.idas.api.IComplexAttrValue;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.IContextId;
import org.eclipse.higgins.idas.api.IFilter;
import org.eclipse.higgins.idas.api.IFilterAttributeAssertion;
import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.ISimpleAttrValue;
import org.eclipse.higgins.idas.api.ITypedValue;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.common.AuthNNamePasswordMaterials;
import org.eclipse.higgins.idas.registry.IdASRegistry;
import org.eclipse.higgins.idas.registry.contextid.ContextIdFactory;
import org.eclipse.higgins.idas.registry.discovery.FileDiscovery;
import org.eclipse.higgins.registry.IConfiguration;
import org.eclipse.higgins.userprofile.CategoryNotFoundException;
import org.eclipse.higgins.userprofile.ICardUsageManager;
import org.eclipse.higgins.userprofile.IUserProfileService;
import org.eclipse.higgins.userprofile.NoSuchEntityException;
import org.eclipse.higgins.userprofile.UserAlreadyExistsException;
import org.eclipse.higgins.userprofile.UserNotFoundException;
import org.eclipse.higgins.userprofile.UserProfileAuthenticationException;
import org.eclipse.higgins.userprofile.UserProfileException;
import org.eclipse.higgins.userprofile.WrongPasswordException;
import org.eclipse.higgins.userprofile.entity.Captcha;
import org.eclipse.higgins.userprofile.entity.CardCredential;
import org.eclipse.higgins.userprofile.entity.CardInformation;
import org.eclipse.higgins.userprofile.entity.Category;
import org.eclipse.higgins.userprofile.entity.PolicyVersion;
import org.eclipse.higgins.userprofile.entity.UserProfile;
import org.eclipse.higgins.userprofile.entity.WebForm;
import org.eclipse.higgins.userprofile.idas.util.IdasCtxUris;
import org.eclipse.higgins.userprofile.idas.util.IdasUtils;

public class IdasBasedUserProfileService implements IUserProfileService {
	private String IDAS_CONTEXT_ID = "idasContext.id";
	private String DISCOVERY_FILE = "idasDiscovery.filename";
	private String IDAS_USER_NAME = "idasUser.name";
	private String IDAS_USER_PASSWORD = "idasUser.password";
	// ***************************************

	private static Log log = LogFactory.getLog(IdasBasedUserProfileService.class);
	private String id_ = "org.eclipse.higgins.userprofile.idas";

	private IConfiguration config_;
	private IdASRegistry registry_;
	private String sCtx;
	private IContextId ctxId;
	private String sUser;
	private String sPassword;

	// quick fix
	private static ICache userProfileCache = CacheProviderFactory.getCacheProvider().getCache("UserProfile.UserProfiles");
	private static boolean isNoCache = userProfileCache instanceof NOCache;

	// cardInformation entity cache
	// private static ICache cardInformationCache =
	// CacheProviderFactory.getCacheProvider().getCache("UserProfileCardInforamationCache");
	// private static boolean isNoCache = cardInformationCache instanceof
	// NOCache;

	/**
	 * TODO USer name should not be changed. This logic should be removed from
	 * userprofile.
	 * 
	 * @param userId
	 * @return
	 */
	private static String normalizeUserID(String userId) {
		return "urn:" + userId.replace(' ', '_');
	}

	private static class UserCreds {
		private String name_;
		private String password_;

		public UserCreds(final CallbackHandler handler) throws UserProfileException {
			NameCallback nc = new NameCallback("User name: ");
			PasswordCallback pc = new PasswordCallback("Password: ", false);
			Callback[] callbacks = new Callback[] { nc, pc };
			try {
				handler.handle(callbacks);
			} catch (Exception e) {
				log.error(e);
				throw new UserProfileAuthenticationException("Can't get user creds caused by " + e);
			}
			this.name_ = nc.getName();
			if (getName() == null || getName().trim().length() == 0)
				throw new UserProfileAuthenticationException("Empty user name.");

			if (!isUserId()) { // if name is userId, dont process password
				this.password_ = new String(pc.getPassword());
				pc.clearPassword();
				if (this.password_ == null || this.password_.trim().length() == 0)
					throw new UserProfileAuthenticationException("Empty password.");
			}
		}

		public boolean isUserId() {
			return (this.name_.length() >= 4 ? this.name_.substring(0, 4).equals("urn:") : false);
		}

		public String getName() {
			return this.name_;
		}

		public String getUriName() {
			if (isUserId())
				return this.name_;
			return normalizeUserID(this.name_);
		}

		public static ByteBuffer getPasswordHash(final String password) throws UserProfileException {
			if (password != null && password.length() > 0)
				try {
					MessageDigest md = MessageDigest.getInstance("SHA-256");
					return ByteBuffer.wrap(md.digest(password.getBytes("UTF-8")));
				} catch (Exception e) {
					log.error(e);
					throw new UserProfileException(e);
				}
			else
				return null;
		}

		public void checkPasswordHash(final ByteBuffer pwHash) throws UserProfileException {
			ByteBuffer testedHash = getPasswordHash(this.password_);
			if (!pwHash.position(0).equals(testedHash.position(0)))
				throw new WrongPasswordException("Invalid password! User " + getName());
		}
	}

	private class CachedItem {
		private UserProfile userProfile;
		private ByteBuffer passwordHash;
		private ByteBuffer newPasswordHash;
	}

	private void loadConfig() throws IdASException {
		String sFactories = getConfiguration().getProperty(this.DISCOVERY_FILE, "contextfactories.xrds");
		FileDiscovery factories = new FileDiscovery(new File(sFactories));
		this.registry_ = IdASRegistry.getInstance();
		this.registry_.setDiscovery(factories);
		this.sCtx = getConfiguration().getProperty(this.IDAS_CONTEXT_ID, "userProfileContext.xrds");
		this.ctxId = ContextIdFactory.fromString(sCtx);
		this.sUser = getConfiguration().getProperty(this.IDAS_USER_NAME, "testUser");
		this.sPassword = getConfiguration().getProperty(this.IDAS_USER_PASSWORD, "testPassword");
	}

	private IContext getContext() throws IdASException, UserProfileException {
		if (this.registry_ == null)
			loadConfig();
		IContext ctx = this.registry_.createContext(this.ctxId);
		if (ctx == null)
			throw new UserProfileException("User profile context not found with ID " + this.ctxId);
		ctx.open(new AuthNNamePasswordMaterials(ctx, this.sUser, this.sPassword));
		return ctx;

		/*
		 * if (ctx == null) { ctx = createContext(); } return ctx;
		 */
	}

	private IEntity authenticate(final CallbackHandler handler) throws UserProfileException, IdASException {
		UserCreds uc = new UserCreds(handler);
		IContext ctx = getContext();
		IEntity dsUser = null;
		try {
			dsUser = ctx.getEntity(uc.getUriName());
		} catch (org.eclipse.higgins.idas.api.NoSuchEntityException e) {
			dsUser = null;
		}
		if (dsUser == null) {
			ctx.close();
			throw new UserNotFoundException("User Profile " + uc.getName() + " not found!");
		}
		if (!uc.isUserId()) { // if name is userId, dont process password
			try {
				ByteBuffer newAuthHash = (ByteBuffer) IdasUtils.getSimpleValueData(dsUser, IdasCtxUris.UP_newAuthPasswordHash);
				if (newAuthHash != null && newAuthHash.hasArray()) {
					try {
						uc.checkPasswordHash(newAuthHash);
					} catch (UserProfileAuthenticationException e) {
						uc.checkPasswordHash((ByteBuffer) IdasUtils.getSimpleValueData(dsUser, IdasCtxUris.UP_authPasswordHash));
					}
				} else {
					uc.checkPasswordHash((ByteBuffer) IdasUtils.getSimpleValueData(dsUser, IdasCtxUris.UP_authPasswordHash));
				}
			} catch (UserProfileException e) {
				ctx.close();
				throw e;
			}
		}
		return dsUser;
	}

	/**
	 * This method should replace authenticate() method after refactoring user
	 * profile
	 * 
	 * @param handler
	 * @return
	 * @throws UserProfileException
	 * @throws IdASException
	 */
	private IEntity getEntity(String userID) throws UserProfileException, IdASException {
		if (userID == null)
			throw new UserProfileException("Parameter \"userID\" is null");
		IContext ctx = getContext();
		IEntity dsUser = null;
		try {
			dsUser = ctx.getEntity(userID);
		} catch (org.eclipse.higgins.idas.api.NoSuchEntityException e) {
			dsUser = null;
		}
		if (dsUser == null) {
			ctx.close();
			throw new UserNotFoundException("User Profile " + userID + " not found!");
		}
		return dsUser;
	}

	private UserProfile loadUserProfile(String userID) throws UserProfileException, IdASException {
		IEntity ds = getEntity(userID);
		try {
			UserProfile up = initFromDS_UP(ds);
			return up;
		} finally {
			ds.getContext().close();
		}
	}

	public UserProfile getUserProfile(String userID) throws UserProfileException {
		String id = normalizeUserID(userID);
		try {
			if (isNoCache)
				return loadUserProfile(id);
			else {
				UserCacheKey key = new UserCacheKey(id);
				if (userProfileCache.isKeyInCache(key)) {
					CachedItem cached = (CachedItem) userProfileCache.get(key);
					UserProfile up = cached.userProfile;
					return up;
				} else {
					IEntity dsUser = getEntity(id);
					try {
						CachedItem cached = new CachedItem();
						cached.userProfile = initFromDS_UP(dsUser);
						cached.passwordHash = (ByteBuffer) IdasUtils.getSimpleValueData(dsUser, IdasCtxUris.UP_authPasswordHash);
						cached.newPasswordHash = (ByteBuffer) IdasUtils.getSimpleValueData(dsUser, IdasCtxUris.UP_newAuthPasswordHash);
						userProfileCache.put(key, cached);
						return cached.userProfile;
					} finally {
						dsUser.getContext().close();
					}
				}
			}
		} catch (UserProfileAuthenticationException e) {
			log.error(e, e);
			throw e;
		} catch (Exception e) {
			log.error(e, e);
			return null;
		}
	}

	private IEntity authenticate(final String userId, final byte[] passwordIsapMd5) throws UserProfileException, IdASException {
		// UserCreds uc = new UserCreds(handler);
		IContext ctx = getContext();
		IEntity dsUser = null;
		try {
			dsUser = ctx.getEntity("urn:" + userId.replace(' ', '_'));
		} catch (org.eclipse.higgins.idas.api.NoSuchEntityException e) {
			dsUser = null;
		}
		if (dsUser == null) {
			ctx.close();
			throw new UserNotFoundException("User Profile " + userId + " not found!");
		}
		ByteBuffer passwordIsapMd5stored = (ByteBuffer) IdasUtils.getSimpleValueData(dsUser, IdasCtxUris.UP_passwordIsapMd5);
		if (passwordIsapMd5stored != null && passwordIsapMd5stored.hasArray())
			if (!passwordIsapMd5stored.position(0).equals(ByteBuffer.wrap(passwordIsapMd5).position(0)))
				throw new WrongPasswordException("Invalid passwordIsapMd5! User " + userId);
		return dsUser;
	}

	private void saveToDS_UP(final UserProfile up, final IEntity ds) throws Exception {
		try {
			ds.getAttribute(IdasCtxUris.UP_loginName).addSimpleValue(ITypedValue.STRING_TYPE_URI, up.getLoginName());
			ds.getAttribute(IdasCtxUris.UP_firstName).addSimpleValue(ITypedValue.STRING_TYPE_URI, up.getFirstName());
			ds.getAttribute(IdasCtxUris.UP_lastName).addSimpleValue(ITypedValue.STRING_TYPE_URI, up.getLastName());
			ds.getAttribute(IdasCtxUris.UP_email).addSimpleValue(ITypedValue.STRING_TYPE_URI, up.getEmail());
			// ds.getAttribute(IdasCtxUris.UP_password).addSimpleValue(ITypedValue.STRING_TYPE_URI,
			// up.getPassword());
			ds.getAttribute(IdasCtxUris.UP_modified).addSimpleValue(ITypedValue.DATETIME_TYPE_URI, up.getModified());
			ds.getAttribute(IdasCtxUris.UP_status).addSimpleValue(ITypedValue.STRING_TYPE_URI, up.getStatus());
			ds.getAttribute(IdasCtxUris.UP_usedFrom).addSimpleValue(ITypedValue.STRING_TYPE_URI, up.getUsedFrom());
			ds.getAttribute(IdasCtxUris.UP_privateUserINumber).addSimpleValue(ITypedValue.STRING_TYPE_URI, up.getPrivateUserINumber());

			IdasUtils.saveSimpleValueCollection(ds.getAttribute(IdasCtxUris.UP_privateSelectorINumbers), up.getPrivateSelectorINumbers());
			IdasUtils.saveComplexValueMap(ds.getAttribute(IdasCtxUris.UP_oneTimePasswords), IdasCtxUris.OT_otpTimestamp, IdasCtxUris.OT_otpValue, up
					.getOneTimePasswords());
			IdasUtils.saveSimpleValueCollection(ds.getAttribute(IdasCtxUris.UP_blackListedSites), up.getBlackListedSites());

			ds.getAttribute(IdasCtxUris.UP_privateKey).addSimpleValue(ITypedValue.BASE64BINARY_TYPE_URI,
					(up.getPrivateKey() == null ? null : ByteBuffer.wrap(up.getPrivateKey())));
			ds.getAttribute(IdasCtxUris.UP_passwordIsapMd5).addSimpleValue(ITypedValue.BASE64BINARY_TYPE_URI,
					(up.getPasswordIsapMd5() == null ? null : ByteBuffer.wrap(up.getPasswordIsapMd5())));
			ds.getAttribute(IdasCtxUris.UP_sms).addSimpleValue(ITypedValue.STRING_TYPE_URI, up.getSms());
			ds.getAttribute(IdasCtxUris.UP_defaultPCardCUID).addSimpleValue(ITypedValue.STRING_TYPE_URI, up.getDefaultPCardCUID());
			ds.getAttribute(IdasCtxUris.UP_passphraseHash).addSimpleValue(ITypedValue.STRING_TYPE_URI, up.getPassphraseHash());
			ds.getAttribute(IdasCtxUris.UP_passwordResetCode).addSimpleValue(ITypedValue.STRING_TYPE_URI, up.getPasswordResetCode());
			ds.getAttribute(IdasCtxUris.UP_resetCodeDate).addSimpleValue(ITypedValue.DATETIME_TYPE_URI, up.getResetCodeDate());
		} catch (Exception e) {
			ds.getContext().cancelUpdates();
			throw e;
		}
	}

	private UserProfile initFromDS_UP(final IEntity ds) throws UserProfileException, IdASException {
		UserProfile up = new UserProfile();

		up.setLoginName((String) IdasUtils.getSimpleValueData(ds, IdasCtxUris.UP_loginName));
		// up.setPassword(((String) IdasUtils.getSimpleValueData(ds,
		// IdasCtxUris.UP_password)));
		up.setFirstName((String) IdasUtils.getSimpleValueData(ds, IdasCtxUris.UP_firstName));
		up.setLastName((String) IdasUtils.getSimpleValueData(ds, IdasCtxUris.UP_lastName));
		up.setEmail((String) IdasUtils.getSimpleValueData(ds, IdasCtxUris.UP_email));
		up.setCreated((Date) IdasUtils.getSimpleValueData(ds, IdasCtxUris.UP_created));
		up.setModified((Date) IdasUtils.getSimpleValueData(ds, IdasCtxUris.UP_modified));
		up.setStatus((String) IdasUtils.getSimpleValueData(ds, IdasCtxUris.UP_status));
		up.setUsedFrom((String) IdasUtils.getSimpleValueData(ds, IdasCtxUris.UP_usedFrom));
		// up.setPasswordHash((ByteBuffer) IdasUtils.getSimpleValueData(ds,
		// IdasCtxUris.UP_authPasswordHash));
		up.setPrivateUserINumber((String) IdasUtils.getSimpleValueData(ds, IdasCtxUris.UP_privateUserINumber));

		up.setPrivateSelectorINumbers(IdasUtils.loadSimpleValueList(ds.getAttribute(IdasCtxUris.UP_privateSelectorINumbers)));
		up.setOneTimePasswords(IdasUtils.loadComplexValueMap(ds.getAttribute(IdasCtxUris.UP_oneTimePasswords), IdasCtxUris.OT_otpTimestamp,
				IdasCtxUris.OT_otpValue));
		up.setBlackListedSites(IdasUtils.loadSimpleValueSet(ds.getAttribute(IdasCtxUris.UP_blackListedSites)));

		up.setPrivateKey(IdasUtils.getSimpleValueDataAsByteArray(ds, IdasCtxUris.UP_privateKey));
		up.setPasswordIsapMd5(IdasUtils.getSimpleValueDataAsByteArray(ds, IdasCtxUris.UP_passwordIsapMd5));
		up.setSms((String) IdasUtils.getSimpleValueData(ds, IdasCtxUris.UP_sms));
		up.setDefaultPCardCUID((String) IdasUtils.getSimpleValueData(ds, IdasCtxUris.UP_defaultPCardCUID));
		up.setPassphraseHash((String) IdasUtils.getSimpleValueData(ds, IdasCtxUris.UP_passphraseHash));
		up.setPasswordResetCode((String) IdasUtils.getSimpleValueData(ds, IdasCtxUris.UP_passwordResetCode));
		up.setResetCodeDate((Date) IdasUtils.getSimpleValueData(ds, IdasCtxUris.UP_resetCodeDate));
		up.setUserIdentifier(ds.getEntityID());
		return up;
	}

	private void saveToDS_cat(final Category cat, final IEntity ds) throws Exception {
		// changed to manage ID inside idas - Entity ID
		cat.setId(ds.getEntityID());
		ds.getAttribute(IdasCtxUris.CC_parentId).addSimpleValue(ITypedValue.STRING_TYPE_URI, cat.getParentId());
		ds.getAttribute(IdasCtxUris.CC_name).addSimpleValue(ITypedValue.STRING_TYPE_URI, cat.getName());
		ds.getAttribute(IdasCtxUris.CC_icon).addSimpleValue(ITypedValue.STRING_TYPE_URI, cat.getIcon());
		ds.getAttribute(IdasCtxUris.CC_selectedIcon).addSimpleValue(ITypedValue.STRING_TYPE_URI, cat.getSelectedIcon());
		ds.getAttribute(IdasCtxUris.CC_idx).addSimpleValue(ITypedValue.INTEGER_TYPE_URI, new Integer(cat.getIdx()));
		ds.getAttribute(IdasCtxUris.CC_type).addSimpleValue(ITypedValue.STRING_TYPE_URI, cat.getType());

		IAttribute attr = ds.getAttribute(IdasCtxUris.CC_cuid);
		attr.remove();
		if (cat.getCardList() != null && !cat.getCardList().isEmpty())
			for (Iterator iter = cat.getCardList().iterator(); iter.hasNext();)
				// TODO !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
				// CUID cuid = (CUID)iter.next();
				attr.addSimpleValue(ITypedValue.STRING_TYPE_URI, iter.next());
	}

	private Category initFromDS_cat(final IEntity ds) throws UserProfileException, IdASException {
		Category cat = new Category();

		cat.setId(ds.getEntityID());
		cat.setParentId((String) IdasUtils.getSimpleValueData(ds, IdasCtxUris.CC_parentId));
		cat.setName((String) IdasUtils.getSimpleValueData(ds, IdasCtxUris.CC_name));
		cat.setIcon((String) IdasUtils.getSimpleValueData(ds, IdasCtxUris.CC_icon));
		cat.setSelectedIcon((String) IdasUtils.getSimpleValueData(ds, IdasCtxUris.CC_selectedIcon));
		Integer idx = (Integer) IdasUtils.getSimpleValueData(ds, IdasCtxUris.CC_idx);
		if (idx == null)
			cat.setIdx(0);
		else
			cat.setIdx(idx.intValue());
		cat.setType((String) IdasUtils.getSimpleValueData(ds, IdasCtxUris.CC_type));

		IAttribute att = ds.getAttribute(IdasCtxUris.CC_cuid);

		String cuid = null;
		List cuids = new ArrayList();
		if (att.getValues() != null)
			for (Iterator iter = att.getValues(); iter.hasNext();) {
				IAttributeValue v = (IAttributeValue) iter.next();
				if (v.isSimple()) {
					ISimpleAttrValue sv = (ISimpleAttrValue) v;
					cuid = (String) sv.getData();
				} else
					throw new UserProfileException("Attribute " + (att.getAttrID() != null ? att.getAttrID().toString() : "")
							+ " contains non simple value");

				cuids.add(cuid);
			}
		cat.setCardList(cuids);

		return cat;
	}

	public IdasBasedUserProfileService() {
	}

	public UserProfile addUserProfile(final CallbackHandler handler, final UserProfile userProfile) throws UserProfileException {
		try {
			UserCreds uc = new UserCreds(handler);
			if (userIdentifierExists(uc.getName()))
				throw new UserAlreadyExistsException("User with identifier \"" + uc.getName() + "\" already exists!");
			if (userProfile == null)
				throw new IllegalArgumentException("Parameter \"userProfile\" is null");

			IEntity ds = getContext().addEntity(IdasCtxUris.UP_UserProfile, uc.getUriName());
			try {
				if (ds == null)
					throw new UserProfileException("UserProfile not added.");

				userProfile.setCreated(new Date()); // TODO do not modify source
				// data, modify and return
				// the new object
				userProfile.setUserIdentifier(ds.getEntityID());
				ds.getAttribute(IdasCtxUris.UP_created).addSimpleValue(ITypedValue.DATETIME_TYPE_URI, userProfile.getCreated());
				ds.getAttribute(IdasCtxUris.UP_authPasswordHash).addSimpleValue(ITypedValue.BASE64BINARY_TYPE_URI,
						UserCreds.getPasswordHash(uc.password_));

				saveToDS_UP(userProfile, ds);
				ds.getContext().applyUpdates();
			} finally {
				ds.getContext().close();
			}

			if (!isNoCache) {
				CachedItem cached = new CachedItem();
				cached.userProfile = userProfile;
				cached.passwordHash = UserCreds.getPasswordHash(uc.password_);
				UserCacheKey key = new UserCacheKey(uc.getUriName());
				userProfileCache.put(key, cached);
			}
		} catch (Exception e) {
			log.error(e, e);
			throw new UserProfileException(e);
		}
		return userProfile;
	}

	public void deleteUserProfile(final CallbackHandler handler) throws UserProfileException {
		try {
			IEntity dsUser = authenticate(handler);
			try {
				deleteAllCategories(dsUser);
				dsUser.remove();
				dsUser.getContext().applyUpdates();
			} finally {
				dsUser.getContext().close();
			}

			if (!isNoCache) {
				UserCacheKey key = new UserCacheKey(new UserCreds(handler).getUriName());
				userProfileCache.remove(key);
			}
		} catch (Exception e) {
			log.error(e, e);
			throw new UserProfileException(e);
		}
	}

	private void deleteAllCategories(final IEntity dsUser) throws IdASException, UserProfileException {
		IContext ctx = dsUser.getContext();
		IFilter filter = ctx.buildFilter();
		IFilterAttributeAssertion fas = ctx.buildAttributeAssertion();
		fas.includeSubtypes(true);
		fas.setComparator(IFilterAttributeAssertion.COMP_ATTR_EQ);
		fas.setID(IdasCtxUris.CC_userIdentifier);
		ISimpleAttrValue val = ctx.buildSimpleAttrValue(ITypedValue.STRING_TYPE_URI, dsUser.getEntityID());
		fas.setAssertionValue(val);
		filter.setAssertion(fas);

		Iterator subjList = ctx.getEntities(filter);
		while (subjList.hasNext()) {
			IEntity subj = (IEntity) subjList.next();
			subj.remove();
		}
	}

	private UserProfile loadUserProfile(final CallbackHandler handler) throws UserProfileException, IdASException {
		IEntity ds = authenticate(handler);
		try {
			UserProfile up = initFromDS_UP(ds);
			return up;
		} finally {
			ds.getContext().close();
		}
	}

	public UserProfile getUserProfile(final CallbackHandler handler) throws UserProfileException {
		try {
			if (isNoCache)
				return loadUserProfile(handler);
			else {
				UserCreds userCreds = new UserCreds(handler);
				UserCacheKey key = new UserCacheKey(userCreds.getUriName());

				if (userProfileCache.isKeyInCache(key)) {
					CachedItem cached = (CachedItem) userProfileCache.get(key);
					UserProfile up = cached.userProfile;

					if (!userCreds.isUserId()) // if name is userId, dont
						// process password
						if (cached.newPasswordHash != null)
							try {
								userCreds.checkPasswordHash(cached.newPasswordHash);
							} catch (UserProfileAuthenticationException e) {
								userCreds.checkPasswordHash(cached.passwordHash);
							}
						else
							userCreds.checkPasswordHash(cached.passwordHash);

					return up;
				} else {
					IEntity dsUser = authenticate(handler);
					try {
						CachedItem cached = new CachedItem();
						cached.userProfile = initFromDS_UP(dsUser);
						cached.passwordHash = (ByteBuffer) IdasUtils.getSimpleValueData(dsUser, IdasCtxUris.UP_authPasswordHash);
						cached.newPasswordHash = (ByteBuffer) IdasUtils.getSimpleValueData(dsUser, IdasCtxUris.UP_newAuthPasswordHash);
						userProfileCache.put(key, cached);
						return cached.userProfile;
					} finally {
						dsUser.getContext().close();
					}
				}
			}
		} catch (UserProfileAuthenticationException e) {
			log.error(e, e);
			throw e;
		} catch (Exception e) {
			log.error(e, e);
			return null;
		}
	}

	public UserProfile modifyUserProfile(final CallbackHandler handler, final UserProfile userProfile) throws UserProfileException {
		try {
			IEntity ds = authenticate(handler);
			try {
				if (userProfile == null)
					throw new IllegalArgumentException("Parameter \"userProfile\" can not be null");

				userProfile.setModified(new Date());
				saveToDS_UP(userProfile, ds);

				ds.getContext().applyUpdates();
			} finally {
				ds.getContext().close();
			}

			if (!isNoCache) {
				UserCacheKey key = new UserCacheKey(new UserCreds(handler).getUriName());
				CachedItem cached = (CachedItem) userProfileCache.get(key);
				cached.userProfile = userProfile;
				// userProfileCache.put(key, userProfile);
				userProfileCache.put(key, cached);
			}
		} catch (Exception e) {
			log.error(e, e);
			throw new UserProfileException(e);
		}
		return userProfile;
	}

	public UserProfile modifyUserProfile(UserProfile userProfile) throws UserProfileException {
		try {
			String userId = userProfile.getUserIdentifier();
			IEntity ds = getEntity(userId);
			try {
				if (userProfile == null)
					throw new IllegalArgumentException("Parameter \"userProfile\" can not be null");
				userProfile.setModified(new Date());
				saveToDS_UP(userProfile, ds);
				ds.getContext().applyUpdates();
			} finally {
				ds.getContext().close();
			}
			if (!isNoCache) {
				UserCacheKey key = new UserCacheKey(userId);
				CachedItem cached = (CachedItem) userProfileCache.get(key);
				cached.userProfile = userProfile;
				userProfileCache.put(key, cached);
			}
		} catch (Exception e) {
			log.error(e, e);
			throw new UserProfileException(e);
		}
		return userProfile;
	}

	public boolean userIdentifierExists(final String userIdentifier) throws UserProfileException {
		IContext ctx = null;
		IEntity ds = null;
		try {
			try {
				ctx = getContext();
				ds = ctx.getEntity("urn:" + userIdentifier.replace(' ', '_'));
			} finally {
				if (ctx != null)
					ctx.close();
			}
			return ds != null;
		} catch (org.eclipse.higgins.idas.api.NoSuchEntityException e) {
			return false;
		} catch (Exception e) {
			log.error(e, e);
			throw new UserProfileException(e);
		}
	}

	public String resetPassword(final CallbackHandler handler) throws UserProfileException {
		// TODO implement resetPassword!!!
		// TODO needs password generating algorithm so as policy (letters chars
		// etc)
		/*
		 * try { IEntity ds = _ctx.getSubject(userIdentifier); if (ds == null)
		 * throw new UserMetaInfoException("User Profile " + userIdentifier + "
		 * not found!");
		 * 
		 * UserProfile up = initFromDS(ds); up.clearPassword(); //TODO get
		 * password random generated according to some policy (elaborate!!!)
		 * up.setPassword("new password".toCharArray()); up.setModified(new
		 * Date()); saveToDS(up, ds); return up; } catch (Exception e) {
		 * log.error(e); throw new UserMetaInfoException(e); }
		 */
		throw new UserProfileException("Not implemented!");
	}

	public String resolveUserIdentifier(final String privateSelectorINumber) throws UserProfileException {
		try {
			IContext ctx = getContext();
			try {
				IFilter filter = ctx.buildFilter();
				IFilterAttributeAssertion assertion = ctx.buildAttributeAssertion();
				assertion.setComparator(IFilterAttributeAssertion.COMP_ATTR_EQ);
				assertion.setID(IdasCtxUris.UP_privateSelectorINumbers);
				assertion.setAssertionValue(ctx.buildSimpleAttrValue(ITypedValue.STRING_TYPE_URI, privateSelectorINumber));
				filter.setAssertion(assertion);

				for (Iterator it = ctx.getEntities(filter); it.hasNext();) {
					IEntity e = (IEntity) it.next();
					return e.getEntityID().substring(4);
				}
			} finally {
				ctx.close();
			}

			return null;
		} catch (Exception e) {
			log.error(e, e);
			throw new UserProfileException(e);
		}
	}

	public Category addCategory(final CallbackHandler handler, final Category category) throws UserProfileException {
		try {
			IEntity dsUser = authenticate(handler);
			try {
				if (category == null)
					throw new IllegalArgumentException("Parameter \"category\" is null");

				IEntity ds = dsUser.getContext().addEntity(IdasCtxUris.CC_CardCategory, null);
				if (ds == null)
					throw new UserProfileException("Category " + category.getId() + " not added.");

				ds.getAttribute(IdasCtxUris.CC_userIdentifier).addSimpleValue(ITypedValue.STRING_TYPE_URI, dsUser.getEntityID());
				saveToDS_cat(category, ds);

				dsUser.getContext().applyUpdates();
			} finally {
				dsUser.getContext().close();
			}
		} catch (Exception e) {
			log.error(e, e);
			throw new UserProfileException(e);
		}
		return category;
	}

	public void deleteCategory(final CallbackHandler handler, final String categoryId) throws UserProfileException {
		IContext ctx = null;
		try {
			try {
				if (categoryId == null || categoryId.trim().length() == 0)
					throw new IllegalArgumentException("Parameter \"categoryId\" is null");

				ctx = authenticate(handler).getContext();
				IEntity ds = ctx.getEntity(categoryId);
				if (ds == null)
					throw new UserProfileException("Category \"" + categoryId + "\" not found");
				ds.remove();

				ctx.applyUpdates();
			} catch (IdASException e) {
				try {
					ctx.cancelUpdates();
				} catch (IdASException e1) {
					log.error(e1, e1);
				}
				log.error(e, e);
				throw new UserProfileException(e);
			}
		} finally {
			try {
				ctx.close();
			} catch (IdASException e) {
				log.error(e, e);
				throw new UserProfileException(e);
			}
		}
	}

	public List getCategories(final CallbackHandler handler) throws UserProfileException {
		log.trace("IdasBasedUserProfileService.getCategories(CallbackHandler) started");
		try {
			IEntity dsUser = authenticate(handler);
			try {
				IFilter filter = dsUser.getContext().buildFilter();
				// filter.setOperator(IFilter.OP_AND);
				IFilterAttributeAssertion fas = dsUser.getContext().buildAttributeAssertion();
				fas.includeSubtypes(true);
				fas.setComparator(IFilterAttributeAssertion.COMP_ATTR_EQ);
				fas.setID(IdasCtxUris.CC_userIdentifier);
				ISimpleAttrValue val = dsUser.getContext().buildSimpleAttrValue(ITypedValue.STRING_TYPE_URI, dsUser.getEntityID());
				fas.setAssertionValue(val);
				filter.setAssertion(fas);

				Iterator subjList = dsUser.getContext().getEntities(filter);
				List categories = new ArrayList();
				while (subjList.hasNext()) {
					IEntity subj = (IEntity) subjList.next();
					if (IdasCtxUris.CC_CardCategory.equals(subj.getModel().getType())) {
						Category category = initFromDS_cat(subj);
						categories.add(category);
					}
				}
				return categories;
			} finally {
				dsUser.getContext().close();
			}
		} catch (Exception e) {
			log.error(e, e);
			throw new UserProfileException(e);
		}
	}

	public Category modifyCategory(final CallbackHandler handler, final Category category) throws UserProfileException {
		try {
			IEntity dsUser = authenticate(handler);
			try {
				if (category == null)
					throw new IllegalArgumentException("Parameter \"category\" is null");
				if (category.getId() == null || category.getId().trim().length() == 0)
					throw new IllegalArgumentException("Category ID can not be null");

				IEntity ds = dsUser.getContext().getEntity(category.getId());
				if (ds == null)
					throw new CategoryNotFoundException("Category \"" + category.getId() + "\" not found!");
				saveToDS_cat(category, ds);
				dsUser.getContext().applyUpdates();

				return category;
			} finally {
				dsUser.getContext().close();
			}
		} catch (Exception e) {
			log.error(e, e);
			throw new UserProfileException(e);
		}
	}

	public CardInformation setCardCredential(final CallbackHandler handler, final CUID cuid, final CardCredential cardCredential)
			throws UserProfileException {
		try {
			if (cuid == null)
				throw new IllegalArgumentException("Parameter \"cuid\" is null");
			if (cardCredential == null)
				throw new IllegalArgumentException("Parameter \"cardCredential\" is null");

			IEntity dsUser = authenticate(handler);
			try {
				CardInformationBinding cib = new CardInformationBinding(dsUser);
				try {
					cib.saveCardCredential(cuid, cardCredential);

					CardInformation ci = cib.getCardInformation(cuid);
					cib.applyUpdates();

					return ci;
				} catch (Exception e) {
					cib.cancelUpdates();
					throw e;
				}
			} finally {
				dsUser.getContext().close();
			}
		} catch (Exception e) {
			log.error(e, e);
			throw new UserProfileException(e);
		}
	}

	public CardInformation clearCardCredential(final CallbackHandler handler, final CUID cuid) throws UserProfileException {
		try {
			if (cuid == null)
				throw new IllegalArgumentException("Parameter \"cuid\" is null");

			IEntity dsUser = authenticate(handler);
			try {
				CardInformationBinding cib = new CardInformationBinding(dsUser);
				try {
					cib.deleteCardCredential(cuid);
					CardInformation ci = cib.getCardInformation(cuid);

					cib.applyUpdates();
					return ci;
				} catch (Exception e) {
					cib.cancelUpdates();
					throw e;
				}
			} finally {
				dsUser.getContext().close();
			}
		} catch (Exception e) {
			log.error(e, e);
			throw new UserProfileException(e);
		}
	}

	public CardInformation getCardInformation(final CallbackHandler handler, final CUID cuid) throws UserProfileException {
		try {
			if (cuid == null || cuid.toString().length() == 0)
				throw new IllegalArgumentException("Card ID can not be null");

			IEntity dsUser = authenticate(handler);
			try {
				CardInformationBinding cib = new CardInformationBinding(dsUser);
				CardInformation ci = cib.getCardInformation(cuid);
				dsUser.getContext().applyUpdates();
				return ci;
			} finally {
				dsUser.getContext().close();
			}
		} catch (Exception e) {
			log.error(e, e);
			throw new UserProfileException(e);
		}
	}

	public IConfiguration getConfiguration() {
		if (this.config_ == null)
			this.config_ = new ProviderConfiguration(getID());
		return this.config_;
	}

	public String getID() {
		return this.id_;
	}

	public void setID(final String id) throws Exception {
		throw new UnsupportedOperationException();
	}

	public ICardUsageManager getCardUsageManager(final CallbackHandler handler) throws UserProfileException {
		IEntity dsUser = null;
		try {
			dsUser = authenticate(handler);
			try {
				return new CardUsageManager(dsUser);
			} catch (Exception e) {
				log.warn("Closing context of User " + dsUser.getEntityID() + ", in result of Exception:");
				log.warn(e);
				dsUser.getContext().close();
				log.warn("Context might be closed. User " + dsUser.getEntityID());
			}
			return null;
		} catch (Exception e) {
			log.error(e, e);
			throw new UserProfileException(e);
		}
	}

	public PolicyVersion getRPPolicyVersion(final CallbackHandler handler, final WebForm form) throws UserProfileException {
		try {
			IEntity dsUser = authenticate(handler);
			try {
				IAttribute attPV = dsUser.getAttribute(IdasCtxUris.UP_policyVersion);
				for (Iterator it = attPV.getValues(); it.hasNext();) {
					IComplexAttrValue valPV = (IComplexAttrValue) it.next();
					WebForm wfFound = new WebForm((URI) IdasUtils.getSimpleValueData((IComplexAttrValue) valPV, IdasCtxUris.CU_url),
							(String) IdasUtils.getSimpleValueData((IComplexAttrValue) valPV, IdasCtxUris.CU_formName), (String) IdasUtils
									.getSimpleValueData((IComplexAttrValue) valPV, IdasCtxUris.CU_formId), (String) IdasUtils.getSimpleValueData(
									(IComplexAttrValue) valPV, IdasCtxUris.CU_formAction));
					// URI urlFound = (URI) IdasUtils.getSimpleValueData(valPV,
					// IdasCtxUris.CU_url);
					if (wfFound.equals(form)) {
						PolicyVersion pv = new PolicyVersion();
						pv.setForm(wfFound);
						pv.setPolicy(new String(((ByteBuffer) IdasUtils.getSimpleValueData(valPV, IdasCtxUris.PV_policy)).array(), "UTF-8"));
						pv.setVersion((String) IdasUtils.getSimpleValueData(valPV, IdasCtxUris.PV_version));
						return pv;
					}
				}
			} finally {
				dsUser.getContext().close();
			}
			return null;
		} catch (Exception e) {
			log.error(e, e);
			throw new UserProfileException(e);
		}
	}

	public void setRPPolicyVersion(final CallbackHandler handler, final PolicyVersion policyVersion) throws UserProfileException {
		try {
			if (policyVersion == null)
				throw new IllegalArgumentException("Parameter \"policyVersion\" can not be null");
			if (policyVersion.getForm() == null)
				throw new IllegalArgumentException("Parameter \"policyVersion.webForm\" can not be null");
			if (policyVersion.getPolicy() == null)
				throw new IllegalArgumentException("Parameter \"policyVersion.policy\" can not be null");
			if (policyVersion.getVersion() == null)
				throw new IllegalArgumentException("Parameter \"policyVersion.version\" can not be null");

			IEntity dsUser = authenticate(handler);
			try {
				IAttribute attPV = dsUser.getAttribute(IdasCtxUris.UP_policyVersion);

				for (Iterator it = attPV.getValues(); it.hasNext();) {
					IComplexAttrValue valPV = (IComplexAttrValue) it.next();
					WebForm wfFound = new WebForm((URI) IdasUtils.getSimpleValueData((IComplexAttrValue) valPV, IdasCtxUris.CU_url),
							(String) IdasUtils.getSimpleValueData((IComplexAttrValue) valPV, IdasCtxUris.CU_formName), (String) IdasUtils
									.getSimpleValueData((IComplexAttrValue) valPV, IdasCtxUris.CU_formId), (String) IdasUtils.getSimpleValueData(
									(IComplexAttrValue) valPV, IdasCtxUris.CU_formAction));
					// URI urlFound = (URI) IdasUtils.getSimpleValueData(valPV,
					// IdasCtxUris.CU_url);
					if (wfFound != null && wfFound.equals(policyVersion.getForm())) {
						valPV.getAttribute(IdasCtxUris.PV_policy).addSimpleValue(
						// TODO need IdAS type that maps to SQL TEXT
								// type, this is just a workaround for big
								// strings
								ITypedValue.BASE64BINARY_TYPE_URI, ByteBuffer.wrap(policyVersion.getPolicy().getBytes("UTF-8")));
						valPV.getAttribute(IdasCtxUris.PV_version).addSimpleValue(ITypedValue.STRING_TYPE_URI, policyVersion.getVersion());

						dsUser.getContext().applyUpdates();
						return;
					}
				}

				IComplexAttrValue valPV = attPV.addComplexValue(IdasCtxUris.PV_PolicyVersion);
				valPV.getAttribute(IdasCtxUris.CU_url).addSimpleValue(ITypedValue.ANYURI_TYPE_URI, policyVersion.getForm().getUrl());
				valPV.getAttribute(IdasCtxUris.CU_formAction).addSimpleValue(ITypedValue.STRING_TYPE_URI, policyVersion.getForm().getFormAction());
				valPV.getAttribute(IdasCtxUris.CU_formId).addSimpleValue(ITypedValue.STRING_TYPE_URI, policyVersion.getForm().getFormId());
				valPV.getAttribute(IdasCtxUris.CU_formName).addSimpleValue(ITypedValue.STRING_TYPE_URI, policyVersion.getForm().getFormName());

				valPV.getAttribute(IdasCtxUris.PV_policy).addSimpleValue(
				// TODO need IdAS type that maps to SQL TEXT
						// type, this is workaround for big strings
						ITypedValue.BASE64BINARY_TYPE_URI, ByteBuffer.wrap(policyVersion.getPolicy().getBytes("UTF-8")));
				valPV.getAttribute(IdasCtxUris.PV_version).addSimpleValue(ITypedValue.STRING_TYPE_URI, policyVersion.getVersion());

				dsUser.getContext().applyUpdates();
			} finally {
				dsUser.getContext().close();
			}
		} catch (Exception e) {
			log.error(e, e);
			throw new UserProfileException(e);
		}
	}

	public void setNewPassword(final CallbackHandler handler, final String newPassword) throws UserProfileException {
		try {
			IEntity dsUser = authenticate(handler);

			try {
				if (newPassword == null)
					throw new IllegalArgumentException("Parameter \"newPassword\" can not be null!");

				dsUser.getAttribute(IdasCtxUris.UP_newAuthPasswordHash).addSimpleValue(ITypedValue.BASE64BINARY_TYPE_URI,
						UserCreds.getPasswordHash(newPassword));
				dsUser.getContext().applyUpdates();

				if (!isNoCache) {
					UserCacheKey key = new UserCacheKey(new UserCreds(handler).getUriName());
					CachedItem cached = (CachedItem) userProfileCache.get(key);
					if (cached == null) {
						cached = new CachedItem();
						cached.userProfile = initFromDS_UP(dsUser);
						cached.passwordHash = (ByteBuffer) IdasUtils.getSimpleValueData(dsUser, IdasCtxUris.UP_authPasswordHash);
					}
					cached.newPasswordHash = UserCreds.getPasswordHash(newPassword);
					userProfileCache.put(key, cached);
				}
			} finally {
				dsUser.getContext().close();
			}
		} catch (Exception e) {
			log.error(e, e);
			throw new UserProfileException(e);
		}
	}

	public void modifyPassword(String userId, String newPassword) throws UserProfileException {
		if (userId == null)
			throw new UserProfileException("Parameter \"userId\" is null.");
		String password = (newPassword != null) ? newPassword : "";
		String id = normalizeUserID(userId);
		try {
			IEntity dsUser = getEntity(id);
			try {
				dsUser.getAttribute(IdasCtxUris.UP_authPasswordHash).addSimpleValue(ITypedValue.BASE64BINARY_TYPE_URI,
						UserCreds.getPasswordHash(password));
				dsUser.getContext().applyUpdates();
				if (!isNoCache) {
					UserCacheKey key = new UserCacheKey(id);
					CachedItem cached = (CachedItem) userProfileCache.get(key);
					if (cached == null) {
						cached = new CachedItem();
						cached.userProfile = initFromDS_UP(dsUser);
					}
					cached.passwordHash = (ByteBuffer) IdasUtils.getSimpleValueData(dsUser, IdasCtxUris.UP_authPasswordHash);
					userProfileCache.put(key, cached);
				}
			} finally {
				dsUser.getContext().close();
			}
		} catch (Exception e) {
			log.error(e, e);
			throw new UserProfileException(e);
		}
	}

	public void setNewPasswordByPasswordIsapMd5(final String userId, final byte[] passwordIsapMd5, final String newPassword)
			throws UserProfileException {
		try {
			IEntity dsUser = authenticate(userId, passwordIsapMd5);

			try {
				if (newPassword == null)
					throw new IllegalArgumentException("Parameter \"newPassword\" can not be null!");

				dsUser.getAttribute(IdasCtxUris.UP_newAuthPasswordHash).addSimpleValue(ITypedValue.BASE64BINARY_TYPE_URI,
						UserCreds.getPasswordHash(newPassword));
				dsUser.getContext().applyUpdates();

				if (!isNoCache) {
					UserCacheKey key = new UserCacheKey("urn:" + userId.replace(' ', '_'));
					CachedItem cached = (CachedItem) userProfileCache.get(key);
					if (cached == null) {
						cached = new CachedItem();
						cached.userProfile = initFromDS_UP(dsUser);
						cached.passwordHash = (ByteBuffer) IdasUtils.getSimpleValueData(dsUser, IdasCtxUris.UP_authPasswordHash);
					}
					cached.newPasswordHash = UserCreds.getPasswordHash(newPassword);
					userProfileCache.put(key, cached);
				}
			} finally {
				dsUser.getContext().close();
			}
		} catch (Exception e) {
			log.error(e, e);
			throw new UserProfileException(e);
		}
	}

	public void deleteOldPassword(final CallbackHandler handler) throws UserProfileException {
		try {
			IEntity dsUser = authenticate(handler);

			try {
				ByteBuffer newAuthHash = (ByteBuffer) IdasUtils.getSimpleValueData(dsUser, IdasCtxUris.UP_newAuthPasswordHash);
				if (newAuthHash == null || !newAuthHash.hasArray())
					throw new WrongPasswordException("Can not delete old password! Set the new one first!");

				// change password, set new password instead of old one, from
				// now
				// authentication will be performed against this (changed)
				// password
				dsUser.getAttribute(IdasCtxUris.UP_newAuthPasswordHash).remove();
				dsUser.getAttribute(IdasCtxUris.UP_authPasswordHash).addSimpleValue(ITypedValue.BASE64BINARY_TYPE_URI, newAuthHash);
				dsUser.getContext().applyUpdates();

				if (!isNoCache) {
					UserCacheKey key = new UserCacheKey(new UserCreds(handler).getUriName());
					CachedItem cached = (CachedItem) userProfileCache.get(key);
					if (cached == null) {
						cached = new CachedItem();
						cached.userProfile = initFromDS_UP(dsUser);
					}
					cached.newPasswordHash = null;
					cached.passwordHash = newAuthHash;
					userProfileCache.put(key, cached);
				}
			} finally {
				dsUser.getContext().close();
			}
		} catch (UserProfileAuthenticationException e) {
			throw e;
		} catch (Exception e) {
			log.error(e, e);
			throw new UserProfileException(e);
		}
	}

	public void addSelectorAndSetPasswordByPasswordIsapMd5(final String userId, final String privateSelectorINumber, final byte[] passwordIsapMd5,
			final String newPassword) throws UserProfileException {
		try {
			IEntity dsUser = authenticate(userId, passwordIsapMd5);

			try {
				if (privateSelectorINumber == null)
					throw new IllegalArgumentException("Parameter \"privateSelectorINumber\" can not be null!");
				if (newPassword == null)
					throw new IllegalArgumentException("Parameter \"newPassword\" can not be null!");

				UserProfile up = initFromDS_UP(dsUser);

				List listPSINum = up.getPrivateSelectorINumbers();
				if (listPSINum == null)
					listPSINum = new ArrayList();
				listPSINum.add(privateSelectorINumber);
				up.setPrivateSelectorINumbers(listPSINum);
				// up.setOneTimePasswords(null);
				saveToDS_UP(up, dsUser);

				dsUser.getAttribute(IdasCtxUris.UP_newAuthPasswordHash).remove();
				dsUser.getAttribute(IdasCtxUris.UP_authPasswordHash).addSimpleValue(ITypedValue.BASE64BINARY_TYPE_URI,
						UserCreds.getPasswordHash(newPassword));

				dsUser.getContext().applyUpdates();

				if (!isNoCache) {
					UserCacheKey key = new UserCacheKey("urn:" + userId.replace(' ', '_'));
					CachedItem cached = (CachedItem) userProfileCache.get(key);
					if (cached == null) {
						cached = new CachedItem();
						cached.userProfile = up; // initFromDS_UP(dsUser);
					}
					cached.newPasswordHash = null;
					cached.passwordHash = UserCreds.getPasswordHash(newPassword);
					userProfileCache.put(key, cached);
				}
			} finally {
				dsUser.getContext().close();
			}
		} catch (Exception e) {
			log.error(e, e);
			throw new UserProfileException(e);
		}
	}

	public Captcha addCaptcha(Captcha captcha) throws UserProfileException {
		if (captcha == null)
			throw new UserProfileException("Parameter \"captcha\" is null.");
		if (captcha.getId() != null)
			throw new UserProfileException("Passed captcha has been already stored.");
		String cKey = captcha.getKey();
		if (cKey == null || cKey.trim().length() == 0)
			throw new UserProfileException("Passed captcha contains no key.");
		Date cDate = captcha.getDtCreation();
		if (cDate == null)
			throw new UserProfileException("Passed captcha contains no creation date.");
		IContext ctx = null;
		try {
			ctx = getContext();
			IEntity cEntity = ctx.addEntity(IdasCtxUris.UP_Captcha, null);
			cEntity.getAttribute(IdasCtxUris.UP_key).addSimpleValue(ITypedValue.STRING_TYPE_URI, cKey);
			cEntity.getAttribute(IdasCtxUris.UP_dtCreation).addSimpleValue(ITypedValue.DATETIME_TYPE_URI, cDate);
			ctx.applyUpdates();
			String newId = cEntity.getEntityID();
			captcha.setId(newId);
		} catch (IdASException e) {
			log.error(e, e);
			cancelUpdates(ctx);
			throw new UserProfileException(e);
		} finally {
			closeContext(ctx);
		}
		return captcha;
	}

	private void closeContext(IContext ctx) {
		if (ctx != null) {
			try {
				ctx.close();
			} catch (IdASException e) {
				log.error(e, e);
			}
		}
	}

	private void cancelUpdates(IContext ctx) throws UserProfileException {
		try {
			if (ctx != null)
				ctx.cancelUpdates();
		} catch (Exception e) {
			log.error(e, e);
			throw new UserProfileException(e);
		}
	}

	public void deleteCaptcha(Captcha captcha) throws UserProfileException {
		if (captcha == null)
			throw new UserProfileException("Parameter \"captcha\" is null.");
		String id = captcha.getId();
		if (id != null) {
			IContext ctx = null;
			try {
				ctx = getContext();
				try {
					IEntity cEntity = ctx.getEntity(id);
					URI type = cEntity.getModel().getType();
					if (IdasCtxUris.UP_Captcha.equals(type))
						cEntity.remove();
					else
						throw new NoSuchEntityException("Entity with required ID has unexpected type " + type.toString());
				} catch (org.eclipse.higgins.idas.api.NoSuchEntityException nee) {
					log.error("Could not find IdAS entity by passed ID.", nee);
					throw new NoSuchEntityException("Could not find IdAS entity by passed ID.");
				}
				ctx.applyUpdates();
			} catch (Exception e) {
				log.error(e, e);
				cancelUpdates(ctx);
				throw new UserProfileException(e);
			} finally {
				closeContext(ctx);
			}
		} else
			throw new NoSuchEntityException("Passed captcha has no id.");
	}

	public Captcha getCaptcha(String captchaID) throws UserProfileException {
		if (captchaID == null)
			throw new UserProfileException("Parameter \"captchaID\" is null.");
		IContext ctx = null;
		try {
			ctx = getContext();
			IEntity cEntity = ctx.getEntity(captchaID);
			String id = cEntity.getEntityID();
			String key = (String) IdasUtils.getSimpleValueData(cEntity, IdasCtxUris.UP_key);
			Date dtCreation = (Date) IdasUtils.getSimpleValueData(cEntity, IdasCtxUris.UP_dtCreation);
			Captcha captcha = new Captcha();
			captcha.setDtCreation(dtCreation);
			captcha.setId(id);
			captcha.setKey(key);
			return captcha;
		} catch (org.eclipse.higgins.idas.api.NoSuchEntityException nee) {
			log.error("Could not find IdAS entity by passed ID.", nee);
			throw new NoSuchEntityException("Could not find IdAS entity by passed ID.");
		} catch (Exception e) {
			log.error(e, e);
			throw new UserProfileException(e);
		} finally {
			closeContext(ctx);
		}
	}

	public Captcha modifyCaptcha(Captcha captcha) throws UserProfileException {
		throw new UserProfileException("NOt implemented.");
	}

}
