/**
 * Copyright (c) 2007-2008 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *		Yuriy Pilipenko - API and implementation
 *
 */
package org.eclipse.higgins.userprofile.idas.util;

import java.net.URI;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IAttributeValue;
import org.eclipse.higgins.idas.api.IComplexAttrValue;
import org.eclipse.higgins.idas.api.IHasAttributes;
import org.eclipse.higgins.idas.api.ISimpleAttrValue;
import org.eclipse.higgins.idas.api.ISingleValuedAttribute;
import org.eclipse.higgins.idas.api.ITypedValue;
import org.eclipse.higgins.idas.api.IdASException;

/**
 * Class with helper routines for IdAS.
 *
 * @author Pilipenko
 *
 */
public final class IdasUtils {

	private IdasUtils() {
	}

	private static Object getSVAttrValue(final ISingleValuedAttribute attr) throws IdASException {
		if (attr != null) {
			IAttributeValue v = attr.getValue();
			if (v != null)
				if (v.isSimple()) {
					ISimpleAttrValue sv = (ISimpleAttrValue) v;
					return sv.getData();
				} else
					throw new IdASException("Attribute "
							+ (attr.getAttrID() != null ? attr.getAttrID().toString() : "")
							+ " contains non simple value");
		}
		return null;
	}

	/**
	 * @param ha
	 *            IHasAttributes instance
	 * @param attribute
	 *            URI to get a value from
	 * @return Object
	 * @throws IdASException
	 *             if some errors occurs during getting data
	 */
	public static Object getSimpleValueData(final IHasAttributes ha, final URI attribute) throws IdASException {
		ISingleValuedAttribute attr = ha.getSingleValuedAttribute(attribute);
		return getSVAttrValue(attr);
	}

	/**
	 * @param ha
	 *            IHasAttributes instance
	 * @param attribute
	 *            URI to get a value from
	 * @return ByteBuffer
	 * @throws IdASException
	 *             if some errors occurs during getting data
	 */
	public static ByteBuffer getSimpleValueDataAsByteBuffer(final IHasAttributes ha, final URI attribute)
			throws IdASException {
		return (ByteBuffer) getSimpleValueData(ha, attribute);
	}

	/**
	 * @param ha IHasAttributes instance
	 * @param attribure URI to get a value from
	 * @return byte[]
	 * @throws IdASException if some errors occurs during getting data
	 */
	public static byte[] getSimpleValueDataAsByteArray(final IHasAttributes ha, final URI attribure)
			throws IdASException {
		ByteBuffer bb = IdasUtils.getSimpleValueDataAsByteBuffer(ha, attribure);
		return (bb != null && bb.hasArray() ? bb.array() : null);
	}

	/**
	 * @param att IAttribute to which store values collection
	 * @param collection values to be stored
	 * @throws IdASException if some errors occurs during saving data
	 */
	public static void saveSimpleValueCollection(final IAttribute att, final Collection collection)
			throws IdASException {
		att.remove();
		if (collection != null && collection.size() > 0)
			for (Iterator iterator = collection.iterator(); iterator.hasNext();)
				//TODO change to store different data types
				att.addSimpleValue(ITypedValue.STRING_TYPE_URI, iterator.next());
	}

	/**
	 * @param att IAttribute from which get the values collection
	 * @return List
	 * @throws IdASException if some errors occurs during getting data
	 */
	public static List loadSimpleValueList(final IAttribute att) throws IdASException {
		if (att != null) {
			Iterator it = att.getValues();
			if (it.hasNext()) {
				List lst = new ArrayList();
				while (it.hasNext())
					lst.add(((ISimpleAttrValue) it.next()).getData());
				return lst;
			}
		}
		return null;
	}

	/**
	 * @param att IAttribute from which get the values collection
	 * @return Set
	 * @throws IdASException if some errors occurs during getting data
	 */
	public static Set loadSimpleValueSet(final IAttribute att) throws IdASException {
		if (att != null) {
			Iterator it = att.getValues();
			if (it.hasNext()) {
				Set set = new HashSet();
				while (it.hasNext())
					set.add(((ISimpleAttrValue) it.next()).getData());
				return set;
			}
		}
		return null;
	}

	/**
	 * @param complexAttr attribute with complex values to which to store data
	 * @param keyAttr simple attribute in the complexAttr containing key for map 
	 * @param valueAttr simple attribute in the complexAttr containing value for map
	 * @param map containing the data to store
	 * @throws IdASException IdASException if any idas error occurs
	 */
	public static void saveComplexValueMap(final IAttribute complexAttr, final URI keyAttr, final URI valueAttr,
			final Map map) throws IdASException {
		complexAttr.remove();
		if (map != null)
			for (Iterator it = map.entrySet().iterator(); it.hasNext();) {
				Map.Entry e = (Map.Entry) it.next();
				IComplexAttrValue val = complexAttr.addComplexValue(null);
				val.getAttribute(keyAttr).addSimpleValue(null, e.getKey());
				val.getAttribute(valueAttr).addSimpleValue(null, e.getValue());
			}
	}

	/**
	 * @param complexAttr attribute with complex values from which to load data
	 * @param keyAttr simple attribute in the complexAttr containing key for map 
	 * @param valueAttr simple attribute in the complexAttr containing value for map
	 * @return Map with loaded data
	 * @throws IdASException if any idas error occurs
	 */
	public static Map loadComplexValueMap(final IAttribute complexAttr, final URI keyAttr, final URI valueAttr)
			throws IdASException {
		Iterator vals = complexAttr.getValues();
		if (vals.hasNext()) {
			Map map = new HashMap();
			while (vals.hasNext()) {
				IAttributeValue val = (IAttributeValue) vals.next();
				if (!val.isSimple())
					map.put(
							getSimpleValueData(val, keyAttr),
							getSimpleValueData(val, valueAttr));
				else
					throw new IdASException("Complex value expected, but found " + complexAttr.getModel().getType());
			}
			return map;
		}
		return null;
	}

}
