/** 
 * Copyright (c) 2007 Parity Communications, Inc.  
 * All rights reserved. This program and the accompanying materials  
 * are made available under the terms of the Eclipse Public License v1.0  
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 *  
 * Contributors: 
 *		Yuriy Pilipenko - API and implementation 
 * 
 */
package org.eclipse.higgins.userprofile;

import java.util.List;
import javax.security.auth.callback.CallbackHandler;
import org.eclipse.higgins.icard.CUID;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.registry.IServiceProvider;
import org.eclipse.higgins.userprofile.entity.Captcha;
import org.eclipse.higgins.userprofile.entity.CardCredential;
import org.eclipse.higgins.userprofile.entity.CardInformation;
import org.eclipse.higgins.userprofile.entity.Category;
import org.eclipse.higgins.userprofile.entity.PolicyVersion;
import org.eclipse.higgins.userprofile.entity.UserProfile;
import org.eclipse.higgins.userprofile.entity.WebForm;


/**
 * @author Yuriy Pilipenko
 *
 */
public interface IUserProfileService extends IServiceProvider {

	/**
	 * @param userIdentifier
	 * @param password
	 * @return  
	 * @throws CardException 
	 */
	public UserProfile getUserProfile(CallbackHandler handler) throws UserProfileException;

	/**
	 * @param userID
	 * @return
	 * @throws UserProfileException
	 */
	public UserProfile getUserProfile(String userID) throws UserProfileException;

	/**
	 * @param userProfile
	 * @return  
	 * @throws CardException 
	 */
	public UserProfile addUserProfile(CallbackHandler handler, UserProfile userProfile) throws UserProfileException;

	/**
	 * @param userProfile
	 * @return  
	 */
	public UserProfile modifyUserProfile(CallbackHandler handler, UserProfile userProfile) throws UserProfileException;

	/**
	 * @param userProfile
	 * @return
	 * @throws UserProfileException
	 */
	public UserProfile modifyUserProfile(UserProfile userProfile) throws UserProfileException;

	/**
	 * @param userIdentifier
	 * @param password
	 * @return  
	 */
	public void deleteUserProfile(CallbackHandler handler) throws UserProfileException;

	/**
	 * @param userIdentifier
	 * @return
	 */
	public String resetPassword(CallbackHandler handler) throws UserProfileException;

	/**
	 * @param userId
	 * @param newPassword
	 * @throws UserProfileException
	 */
	public void modifyPassword(String userId, String newPassword) throws UserProfileException;

	/**
	 * @param userIdentifier
	 * @return true if the specified user profile already exists
	 */
	public boolean userIdentifierExists(String userIdentifier) throws UserProfileException;
	
	/**
	 * Resolves user private Selector I-Number to user Identifier
	 * @param privateSelectorINumber
	 * @return
	 * @throws UserProfileException
	 */
	public String resolveUserIdentifier(String privateSelectorINumber) throws UserProfileException;
	
	/**
	 * @param handler - authentication handler
	 * @return List object with all user categories.  
	 */
	public List getCategories(CallbackHandler handler) throws UserProfileException;
	
	/**
	 * @param handler - authentication handler
	 * @param category
	 * @return new created category instance.
	 */
	public Category addCategory(CallbackHandler handler, Category category) throws UserProfileException;

	/**
	 * @param handler - authentication handler
	 * @param category
	 * @return updated category instance .
	 */
	public Category modifyCategory(CallbackHandler handler, Category category) throws UserProfileException;

	/**
	 * @param categoryId
	 * @param userIdentifier
	 * @param password
	 * @return .
	 */
	public void deleteCategory(CallbackHandler handler, String categoryId) throws UserProfileException;
	
	/**
	 * Sets credential information for the given card.
	 * @param handler
	 * @param cuid
	 * @param cardCredential
	 * @return CardInformation object with all related card information, or null if card info not found
	 */
	public CardInformation setCardCredential(CallbackHandler handler, CUID cuid, CardCredential cardCredential) throws UserProfileException;
	
	/**
	 * Clears credential information for the given card.
	 * @param handler
	 * @param cardCredentialInfo  
	 * @return CardInformation object with all related card information, or null if card info not found 
	 * @throws UserProfileException
	 */ 
	public CardInformation clearCardCredential(CallbackHandler handler, CUID cuid) throws UserProfileException;
	
/*	*//**
	 * Returns information about card remebered for this site.
	 * @param handler
	 * @param URI
	 * @return CardInformation object with all related card information, or null if card info not found
	 * @throws UserProfileException
	 *//*
	public CardInformation getCardInformation(CallbackHandler handler, URI URI) throws UserProfileException;
*/	
	/**
	 * Returns the whole information about the given card 
	 * @param handler
	 * @param cuid
	 * @return
	 */
	public CardInformation getCardInformation(CallbackHandler handler, CUID cuid) throws UserProfileException;
	
	/**
	 * Returns class that manages all Card usage and URI relations information
	 * @param handler
	 * @return instance of ICardUsageRelationManager
	 * @throws UserProfileException
	 */
	public ICardUsageManager getCardUsageManager(CallbackHandler handler) throws UserProfileException;
	
	
	public PolicyVersion getRPPolicyVersion(CallbackHandler handler, WebForm form) throws UserProfileException;
		
	public void setRPPolicyVersion(CallbackHandler handler, PolicyVersion policyVersion) throws UserProfileException;
	
	/**
	 * Set authentication passwords, new and old ones, to support ISAP change password sequence.
	 * Authentication is made against password hash stored in the DB. Be careful - all subsequent 
	 * operations will be authenticated against password hash given in the 'AuthData.authPassword' 
	 * field if this operation succeeded.
	 * @param handler Callback handler
	 * @param authData
	 * @throws UserProfileException
	 */
	//public void setAuthData(CallbackHandler handler, AuthData authData) throws UserProfileException;
	
	/**
	 * Get authentication passwords, new and old ones, to support ISAP change password sequence.
	 * Authentication is made against password hash stored in the DB which is returned 
	 * in 'AuthData.authPassword' field.
	 * @param handler
	 * @return
	 * @throws UserProfileException
	 */
	//public AuthData getAuthData(CallbackHandler handler) throws UserProfileException;
	
	/**
	 * Sets the new password allowing authentication both by new and old passwords
	 * until the successful deleteOldPassword() call. This implements change password logic 
	 * according to ISAP sequences CP1 and CP2.
	 * @param handler - Callback handler containing current and valid "old password" or the new one.
	 * @param newPassword - password, on which the authentication will try. If not succeeded, 
	 * will conduct on the old one. 
	 * @throws UserProfileException
	 */
	public void setNewPassword(CallbackHandler handler, String newPassword) throws UserProfileException;
	
	/**
	 * Sets the new password allowing authentication both by new and old
	 * passwords until the successful deleteOldPassword() call. This implements
	 * change password logic according to ISAP sequences CP1 and CP2. The
	 * Private Broker MUST authenticate this message came from an authorized
	 * Selector by verifying that the $password$isap$md5 value matches the hash
	 * of the same value registered with the Private Broker in ISAP Sequence P1.
	 * This value is the Installation Key MD5 defined in the Credentials section
	 * of ISAP Common. Note that the Private Broker should store a HASH of this
	 * value, even though the value is already a hash of the Installation Key
	 * 
	 * @param passwordIsapMd5 - hash of Installation Key MD5 by which authentication 
	 * 			occurs here 
	 * @param newPassword - password, on which the authentication will try. If not
	 *            succeeded, will conduct on the old one.
	 * @throws UserProfileException
	 */
	public void setNewPasswordByPasswordIsapMd5(String userId, byte[] passwordIsapMd5, String newPassword) throws UserProfileException;
	
	/**
	 * Deletes the old password and change it to the new one. After this successful call change 
	 * password process assumed completed and all subsequent authentication will conduct
	 * on the changed (new) password. This implements change password logic 
	 * according to ISAP sequences CP1 and CP2.
	 * @param handler - Callback handler containing current and valid "old password" or the new one.
	 * @throws UserProfileException
	 */
	public void deleteOldPassword(CallbackHandler handler) throws UserProfileException;
	
	/**
	 * Add the new Selector given by the privateSelectorINumber and set the new password
	 * without remembering it. Note that all subsuquent operations will be authenticated
	 * by this new password.
	 * 
	 * The Private Broker MUST authenticate this message came from an authorized Selector by verifying 
	 * that the Installation Key MD5 matches the Installation Key MD5 registered with the Private Broker 
	 * in ISAP Sequence P1.<br>
	 * The Private Broker MUST:<br> 
	 * 1. Register the new Selector as defined in ISAP Sequence P2.<br>
	 * 2. Change the Private User Password and defined in ISAP Sequence CP2 except that in this sequence 
	 * 		the old Private User Password is NOT saved because a new Selector is being installed. 
	 * 
	 * @param userId - private-user-i-number at the Private Broker
	 * @param privateSelectorINumber
	 * @param passwordIsapMd5
	 * @param newPassword
	 * @throws UserProfileException
	 */
	public void addSelectorAndSetPasswordByPasswordIsapMd5(String userId, String privateSelectorINumber,
			byte[] passwordIsapMd5, String newPassword) throws UserProfileException;
	
	/**
	 * @param captchaID
	 * @return
	 * @throws UserProfileException
	 */
	public Captcha getCaptcha(String captchaID) throws UserProfileException;

	/**
	 * @param captcha
	 * @return
	 * @throws UserProfileException
	 */
	public Captcha addCaptcha(Captcha captcha) throws UserProfileException;

	/**
	 * @param captcha
	 * @return
	 * @throws UserProfileException
	 */
	public void deleteCaptcha(Captcha captcha) throws UserProfileException;

	/**
	 * @param captcha
	 * @return
	 * @throws UserProfileException
	 */
	public Captcha modifyCaptcha(Captcha captcha) throws UserProfileException;
}