/**
 * Copyright (c) 2007-2008 Novell, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *  	Duane Buss
 *  	Tom Doman
 *		Jim Sermersheim
 */

package org.eclipse.higgins.util.idas.cp;

import java.net.URI;
import java.util.Iterator;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IAttributeValue;
import org.eclipse.higgins.idas.api.IComplexAttrValue;
import org.eclipse.higgins.idas.api.IExtension;
import org.eclipse.higgins.idas.api.IHasAttributes;
import org.eclipse.higgins.idas.api.ISimpleAttrValue;
import org.eclipse.higgins.idas.api.ISingleValuedAttribute;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.InvalidTypeException;
import org.eclipse.higgins.idas.api.NotImplementedException;
import org.eclipse.higgins.idas.api.NotSingleValuedAttributeException;
import org.eclipse.higgins.idas.api.UnhandledExtensionException;
import org.eclipse.higgins.idas.api.model.IAttributeModel;
import org.eclipse.higgins.idas.spi.BasicAttribute;

/**
 * Helper routines for <a href="http://www.eclipse.org/higgins/">Higgins</a>
 * Context providers that implement some type of PDP. This implementation allows
 * for PDPs at critical points for converting types and values, both for the core
 * properties and for the metadata associated with the attribute.
 * 
 * <ul>
 * <li> {@link #getValues()}</li>
 * </ul>
 * 
 * @author dbuss@novell.com
 */

public class BasicAttributePDP  implements IAttribute
{
	private IAttributePDPs _attributePDP;
	private IAttributePDPs _attributeMetaAttrPDP;
	private IAttributePDPs _attributeValueMetaAttrPDP;
	private IAttribute _attr;
	private boolean _isProviderAttr;
	private boolean _mapForConsumer;
	private URI _attrID;
	private Log _log;

	/**
	 * Create a new PDP based attribute which calls out to do conversions.
	 * 
	 * @param attributePDP
	 *           The PDP for the attribute conversions.
	 * @param attributeMetaAttrPDP
	 *           The PDP for Attribute meta-attribute conversions.
	 * @param attributeValueMetaAttrPDP
	 *           The PDP for Attribute Value meta-attribute conversions.
	 * @param attribute
	 *           The object we will wrap, it is the unmapped value, this could 
	 *           be provider or consumer value
	 * @param  isProviderAttr
	 * 				indicates if the value passed is provider or consumer.  Pass true 
	 * if the value is provider based.
	 * @throws IdASException 
	 */
	public BasicAttributePDP(
		IAttributePDPs attributePDP,
		IAttributePDPs attributeMetaAttrPDP,
		IAttributePDPs attributeValueMetaAttrPDP,
		IAttribute attribute,
		boolean isProviderAttr) throws IdASException
	{
		_log = LogFactory.getLog(this.getClass());
		_log.debug(attribute.getAttrID().toASCIIString());
		_attributePDP = attributePDP;
		_attributeMetaAttrPDP = attributeMetaAttrPDP;
		_attributeValueMetaAttrPDP = attributeValueMetaAttrPDP;
		_attr = attribute;
		_isProviderAttr = isProviderAttr;
		_mapForConsumer = isProviderAttr;
	}

	/**
	 * TODO: currently returns the unmapped model
    * TODO actually implement this function as more than a pass through function 
	 * @see IAttribute#getModel()
	 */
	public  IAttributeModel getModel() throws IdASException
	{
		return _attr.getModel();
	}

	public IAttributeValue addValue(
		URI attrID) throws IdASException, InvalidTypeException
	{
		URI newAttrID = attrID;
		if ( _attributePDP != null)
		{
			if ( _isProviderAttr == true && _mapForConsumer == true)
				newAttrID = _attributePDP.consumerIDToProvider(attrID);
			else if  ( _isProviderAttr == false && _mapForConsumer == false)
				newAttrID = _attributePDP.providerIDToConsumer(attrID);
		}
		
		IAttributeValue value = _attr.addValue(newAttrID);
		
		return value;
	}

	/**
	 * TODO: this doesn't yet perform any policy callouts
	 */
	public IAttributeValue addValue(
		IAttributeValue copyFrom) throws IdASException
	{
		return _attr.addValue(copyFrom);
	}

	public URI getAttrID() throws IdASException
	{
		if ( _attrID == null)
		{
			if (_attributePDP != null)
			{
				if (_isProviderAttr == true && _mapForConsumer == true)
				{
					_attrID = _attributePDP.providerIDToConsumer(_attr.getAttrID());
				}
				else if (_isProviderAttr == false && _mapForConsumer == false)
				{
					_attrID = _attributePDP.consumerIDToProvider(_attr.getAttrID());
				}
			}
			if (_attrID == null)
				_attrID = _attr.getAttrID();
		}
		return _attrID;
	}

	/**
	 * TODO actually implement this function as more than a pass through function
	 * @see org.eclipse.higgins.idas.api.IAttribute#isSingleValued()
	 */
	public boolean isSingleValued() throws IdASException
	{
		return _attr.isSingleValued();
	}

	public IComplexAttrValue addComplexValue(
		URI arg0) throws IdASException, InvalidTypeException
	{
		// TODO Auto-generated method stub
		return null;
	}

	public ISimpleAttrValue addSimpleValue(
		URI dataType,
		Object copyFrom) throws IdASException, InvalidTypeException
	{
		return _attr.addSimpleValue(dataType, copyFrom);
	}

	
	public Iterator getValues() throws IdASException
	{
		return new AttributeValuesPDPIter(
			(_isProviderAttr == true && _mapForConsumer == true) ? _attr.getAttrID() : getAttrID(),
			(_isProviderAttr == false && _mapForConsumer == false) ?  getAttrID() : _attr.getAttrID(),	
			_attributePDP,  _attributeValueMetaAttrPDP, 
			_attr.getValues(), _isProviderAttr);
	}

	public Iterator getValues(IExtension[] extensions) throws IdASException 
	{
		return new AttributeValuesPDPIter(
			(_isProviderAttr == true && _mapForConsumer == true) ? _attr.getAttrID() : getAttrID(),
			(_isProviderAttr == false && _mapForConsumer == false) ?  getAttrID() : _attr.getAttrID(),	
			_attributePDP,  _attributeValueMetaAttrPDP, 
			_attr.getValues(extensions), _isProviderAttr);
	}
	
	/**
	 * TODO actually implement this function as more than a pass through function
	 * @see org.eclipse.higgins.idas.api.IAttribute#remove()
	 */
	public void remove() throws IdASException
	{
		_attr.remove();
	}

	/**
	 * Quick to implement and possibly low performance impl
	 */
	public boolean equals(IAttribute attr) throws IdASException {
		BasicAttribute thisAttr = new BasicAttribute(this, null, null);
		return thisAttr.equals(attr);
	}

	
	/**
	 * Creates a meta-attribute for this object.
	 * 
	 * This Context Provider allows meta-attribute types and values to be mapped from
	 * stored (provider) values to what is presented by consumers of this Context
	 * Provider by inserting JavaScript into the configuration file.
	 * 
	 * The PDP naming is controlled by the Context Provider.
	 * <p>
	 * 
	 * @see org.eclipse.higgins.idas.api.IHasAttrbutes#addAttribute(java.net.URI)
	 */
	public IAttribute addAttribute(URI attrID) throws IdASException,
			InvalidTypeException {
		if (_attributeMetaAttrPDP != null)
		{
			IAttribute newMetadata = null;
			if ( _isProviderAttr == true && _mapForConsumer == true)
				newMetadata = _attr.addAttribute(_attributeMetaAttrPDP.consumerIDToProvider(attrID));
			else if  ( _isProviderAttr == false && _mapForConsumer == false)
				newMetadata = _attr.addAttribute(_attributeMetaAttrPDP.providerIDToConsumer(attrID));
			return new BasicAttributePDP(_attributeMetaAttrPDP, _attributeMetaAttrPDP, _attributeValueMetaAttrPDP, newMetadata, false);
		}
		return _attr.addAttribute(attrID);
	}

	/**
	 * Creates an meta-attribute for this object.
	 * 
	 * This Context Provider allows meta-attribute types and values to be mapped from
	 * stored (provider) values to what is presented by consumers of this Context
	 * Provider by inserting JavaScript into the configuration file.
	 * 
	 * The PDP naming is controlled by the Context Provider.
	 * <p>
	 * TODO this function needs to check the incoming copyFrom and see if it is from this
	 * cp or not, then check the mapping, then copy it while mapping
	 * 
	 * @see org.eclipse.higgins.idas.api.IHasAttributes#addAttribute(org.eclipse.higgins.idas.api.IAttribute)
	 */
	public IAttribute addAttribute(IAttribute copyFrom) throws IdASException {
		// screwy logic but it should work
		if (_attributeMetaAttrPDP != null )
			copyFrom = new BasicAttributePDP(_attributeMetaAttrPDP, _attributeMetaAttrPDP, _attributeValueMetaAttrPDP, copyFrom, false);
		return _attr.addAttribute(copyFrom);
	}

	/**
	 * Quick to implement and possibly low performance impl
	 */
	public boolean equals(IHasAttributes attributes) throws IdASException {
		BasicAttribute thisAttr = new BasicAttribute(this, null, null);
		return thisAttr.equals(attributes);
	}

	/*
	 * TODO optimize this routine, shouldn't have to waste time with every possible
	 * value, but this is currently the quick and the dirty way to do it.
	 * @see org.eclipse.higgins.idas.IHasAttributes#getAttribute(java.net.URI)
	 */
	public IAttribute getAttribute(URI attrID) throws IdASException {
		if (_attributeMetaAttrPDP != null)
		{
			Iterator itr = getAttributes();
			while (itr !=null && itr.hasNext())
			{
				BasicAttributePDP temp = new BasicAttributePDP(_attributeMetaAttrPDP, _attributeMetaAttrPDP, _attributeValueMetaAttrPDP, (IAttribute)itr.next(), 
					_isProviderAttr);
				if (temp.getAttrID() == attrID)
					return temp;
			}
			return null;
		}
		return _attr.getAttribute(attrID);
	}

	public Iterator getAttributes() throws IdASException {
		if (_attributeMetaAttrPDP != null)
		{
			return new BasicAttributePDPIter(_attributeMetaAttrPDP, _attributeMetaAttrPDP, _attributeValueMetaAttrPDP, _attr.getAttributes(), _isProviderAttr);
		}
		return _attr.getAttributes();
	}

	public ISingleValuedAttribute getSingleValuedAttribute(URI attrID)
			throws IdASException, NotSingleValuedAttributeException {
		if (_attributeMetaAttrPDP != null)
		{
			Iterator itr = getAttributes();
			while (itr !=null && itr.hasNext())
			{
				BasicAttributePDP temp = new  BasicAttributePDP(_attributeMetaAttrPDP, _attributeMetaAttrPDP, _attributeValueMetaAttrPDP, (IAttribute)itr.next(), 
					_isProviderAttr);
				if (temp.getAttrID() == attrID)
					return temp.getSingleValuedAttribute(attrID);
			}
			return null;
		}
		return _attr.getSingleValuedAttribute(attrID);
	}

	public void removeAttribute(URI attrID) throws IdASException {
		if (_attributeMetaAttrPDP != null)
		{
			Iterator itr = getAttributes();
			while (itr !=null && itr.hasNext())
			{
				BasicAttributePDP temp = new BasicAttributePDP(_attributeMetaAttrPDP, _attributeMetaAttrPDP, _attributeValueMetaAttrPDP, (IAttribute)itr.next(), 
					_isProviderAttr);
				if (temp.getAttrID() == attrID)
					temp.remove();
			}
			return;
		}
		_attr.removeAttribute(attrID);
	}

	public void removeAttributeValue(URI attrID, Object value)
			throws IdASException {
		if (_attributeMetaAttrPDP != null )
			throw new NotImplementedException();
		_attr.removeAttributeValue(attrID, value);
	}

	public void removeAttributeValue(IAttribute attr) throws IdASException {
		if (_attributeMetaAttrPDP != null )
			throw new NotImplementedException();
		_attr.removeAttributeValue(attr);
	}

}
