/**
 * Copyright (c) 2007 Novell, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *  	Duane Buss
 *  	Tom Doman
 */

package org.eclipse.higgins.util.idas.cp;

import java.net.URI;
import java.util.Iterator;

import org.eclipse.higgins.idas.api.IAttributeValue;
import org.eclipse.higgins.idas.api.IdASException;

/**
 * Part of a <a href="http://www.eclipse.org/higgins/">Higgins</a> Context
 * provider. This module defines the PDP interfaces that are common to most
 * attribute, metadata and value  based Higgins objects.
 *  Assists in mapping between consumer and provider elements
 * 
 * @author dbuss@novell.com
 */

public interface IAttributePDPs
{
	/**
	 * PDP interface for converting from passed by consumer (presented) 
	 * attributes and their to values to
	 * what is correct for storage or relaying on to another Context Provider. 
	 * 
	 * @param consumerID the consumer or unmapped ID of the attribute
	 * @return The actual or provider ID of the attribute value, returning null
	 *         indicates that this property should not be returned to the
	 *         consumer.
	 * @throws IdASException
	 */
	public URI consumerIDToProvider(
		URI consumerID) throws IdASException;

	/**
	 * PDP interface for converting from passed by consumer (presented) values to
	 * what is correct for storage or relaying on to another Context Provider.
	 * 
	 * This may return multiple values for cases where one maps to many
	 * 
	 * @param elem
	 *           The consumer or unmapped ID of the attribute
	 * @return The actual or provider type of the attribute, returning null
	 *         indicates that this property should not be returned to the
	 *         consumer.
	 * 
	 * @throws IdASException
	 */
	public Iterator consumerIDToProviders(
		URI elem) throws IdASException;

	/**
	 * Helper PDP for converting from stored or child Context Provider to
	 * consumer presented value.
	 * 
	 * @param providerID
	 *           The actual or provider type of the attribute value
	 * @return The consumer or mapped type of the attribute value, returning null
	 *         indicates that this property should not be returned to the
	 *         consumer.
	 * @throws IdASException
	 */
	
	public URI providerIDToConsumer(
		URI providerID) throws IdASException;
	
	/**
	 * PDP interface for converting from passed by consumer (presented) values to
	 * what is correct for storage or relaying on to another Context Provider.
	 * 
	 * @param consumerID the consumer or unmapped ID of the attribute
	 * @param consumerType
	 *           The consumer or unmapped type of the attribute
	 * @return The actual or provider type of the property value, returning null
	 *         indicates that this property should not be returned to the
	 *         consumer.
	 * @throws IdASException
	 */
	public URI consumerTypeToProvider(
		URI consumerID,
		URI consumerType) throws IdASException;

	/**
	 * Helper PDP for converting from stored or child Context Provider to
	 * consumer presented value.
	 * 
	 * @param providerID the provider id of the attribute value
	 * @param providerType
	 *           The actual or provider type of the attribute value
	 * @return The consumer or mapped type of the metadata, returning null
	 *         indicates that this property should not be returned to the
	 *         consumer.
	 * @throws IdASException
	 */
	public URI providerTypeToConsumer(
		URI providerID,
		URI providerType) throws IdASException;
	
	/**
	 * Helper PDP for converting from passed (consumer presented/provided} value
	 * to what is correct for storage or relaying on to another Context Provider.
	 * 
	 * 
	 * @param consumerType
	 *           The consumer or mapped type of the attribute value
	 * @param providerType
	 *           The actual or provider type of the attribute value
	 * @param consumerValue
	 *           The data comprising the consumer view of the value
	 * @return If the PDP creates a new value that will be returned, if no
	 *         mapping called for then the original value should be returned. A
	 *         null means this value is not to be presented to the next
	 *         interface.
	 * @throws IdASException
	 */
	public IAttributeValue consumerValueToProvider(
		URI consumerID,
		URI consumerType,
		URI providerID,
		URI providerType,
		IAttributeValue consumerValue) throws IdASException;

	/**
	 * Helper PDP for converting from stored or child Context Provider to
	 * consumer presented value.
	 * 
	 * @param providerType
	 *           The actual or provider type of the attribute value
	 * @param consumerType
	 *           The consumer or mapped type of the attribute value
	 * @param providerValue
	 *           The data stored or child Context Provider view of the value
	 * @return If the PDP creates a new value that will be returned, if no
	 *         mapping called for then the original value should be returned. A
	 *         null means this value is not to be presented to the next
	 *         interface.
	 * @throws IdASException
	 */
	public IAttributeValue providerValueToConsumer(
		URI providerID,
		URI providerType,
		URI consumerID,
		URI consumerType,
		IAttributeValue providerValue) throws IdASException;
}
