/**
 * Copyright (c) 2007-2008 Novell, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *  	Duane Buss
 *  	Tom Doman
 *		Jim Sermersheim
 */

package org.eclipse.higgins.util.idas.cp;

import java.net.URI;
import java.util.Iterator;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IAttributeValue;
import org.eclipse.higgins.idas.api.IHasAttributes;
import org.eclipse.higgins.idas.api.ISimpleAttrValue;
import org.eclipse.higgins.idas.api.ISingleValuedAttribute;
import org.eclipse.higgins.idas.api.ITypedValue;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.InvalidTypeException;
import org.eclipse.higgins.idas.api.NotImplementedException;
import org.eclipse.higgins.idas.api.NotSingleValuedAttributeException;
import org.eclipse.higgins.idas.api.model.IAttributeSimpleValueModel;
import org.eclipse.higgins.idas.spi.BasicAttributeSet;
import org.eclipse.higgins.idas.spi.BasicSimpleValue;

/**
 * @author dbuss
 *
 */
public class SimpleValuePDP  implements ISimpleAttrValue
{
	private URI _providerAttrID;
	private URI _consumerAttrID;
	private URI _dataType;
	private ISimpleAttrValue _simpleValue;
	private IAttributePDPs _attributePDP;
	private IAttributePDPs _attributeValueMetaAttrPDP;
	private boolean _isProviderValue;
	private boolean _mapForConsumer;
	private Log _log;

	/**
	 * 
	 */
	public SimpleValuePDP(
		URI providerAttrID,
		URI consumerAttrType,
		IAttributePDPs attributePDP,
		IAttributePDPs attributeValueMetaAttrPDP,
		ISimpleAttrValue value, boolean isProviderValue)
	{
		_providerAttrID = providerAttrID;
		_consumerAttrID = consumerAttrType;
		_attributePDP = attributePDP;
		_attributeValueMetaAttrPDP = attributeValueMetaAttrPDP;
		_simpleValue = value;
		_isProviderValue = isProviderValue;
		_mapForConsumer = isProviderValue;
		_log = LogFactory.getLog(this.getClass());
	}

	/**
	 * @return a mapped version of the metadata
	 */
	protected ITypedValue mapValue() throws IdASException
	{
		IAttributeValue out = _simpleValue;
		_log.debug("Mapping simple value: " + _simpleValue.getCanonical());
		if (_attributePDP != null)
		{
			if (_isProviderValue == true && _mapForConsumer == true)
			{
				_log.debug("provider value");
				out = _attributePDP.providerValueToConsumer(
					_providerAttrID, ((ISimpleAttrValue)_simpleValue).getModel().getType(),
					_consumerAttrID, getDataType(),
					_simpleValue);
			}
			else if (_isProviderValue == false && _mapForConsumer == false)
			{
				_log.debug("consumer value");
				out = _attributePDP.consumerValueToProvider(
					_consumerAttrID, ((ISimpleAttrValue)_simpleValue).getModel().getType(),
					_providerAttrID, getDataType(),
					_simpleValue);
			}
		}

		if (out != null && out.isSimple() == true)
			return (ITypedValue)out; 

		throw new IdASException();
	}
	
	/* (non-Javadoc)
	 * TODO this currently returns unmapped cononical values
	 * @see org.eclipse.higgins.idas.ISimpleAttrValue#getCanonical()
	 */
	public String getCanonical() throws IdASException
	{
		return mapValue().getCanonical();
	}

		/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.ISimpleAttrValue#getLexical()
	 */
	public String getLexical() throws IdASException
	{
		return mapValue().getLexical();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.ISimpleAttrValue#setData(java.lang.Object)
	 */
	public void setData(
		Object copyFrom) throws IdASException
	{
		if (_attributePDP != null)
		{
			URI  tempType = null;
			if (_isProviderValue == true && _mapForConsumer == true)
				tempType = _attributePDP.providerTypeToConsumer(_providerAttrID, ((ISimpleAttrValue)_simpleValue).getModel().getType());
			else
				tempType = _attributePDP.consumerTypeToProvider(_consumerAttrID, ((ISimpleAttrValue)_simpleValue).getModel().getType());

			IAttributeValue  out = null, 
				temp = new BasicSimpleValue(tempType, copyFrom);

			//logic looks backwards because this is an incoming request
			if (_isProviderValue == true && _mapForConsumer == true)
			{
				out = _attributePDP.consumerValueToProvider(
					_consumerAttrID, ((ISimpleAttrValue)_simpleValue).getModel().getType(),
					_providerAttrID, getDataType(),
					temp);
			}
			else if (_isProviderValue == false && _mapForConsumer == false)
			{
				out = _attributePDP.providerValueToConsumer(
					_consumerAttrID,((ISimpleAttrValue)_simpleValue).getModel().getType(),
					_providerAttrID, getDataType(),
					temp);
				}
			if ( out != null && out.isSimple() == true)
				_simpleValue.setData(((ISimpleAttrValue)out).getData());
		}
		else
			_simpleValue.setData(copyFrom);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.IPropertyValue#isSimple()
	 */
	public boolean isSimple() throws IdASException
	{
		return true;
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.higgins.idas.ISimpleAttrValue#getData()
	 */
	public Object getData() throws IdASException
	{
		return mapValue().getData();
	}

	/*
	 * @see org.eclipse.higgins.idas.api.IAttributeValue#getType()
	 */
	public URI getDataType() throws IdASException
	{
		return getValueType();
	}

	public URI getValueType() throws IdASException {
		if ( _dataType == null)
		{
			if (_attributePDP != null)
			{
				if (_isProviderValue == true && _mapForConsumer == true)
				{
					_dataType = _attributePDP.providerTypeToConsumer(_providerAttrID, ((ISimpleAttrValue)_simpleValue).getModel().getType());
				}
				else if (_isProviderValue == false && _mapForConsumer == false)
				{
					_dataType = _attributePDP.consumerTypeToProvider(_consumerAttrID, ((ISimpleAttrValue)_simpleValue).getModel().getType());
				}
			}
			if (_dataType == null)
				_dataType = ((ISimpleAttrValue)_simpleValue).getModel().getType();
		}
		return _dataType;
	}
	

	/*
	 * @see org.eclipse.higgins.idas.api.IAttributeValue#remove()
	 */
	public void remove() throws IdASException
	{
		_simpleValue.remove();
	}

	/**
	 * Quick to implement and possibly low performance impl
	 */
	public boolean equals(IAttributeValue value) throws IdASException {
		BasicSimpleValue thisVal = new BasicSimpleValue(this.getDataType(), this.getData());
		return thisVal.equals(value);
	}

	public IAttributeSimpleValueModel getModel() throws IdASException {
		throw new NotImplementedException();
	}

	public IAttribute addAttribute(URI attrID) throws IdASException,
			InvalidTypeException {
		if (_attributeValueMetaAttrPDP != null )
		{
			IAttribute newMetadata = null;
			if ( _isProviderValue == true && _mapForConsumer == true)
				newMetadata = _simpleValue.addAttribute(_attributeValueMetaAttrPDP.consumerIDToProvider(attrID));
			else if  ( _isProviderValue == false && _mapForConsumer == false)
				newMetadata = _simpleValue.addAttribute(_attributeValueMetaAttrPDP.providerIDToConsumer(attrID));
			return new BasicAttributePDP(_attributeValueMetaAttrPDP, _attributeValueMetaAttrPDP, _attributeValueMetaAttrPDP, newMetadata, false);
		}
		return _simpleValue.addAttribute(attrID);
	}

	public IAttribute addAttribute(IAttribute copyFrom) throws IdASException {
		if (_attributeValueMetaAttrPDP != null )
			copyFrom = new BasicAttributePDP(_attributeValueMetaAttrPDP, _attributeValueMetaAttrPDP, _attributeValueMetaAttrPDP, copyFrom, false);
		return _simpleValue.addAttribute(copyFrom);
	}

	public boolean equals(IHasAttributes attributes) throws IdASException {
		BasicAttributeSet thisMeta = new BasicAttributeSet(this.getAttributes(), null, null);
		return thisMeta.equals(attributes);
	}

	/*
	 * TODO optimize this routine, shouldn't have to waste time with every possible
	 * value, but this is currently the quick and the dirty way to do it.
	 * @see org.eclipse.higgins.idas.IHasMetadata#getMetadata(java.net.URI)
	 */
	public IAttribute getAttribute(URI attrID) throws IdASException {
		if (_attributeValueMetaAttrPDP != null)
		{
			Iterator itr = getAttributes();
			while (itr !=null && itr.hasNext())
			{
				BasicAttributePDP temp = new  BasicAttributePDP(_attributeValueMetaAttrPDP, _attributeValueMetaAttrPDP, _attributeValueMetaAttrPDP, (IAttribute)itr.next(), 
					_isProviderValue);
				if (temp.getAttrID() == attrID)
					return temp;
			}
			return null;
		}
		return _simpleValue.getAttribute(attrID);
	}

	public Iterator getAttributes() throws IdASException {
		if (_attributeValueMetaAttrPDP != null)
		{
			return new BasicAttributePDPIter(_attributeValueMetaAttrPDP, _attributeValueMetaAttrPDP, _attributeValueMetaAttrPDP, _simpleValue.getAttributes(), _isProviderValue);
		}
		return _simpleValue.getAttributes();
	}

	public ISingleValuedAttribute getSingleValuedAttribute(URI attrID)
			throws IdASException, NotSingleValuedAttributeException {
		if (_attributeValueMetaAttrPDP != null)
		{
			Iterator itr = getAttributes();
			while (itr !=null && itr.hasNext())
			{
				BasicAttributePDP temp = new  BasicAttributePDP(_attributeValueMetaAttrPDP, _attributeValueMetaAttrPDP, _attributeValueMetaAttrPDP, (IAttribute)itr.next(), 
					_isProviderValue);
				if (temp.getAttrID() == attrID)
					return temp.getSingleValuedAttribute(attrID);
			}
			return null;
		}
		return _simpleValue.getSingleValuedAttribute(attrID);
	}

	public void removeAttribute(URI attrID) throws IdASException {
		if (_attributeValueMetaAttrPDP != null)
		{
			Iterator itr = getAttributes();
			while (itr !=null && itr.hasNext())
			{
				BasicAttributePDP temp = new BasicAttributePDP(_attributeValueMetaAttrPDP, _attributeValueMetaAttrPDP, _attributeValueMetaAttrPDP, (IAttribute)itr.next(), 
					_isProviderValue);
				if (temp.getAttrID() == attrID)
					temp.remove();
			}
			return;
		}
		_simpleValue.removeAttribute(attrID);
	}

	public void removeAttributeValue(URI attrID, Object value)
			throws IdASException {
		if (_attributeValueMetaAttrPDP != null )
			throw new NotImplementedException();
		_simpleValue.removeAttributeValue(attrID, value);
	}

	public void removeAttributeValue(IAttribute attr) throws IdASException {
		if (_attributeValueMetaAttrPDP != null )
			throw new NotImplementedException();
		_simpleValue.removeAttributeValue(attr);
	}

}
