/**
 * Copyright (c) 2007 Novell, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *  	Duane Buss
 *  	Tom Doman
 */

package org.eclipse.higgins.util.jscript;

import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.mozilla.javascript.*;

/**
 * Simplified representation of a PDP utilizing Mozilla's  
 * <a href="http://www.mozilla.org/rhino/">Rhino</a> project as a JavaScript
 * interpreter for policies. 
 * 
 * This object is used to evaluate the script and holds references to 
 * the compiled JavaScript fragment within a 
 * {@link org.eclipse.higgins.util.jscript.JScriptScope}<p>
 *  
 * Critical exceptions are logged as errors via commons logging
 * 
 * @author dbuss@novell.com
 */

	public class JScriptExec
	{
		private JScriptScope _jsSharedScope;
		
		private String scriptText;
		private Script script;
		private String _scriptID;

		private Log _log;

		private static int _unnamedScripts = 0;

		/**
		 * Create a new object to cache a compiled version of a JavaScript 
		 * fragement.   Evaluation is delayed until requested.
		 * 
		 * @param sharedScope The scope this script will be compiled in.  
		 * The parameter ay be null if this script is to be evaluated in 
		 * isolation.
		 * @param script The actual JavaScript as text 
		 * @param scriptID Script Identifier, if null a name will be generated 
		 * for the fragment.
		 * @throws JScriptException
		 */
		
		private JScriptException _handleException( Exception e, String scriptID ) throws JScriptException
		{
			if ( e instanceof WrappedException)
			{
				Throwable  x = ((WrappedException)e).getWrappedException();
					if (_log != null)
						_log.error("Error evaluating script " + _scriptID, x);
				throw new JScriptException(x);
			}
			else
			{
				if (_log != null)
					_log.error("Error evaluating script " + _scriptID, e);
				throw new JScriptException(e);
			}
		}
		
		public JScriptExec(JScriptScope sharedScope, String script, String scriptID)
			throws JScriptException
		{
			// setup logging
			_log = LogFactory.getLog(this.getClass());
			
			this.scriptText = script; 
			this._jsSharedScope = sharedScope;
			if (scriptID == null)
				_scriptID = "executable script: " + (++_unnamedScripts);
			else
				_scriptID = scriptID;
			
			Context cx = Context.enter();
			try {
				// precompile the string for faster evaluation later
				this.script = cx.compileString(this.scriptText, scriptID, 1, null);
			}
			catch (Exception e)
			{
				_handleException(e, scriptID);
			}
			finally {
				Context.exit();
			}
		}
		
		/**
		 *  Evaluate a previously complied script fragment.   
		 *  Use this function to pass java objects to the script,
		 *  all objects passed in are added to a local scope.  
		 *  
		 * @param paramNames ordered list of string versions of the parameter 
		 * names.   These are the names which will be stored in the local 
		 * JavaScript scope for use by the JavaScript code fragment 
		 * @param params order list containing the objects represented 
		 * by the string names.  This will be wrapped and made availible 
		 * for use by the Script in the evaluation. 
		 * @return A Java object representing the result of evaluation.   
		 *   By default it returns the contents of the JavaScript variable
		 *   RESULT if defined, otherwise it returns the evaluation result of 
		 *   the script.
		 * @throws JScriptException
		 */
		public Object evaluate(String[] paramNames, Object[] params)
						throws JScriptException
		{
			Context cx = Context.enter();
			ScriptableObject sharedScope = getSharedScope();
			try {
				// init thread specific context 
			   Scriptable localScope = cx.newObject(sharedScope);
			   localScope.setPrototype(sharedScope);
			   localScope.setParentScope(null);
			   
			   if ( paramNames != null)
			   {
					for (int count=0; count < paramNames.length; count++ ) {
						Object wrappedObj = Context.javaToJS(params[count], localScope);
						ScriptableObject.putProperty(localScope, paramNames[count], wrappedObj);
					}
			   }
			   
				Object evalResult = script.exec(cx, localScope);
				Object result = localScope.get("RESULT", localScope);
				Object retVal = null;
				
				if (result == Scriptable.NOT_FOUND || result instanceof Undefined) {
					if ( evalResult != Scriptable.NOT_FOUND && !(evalResult instanceof Undefined))
						retVal = evalResult;
				}else 
					retVal = result;
				if (retVal instanceof Wrapper)
					retVal = ((Wrapper)retVal).unwrap();
				if (retVal instanceof NativeArray)
					retVal = new JScriptObjectImmutableIterator((NativeArray)retVal);
				else if (retVal instanceof Iterator)
					retVal = new JScriptObjectImmutableIterator((Iterator)retVal);
				else if (retVal instanceof Map)
					retVal = new JScriptObjectImmutableIterator((Map)retVal);				
				else if (retVal instanceof HashSet)
					retVal = new JScriptObjectImmutableIterator((HashSet)retVal);
				return retVal;
			}catch (Exception e) {
				return _handleException(e, _scriptID);
			}finally {
				Context.exit();
			}
		}
		
		/**
		 *  Evaluate a previously complied script fragment.   
		 *  Use this function to pass java objects to the script,
		 *  all objects passed in are added to a local scope.  
		 *  
		 * @param paramNames ordered list of string versions of the parameter 
		 * names.   These are the names which will be stored in the local 
		 * JavaScript scope for use by the JavaScript code fragment 
		 * @param params order list containing the objects represented 
		 * by the string names.  This will be wrapped and made availible 
		 * for use by the Script in the evaluation.
		 *  
		 * @return A Java object representing the result of evaluation.   
		 *   By default it returns the contents of the JavaScript variable
		 *   RESULT if defined, otherwise it returns the evaluation result of 
		 *   the script.
		 *   @throws JScriptException
		 */
		public Object evaluate(
			String[] paramNames, Object[] params, 
			String[] outputNames, Object[] outputObjects)
			throws JScriptException
		{
			Context cx = Context.enter();
			ScriptableObject sharedScope = getSharedScope();
			try {
				// init thread specific context 
			   Scriptable localScope = cx.newObject(sharedScope);
			   localScope.setPrototype(sharedScope);
			   localScope.setParentScope(null);
			   
				for (int count=0; count < paramNames.length; count++ ) {
					Object wrappedObj = Context.javaToJS(params[count], localScope);
					ScriptableObject.putProperty(localScope, paramNames[count], wrappedObj);
				}
				Object evalResult = script.exec(cx, localScope);
				
				for (int count=0; count < outputNames.length; count++ ) {
					Object wrappedObj = localScope.get(outputNames[count], localScope);
					if (wrappedObj instanceof Wrapper)
						outputObjects[count] = ((Wrapper)wrappedObj).unwrap();
					else
						outputObjects[count] = wrappedObj;
				}
				
				Object result = localScope.get("RESULT", localScope);
				Object retVal = null;
				
				if (result == Scriptable.NOT_FOUND || result instanceof Undefined) {
					if ( evalResult != Scriptable.NOT_FOUND && !(evalResult instanceof Undefined))
						retVal = evalResult;
				}else 
					retVal = result;
				if (retVal instanceof Wrapper)
					retVal = ((Wrapper)retVal).unwrap();
				return retVal;
			}catch (Exception e) {
				return _handleException(e, _scriptID);
			}finally {
				Context.exit();
			}
		}
		
		/**
		 * Evaluate a previously complied script fragment. Use this version
		 * of evaluate if all that is needed is to set a single java string
		 * in the local scope for use by the script during evaluation.
		 * 
		 * @param ParamName = string name of the parameter
		 * @param Param value of the parameter
		 * @return results of Java Script evaluation
		 * @throws JScriptException
		 */
		public Object evaluate(String ParamName, Object Param)
					throws JScriptException
		{
			String[] paramNames = {ParamName};
			Object[] params = {Param};
			return evaluate(paramNames, params);
		}
		
		/**
		 * 
		 * @return a Rhino JavaScript shared scope, may be null or one which 
		 * is shared cross rhino contexts and threads. 
		 */
		private synchronized  ScriptableObject getSharedScope()
	   {
	   	if ( _jsSharedScope == null)
   			_jsSharedScope = new JScriptScope();
			return _jsSharedScope.getScope();
	   }
	  
	}
