/**
 * Copyright (c) 2007 Novell, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *  	Duane Buss
 *  	Tom Doman
 */

package org.eclipse.higgins.util.jscript;

import java.net.URI;
import java.util.Iterator;

/**
 * Random helper functions for dealing with Java - JavaScript 
 * conversion and storage.
 * 
 * 
 * @author dbuss@novell.com
 */

public class JScriptExecHelper
{
	/**
	 * Allows String Transformation
	 * @param exec The Compiled JavaScript,   If null the in param will be the return value 
	 * @param paramName A string specifying the name of the JavaScript variable 
	 * the input String is accessible by.  
	 * @param param  The original value
	 * @return a String, if more than one String is returned by the JavaScript only the first
	 * one will be returned from this routine.  If for any reason the value cannot be
		 * computed, the in param will be returned.
	 * @throws JScriptException 
	 */
	public static String transformString(
		JScriptExec exec, 
		String paramName, 
		String param) 
		throws JScriptException
	{
		String value = evaluateToString(exec, paramName, param);
		if (value == null)
			value = param;
		return value;
	}
	
	
	/**
	 * Allows URI Transformation
	 * @param exec The Compiled JavaScript,   If null the in param will be the return value 
	 * @param paramNames A string array specifying the names of the JavaScript variables 
	 * the URI accessible by.  
	 * @param in  The original URI value
	 * @return a URI, if more than one URI is returned by the JavaScript only the first
	 * one will be returned from this routine.   If for any reason the value cannot be
		 * computed, the in param will be returned.
	 * @throws JScriptException 
	 */
	public static URI transformURI(
		JScriptExec exec, 
		String[] paramNames, 
		URI in) 
		throws JScriptException
	{
		Object params[] = {in, in};		
			
		if (exec != null)
		{
			try {
				Object value = exec.evaluate(paramNames, params);
				if ( value != null ) {
					if (value instanceof URI)
						return (URI)value;
					else if (value instanceof String)
						return new URI((String)value);
					else if (value instanceof JScriptObjectImmutableIterator) {
	               Object temp = ((JScriptObjectImmutableIterator)value).next();
	               if (temp instanceof URI)
							return (URI)temp;
						else if (temp instanceof String)
							return new URI((String)temp);
					}
	            throw new JScriptException("Illegal object type returned from mapping, got: "
								+ value.getClass().toString());
				}
			}catch (Exception e)	{
				throw new JScriptException("JSPropertyPDP URI helper encountered a problem",  e);
			}
		}
		return in;
	}

	/**
	 * Allows URI Transformation
	 * @param exec The Compiled JavaScript,   If null the in param will be the return value 
	 * @param paramNames A string array specifying the names of the JavaScript variables 
	 * the URI accessible by.  
	 * @param in  The original URI value and other data as needed
	 * @return a URI, if more than one URI is returned by the JavaScript only the first
	 * one will be returned from this routine.   If for any reason the value cannot be
		 * computed, null will be returned.
	 * @throws JScriptException 
	 */
	public static URI transformURI(
		JScriptExec exec, 
		String[] paramNames, 
		URI[] in) 
		throws JScriptException
	{
		if (exec != null)
		{
			try {
				Object value = exec.evaluate(paramNames, in);
				if ( value != null ) {
					if (value instanceof URI)
						return (URI)value;
					else if (value instanceof String)
						return new URI((String)value);
					else if (value instanceof JScriptObjectImmutableIterator) {
	               Object temp = ((JScriptObjectImmutableIterator)value).next();
	               if (temp instanceof URI)
							return (URI)temp;
						else if (temp instanceof String)
							return new URI((String)temp);
					}
	            throw new JScriptException("Illegal object type returned from mapping, got: "
								+ value.getClass().toString());
				}
			}catch (Exception e)	{
				throw new JScriptException("JSPropertyPDP URI helper encountered a problem",  e);
			}
		}
		return null;
	}

		/**
		 * Allows URI Transformation
		 * @param exec The Compiled JavaScript,   If null the in param will be the return value 
		 * @param paramName A string specifying the name of the JavaScript variable 
		 * the URI will accessible by.  
		 * @param in  The original URI value
		 * @return a URI, if more than one URI is returned by the JavaScript only the first
		 * one will be returned from this routine.  If for any reason the value cannot be
		 * computed, the in param will be returned.
		 * @throws JScriptException 
		 */
		public static URI transformURI(
			JScriptExec exec, 
			String paramName, 
			URI in) 
			throws JScriptException
		{
			String [] paramNames = {paramName};
			return transformURI(exec, paramNames, in);
		}
		
		/**
		 * Allows URI Transformation
		 * @param exec The Compiled JavaScript,   If null the in param will be the return value 
		 * @param paramName A string specifying the name of the JavaScript variable 
		 * the URI will accessible by.  
		 * @param in  The original URI value
		 * @return a Iterator, if more than one URI is returned by the JavaScript only the first
		 * one will be returned from this routine.
		 * @throws JScriptException 
		 */
		public static Iterator transformURItoMany(
			JScriptExec exec, 
			String paramName, 
			URI in) 
			throws JScriptException
		{
			if (exec != null)
			{
				try {
					Object value = exec.evaluate(paramName, in);
					
					if ( value != null) {
						if (value instanceof URI)
							return new SingleObjectImmutableIterator((Object)value);
						else if (value instanceof String)
							return new SingleObjectImmutableIterator((Object)(new URI((String)value)));
						else if (value instanceof JScriptObjectImmutableIterator) {
							return new JScriptObjectImmutableIterator((Iterator)value, URI.class);
						}
						else
							throw new JScriptException("Illegal object type returned from mapping, got: "
									+ value.getClass().toString());					
					}
				}
				catch (Exception e) {
					throw new JScriptException(e);
				}
			}
			return new SingleObjectImmutableIterator((Object)(in));
		}

		
		/**
		 * Allows String Transformation or Retrieval
		 * @param exec The Compiled JavaScript,   If nul the in param will be the return value 
		 * @param paramNames An array of strings specifying the names of the JavaScript variables 
		 * @param params  An object array 
		 * @return a String, if more than one String is returned by the JavaScript only the first
		 * one will be returned from this routine.
		 * @throws JScriptException 
		 */
		public static String evaluateToString(
			JScriptExec exec, 
			String[] paramNames, 
			Object[] params) 
			throws JScriptException
		{
			if (exec != null)
			{
				try {
					Object value = exec.evaluate(paramNames, params);
					if ( value != null ) {
						if (value instanceof String)
							return (String)value;
						else if (value instanceof JScriptObjectImmutableIterator) {
		               Object temp = ((JScriptObjectImmutableIterator)value).next();
							if (temp instanceof String)
								return (String)temp;
							else
								 throw new JScriptException("Illegal object type returned from mapping, got: "
										+ temp.getClass().toString());
						}
		            throw new JScriptException("Illegal object type returned from mapping, got: "
									+ value.getClass().toString());
					}
					else {
						return null;
					}
				}catch (Exception e)	{
					throw new JScriptException("evaluateToString helper encountered a problem",  e);
				}
			}
			return null;
		}
		
		/**
		 * Allows String Transformation or Retrieval
		 * @param exec The Compiled JavaScript,   If nul the in param will be the return value 
		 * @param paramName A string specifying the names of the JavaScript variables 
		 * @param in  The original URI value
		 * @return a String, if more than one String is returned by the JavaScript only the first
		 * one will be returned from this routine.
		 * @throws JScriptException 
		 */
		public static String evaluateToString(
			JScriptExec exec, 
			String paramName, 
			Object in) 
			throws JScriptException
		{
			String[] paramNames = {paramName};
			Object[] params = {in};
			
			return evaluateToString(exec, paramNames, params);
		}
		
	}
