/**
 * Copyright (c) 2007 Novell, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *  	Duane Buss
 *  	Tom Doman
 */

package org.eclipse.higgins.util.jscript;

import java.net.URI;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.NoSuchElementException;

import org.mozilla.javascript.NativeArray;
import org.mozilla.javascript.Scriptable;
import org.mozilla.javascript.Wrapper;

/**
 * Allows single objects to be viewed as an iterator 
 * 
 * This is a version of Iterator that overrides the <code>remove</code>
 * method so that items can't be taken out of the Native Array, and helps 
 * with type mapping.
 *  @author dbuss@novell.com
 */

public class JScriptObjectImmutableIterator implements Iterator {

	// if we wrap an interator
	private Iterator _itr;
	
	// the NativeArray we're wrapping
	private NativeArray _na;
	// iterator state if wrapping a Native Array
	private int _pos;

	//if we have to do object conversions
	private Class returnClass;
	
	/**
	 * Create a new ImmutableIterator
	 */
	public JScriptObjectImmutableIterator(NativeArray obj) {
   	 this._na = obj;
   	 _pos = 0;
	}
	       
	public JScriptObjectImmutableIterator(Iterator obj) {
  	 this._itr = obj;
	}
	
	public JScriptObjectImmutableIterator(Iterator obj, Class returnClass) {
	  	 this._itr = obj;
	  	 this.returnClass = returnClass;
		}
	
	public JScriptObjectImmutableIterator(Map obj) {
	  	this(obj.entrySet().iterator());
	}
	
	public JScriptObjectImmutableIterator(HashSet obj) {
	  	 this(obj.iterator());
	}
	       
	/**
	 * Standard hasNext method
	 */
	public boolean hasNext() {
		if (_na != null) {
			Object obj = _na.get(_pos, _na.getParentScope());
			if (obj != null && obj != Scriptable.NOT_FOUND)
				return true;
			else 
				return false;
		} else
			return _itr.hasNext();
	}

	/**
	 * Standard next method
	 */
	public Object next() throws NoSuchElementException {
		Object initialObj, intermediateObj, finalObj = null;
		if (_na != null) {
			initialObj = _na.get(_pos++, _na.getParentScope());
		} else {
			initialObj = _itr.next();
		}
		if (initialObj instanceof Wrapper)
			intermediateObj =  ((Wrapper)initialObj).unwrap();
		else
			intermediateObj  = initialObj;
		if (returnClass == null )
			finalObj = intermediateObj;
		else
		{
			try 
			{
				if (returnClass == URI.class)
				{
					if (intermediateObj instanceof String)
						finalObj = new URI((String)intermediateObj);
					else if (intermediateObj instanceof URI)
						finalObj = intermediateObj; 
				}
			}
			catch (Exception e)
			{
				//visit what to do here?
				;
			}
			if (finalObj == null)
				throw new NoSuchElementException("Illegal object type returned from mapping, got: "
					+ intermediateObj.getClass().toString());
		}
		return finalObj;
	}

	/**
	 * Makes sure that no one can remove any elements from the collection
	 */
	public void remove() throws UnsupportedOperationException {
		throw new UnsupportedOperationException();
	}
}		
