/**
 * Copyright (c) 2007 Novell, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *  	Duane Buss
 *  	Tom Doman
 */

package org.eclipse.higgins.util.jscript;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.mozilla.javascript.*;

/**
 * Simplified representation of a <a href="http://www.mozilla.org/rhino/">Rhino</a>
 * scope for JavaScript evaluation.  This allows multiple JavaScript fragments to
 * be compiled and evaluated within a global shared scope, reusing functions and 
 * variables as appropriate.  
 * 
 * For further information on 
 * <a href="http://www.mozilla.org/rhino/scopes.html">scopes</a> please refer to
 * the Mozilla documentation.
 * 
 * Critical exceptions are logged as errors via commons logging
 * 
 * @author dbuss@novell.com
 */

public class JScriptScope
{
	private ScriptableObject _jsSharedScope;
	private String _scriptID;		

	private Log _log;

	private static int _unnamedScripts = 0;

	/**
	 * Default constructor for a shared JavaScript Scope
	 * This should only be done during startup to prevent syncronization problems.  
	 */
	public JScriptScope()
	{
	   // Shouldn't happen, but just in case...
      if (earlyException != null)
          throw earlyException;

		// setup logging
		_log = LogFactory.getLog(this.getClass());

		// initialize object
		getScope();
	}

	/**
	 * Parses, complies and evaluates the script, the output of the script should
	 * result in the shared scope containing global variables.
	 * 
	 * All Scripts should be added to the scope before it is used.
	 *  
	 * @param scriptText - string version of a JavaScript
	 * @param scriptID - and identifier for the script
	 */
	public synchronized  void addToScope(
		String scriptText, String scriptID) throws JScriptException
	{
		// Shouldn't happen, but just in case...
		if (earlyException != null)
			throw earlyException;

		if (scriptID == null)
			_scriptID = "shared script: " + (++_unnamedScripts);
		else
			_scriptID = scriptID;

	   	Context cx = Context.enter();
		ScriptableObject sharedScope = getScope();
		try {
			Script script  = cx.compileString(scriptText, scriptID, 1, null);

			// ignore the return, we are intrested in side effects not evaluation
			// resuts by golly.
			script.exec(cx, sharedScope);

		} catch (Exception e) {
			// panic, this will happen often with poor scripts
			if (_log != null)
				_log.error("Error adding script to shared scope, script: " + _scriptID, e);				
			throw new JScriptException(e);
		} finally {
			Context.exit();
		}
	}

	/**
	 * @return An instance of a scope suitable for calls to Rhino.   
	 * Currently we setup the scope such that it can be used for calling
	 * back and forth to JAVA.
	 */
	public synchronized  ScriptableObject getScope()
	{
		if ( _jsSharedScope == null) {
			Context cx = Context.enter();
	   		try {
	   			// although the documentation samples show initializing the standard
	   			// objects I found that in order to have the shared scope setup
	   			// properly for importpackage statements I had to make the call
	   			// to ImporterTopLevel.   It seems to do the standard object init
	   			// as well.
	   			//_jsSharedScope = cx.initStandardObjects();
	   			
	   			_jsSharedScope = new ImporterTopLevel(cx, true);
	   			Object wrappedObj = Context.javaToJS(System.out, _jsSharedScope);
					ScriptableObject.putProperty(_jsSharedScope, "out", wrappedObj);
	   			
	   		} finally {
	   			Context.exit();
	   		}
	   	}
		return _jsSharedScope;
	}

	/**
	 * 
	 *
	 */
	static class DynamicScopeFactory extends ContextFactory
	{
		protected boolean hasFeature(Context cx, int featureIndex)
		{
			if (featureIndex == Context.FEATURE_DYNAMIC_SCOPE)
				return true;
			else 
				return super.hasFeature(cx, featureIndex);
		}
	}

   /**
    * RuntimeException that wraps an Exception thrown during the
    * creation of static elements above, null if none.
    */
	private static RuntimeException earlyException;

	/**
	 * Static initializer 
	 */
	static {
		try {
			// setup some javascript singletons
    		ContextFactory.initGlobal(new DynamicScopeFactory());
		} catch (Exception e) {
			earlyException = new IllegalArgumentException();
			earlyException.initCause(e);
		}
	};
}
