/*******************************************************************************
 * Copyright (c) Novell Inc.   2008
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Duane Buss - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.util.saml;

import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Vector;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 * Basic container for a saml attribute
 *
 *The element parsed should be the saml attribute.
 */
public class SAMLAttribute extends XMLElement {

	protected  String nameSpace; 
	
	public SAMLAttribute(Document document) {

		super(document, SAMLConstants.PREFIX_SAML_ASSERTION, SAMLConstants.NS_SAML_ASSERTION, "Attribute");
	}

	public SAMLAttribute() {

		this((Document) null);
	}

	public SAMLAttribute(InputStream stream) throws SAXException, IOException {

		super(stream);
		nameSpace = this.element.getNamespaceURI();
	}

	public SAMLAttribute(Reader reader) throws SAXException, IOException {

		super(reader);
		nameSpace = this.element.getNamespaceURI();
	}

	/**
	 * 
	 * @param document
	 * @param element  the dom element representing the saml attribute
	 */
	public SAMLAttribute(Document document, Element element) {

		super(document, element);
		nameSpace = element.getNamespaceURI();
	}

	public String getNameID() {

		return this.element.getAttribute("AttributeName");
	}

	public void setNameID(String value) {

		this.element.setAttribute("AttributeName", value);
	}

	public String getNameSpace() {

		return this.element.getAttribute("AttributeNamespace");
	}

	public void setNameSpace(String value) {

		this.element.setAttribute("AttributeNamespace", value);
	}
	
	public URI getFullName() {
		try {
			return new URI(getNameSpace() +'/'+ getNameID());
		} catch (URISyntaxException e) {
			return null;
		}
	}
	
	/**
	 * Can I deal with this as just a string?
	 * @return true if the value is a single valued xsd:string
	 */
	public boolean isSimpleSingleValuedValue() {
		NodeList nl = this.element.getElementsByTagNameNS(nameSpace, "AttributeValue");
		if ( nl == null)
			nl = this.element.getElementsByTagName("AttributeValue");
		
		if ( nl == null || (nl != null && nl.getLength() <= 1))
			return true;
		
		return false;
	}
	
	/**
	 * Only works on single valued strings.
	 * @return String value
	 */
	public String getValue() {
		
		if ( isSimpleSingleValuedValue() == false)
			return null;
	
		Element element = (Element)this.element.getElementsByTagNameNS(nameSpace, "AttributeValue").item(0);
		if ( element == null)
			element = (Element)this.element.getElementsByTagName("AttributeValue").item(0);
		
		if ( element == null) return null;
		
		return (getTextContent(element));
	}
	
	/**
	 * 
	 * @return Vector of string values
	 */
	public Vector  getSimpleValues()
	{
		Vector claimValues = new Vector();
		
		NodeList nl = this.element.getElementsByTagNameNS(nameSpace, "AttributeValue");
		if ( nl == null)
			nl = this.element.getElementsByTagName("AttributeValue");
		
		if (nl != null && nl.getLength() >= 1)
		{
			for (int index = 0; index < nl.getLength(); index++)
				claimValues.add(getTextContent( (Element)nl.item(index)));
		}
		return claimValues;
	}
}
