/*******************************************************************************
 * Copyright (c) 2007 Google
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *     Duane Buss - Novell Inc, made more generic, shared code with SAML1Subject
 *******************************************************************************/
package org.eclipse.higgins.util.saml;

import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;

import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.SAXException;

public class SAMLSubject extends XMLElement {

	
	public SAMLSubject(Document document, String nameSpace) {

		super(document, SAMLConstants.PREFIX_SAML_ASSERTION, nameSpace, "Subject");
	}

	public SAMLSubject(String nameSpace) {

		this((Document) null, nameSpace);
		this.nameSpace = nameSpace;
	}

	public SAMLSubject(InputStream stream) throws SAXException, IOException {

		super(stream);
	}

	public SAMLSubject(Reader reader) throws SAXException, IOException {

		super(reader);
	}

	public SAMLSubject(Document document, Element element) {

		super(document, element);
	}

	protected Element getNameIDElement() {
		
		String id;
		if ( nameSpace.equals(SAMLConstants.NS_SAML_1_ASSERTION))
			id = "NameIdentifier";
		else
			id = "NameID";
		
		Element element = (Element) this.element.getElementsByTagNameNS(nameSpace, id).item(0);
		if (element == null) element = (Element) this.element.getElementsByTagName(id).item(0);
		
		return(element);
	}

	public String getNameIDFormat() {

		Element elementNameID = this.getNameIDElement();
		if (elementNameID == null) return(null); 
		
		Attr attrFormat = elementNameID.getAttributeNode("Format");
		if (attrFormat == null) return(null);
		
		return(attrFormat.getValue());
	}

	public void setNameIDFormat(String value) {

		String id;
		if ( nameSpace.equals(SAMLConstants.NS_SAML_1_ASSERTION))
			id = "NameIdentifier";
		else
			id = "NameID";
		
		Element elementNameID = this.getNameIDElement();
		if (elementNameID == null) {

			elementNameID = this.document.createElementNS(nameSpace, id);
			elementNameID.setPrefix(SAMLConstants.PREFIX_SAML_ASSERTION);
			this.element.appendChild(elementNameID);
		}

		elementNameID.setAttribute("Format", value);
	}

	public String getNameID() {

		Element elementNameID = this.getNameIDElement();
		if (elementNameID == null) return(null); 
		
		return(getTextContent(elementNameID));
	}

	public void setNameID(String value) {

		Element elementNameID = this.getNameIDElement();
		if (elementNameID == null) {
			String id;
			if ( nameSpace.equals(SAMLConstants.NS_SAML_1_ASSERTION))
				id = "NameIdentifier";
			else
				id = "NameID";
			elementNameID = this.document.createElementNS(nameSpace, id);
			elementNameID.setPrefix(SAMLConstants.PREFIX_SAML_ASSERTION);
			this.element.appendChild(elementNameID);
		}

		setTextContent(elementNameID, value);
	}

	protected Element getSubjectConfirmationElement() {

		Element element = (Element) this.element.getElementsByTagNameNS(nameSpace, "SubjectConfirmation").item(0);
		if (element == null) element = (Element) this.element.getElementsByTagName("SubjectConfirmation").item(0);
		
		return(element);
	}

	public String getSubjectConfirmationMethod() {

		Element elementSubjectConfirmation = this.getSubjectConfirmationElement();
		if (elementSubjectConfirmation == null) return(null); 
		
		if ( nameSpace.equals(SAMLConstants.NS_SAML_1_ASSERTION)) {
			Element element = (Element)elementSubjectConfirmation.getElementsByTagNameNS(SAMLConstants.NS_SAML_1_ASSERTION, "ConfirmationMethod").item(0);
			if ( element == null)
				element = (Element) elementSubjectConfirmation.getElementsByTagName("ConfirmationMethod").item(0);
			
			if ( element == null) return null;
			
			return (getTextContent(element));
		}	
		else {
			Attr attrFormat = elementSubjectConfirmation.getAttributeNode("Method");
			if (attrFormat == null) return(null);
			
			return(attrFormat.getValue());	
		}
	}

	public void setSubjectConfirmationMethod(String value) {

		Element elementSubjectConfirmation = this.getSubjectConfirmationElement();
		if (elementSubjectConfirmation == null) {

			elementSubjectConfirmation = this.document.createElementNS(nameSpace, "SubjectConfirmation");
			elementSubjectConfirmation.setPrefix(SAMLConstants.PREFIX_SAML_ASSERTION);
			this.element.appendChild(elementSubjectConfirmation);
		}
		if ( nameSpace.equals(SAMLConstants.NS_SAML_1_ASSERTION)) {
			Element elementConfirmationMethod = this.document.createElementNS(nameSpace, "ConfirmationMethod");
			elementConfirmationMethod.setPrefix(SAMLConstants.PREFIX_SAML_ASSERTION);
			setTextContent(elementConfirmationMethod, value);
			elementSubjectConfirmation.appendChild(elementConfirmationMethod);
		}	
		else {
			elementSubjectConfirmation.setAttribute("Method", value);
		}
	}
}
