/**
 * Copyright (c) 2007 Novell, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *		Tom Doman
 */

package org.eclipse.higgins.util.socket;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.net.InetAddress;
import java.net.Socket;
import java.net.UnknownHostException;
import java.security.KeyStore;
import java.security.Security;

import javax.naming.NamingException;
import javax.net.SocketFactory;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.TrustManagerFactory;

public class HigginsSSLSocketFactory extends SSLSocketFactory
{
	private static final Object KEYSTORE_SYNC_OBJECT = new Object();
	private KeyStore _keyStore;
	private File _keyStoreFile;
	private String _keyStorePassword;

	/**
	 * 
	 */
	public static SocketFactory getDefault()
	{
		return new HigginsSSLSocketFactory();
	}

	/**
	 * Constructor for HigginsSSLSocketFactory.
	 */
	private HigginsSSLSocketFactory()
	{
		super();
	}

	/**
	 * 
	 * @param keyStoreFile
	 */
	public void setKeyStore(
		File keyStoreFile,
		String keyStorePassword)
	{
		_keyStoreFile = keyStoreFile;
		_keyStorePassword = keyStorePassword;
	}

	/**
	 * 
	 * @throws NamingException
	 */
	public void reloadKeyStore() throws NamingException
	{
		try
		{
			synchronized (KEYSTORE_SYNC_OBJECT)
			{
				_initializeKeyStore(true);
			}
		}
		catch (Exception e)
		{
			NamingException ne = new NamingException();
			ne.setRootCause(e);
		}
	}

	/**
	 * 
	 */
	public String[] getDefaultCipherSuites()
	{
		return new String[0];
	}

	/**
	 * 
	 */
	public String[] getSupportedCipherSuites()
	{
		return new String[0];
	}

	/**
	 * 
	 * @param bRefresh
	 * @throws Exception
	 */
	private void _initializeKeyStore(
		boolean bRefresh) throws Exception
	{
		if ((_keyStore == null) || bRefresh)
		{
			_keyStore = KeyStore.getInstance(KeyStore.getDefaultType());
			if ((_keyStoreFile != null) && _keyStoreFile.exists())
			{
				FileInputStream fis = new FileInputStream(_keyStoreFile);
				_keyStore.load(fis, null);
				fis.close();
			}
			else
			{
				_keyStore.load(null, null);
			}
		}
	}

	/**
	 * 
	 * @return
	 * @throws IOException
	 */
	private SSLContext _createSSLContext() throws IOException
	{
		SSLContext sslContext = null;

		try
		{
			_initializeKeyStore(false);

			KeyManagerFactory kmf = KeyManagerFactory.getInstance(
				Security.getProperty("ssl.KeyManagerFactory.algorithm"));
			kmf.init(_keyStore, _keyStorePassword.toCharArray());

			TrustManagerFactory tmf = TrustManagerFactory.getInstance(
				Security.getProperty("ssl.TrustManagerFactory.algorithm"));
			tmf.init(_keyStore);

			sslContext = SSLContext.getInstance("TLS");
			sslContext.init(kmf.getKeyManagers(), tmf.getTrustManagers(), null);
		}
		catch (Exception e)
		{
			throw new IOException(e.getMessage());
		}

		return sslContext;
	}

	/**
	 * 
	 * @return
	 * @throws IOException
	 */
	private SSLContext _getSSLContext() throws IOException
	{
		return _createSSLContext();
	}

	/**
	 * 
	 */
	public Socket createSocket(
		String host,
		int port,
		InetAddress clientHost,
		int clientPort) throws IOException, UnknownHostException
	{
		return _getSSLContext().getSocketFactory().createSocket(host, port, clientHost,
			clientPort);
	}

	/**
	 * 
	 */
	public Socket createSocket(
		InetAddress inetAddress,
		int i,
		InetAddress inetAddress1,
		int i1) throws IOException
	{
		return _getSSLContext().getSocketFactory().createSocket(inetAddress, i,
			inetAddress1, i1);
	}

	/**
	 * 
	 */
	public Socket createSocket(
		String host,
		int port) throws IOException, UnknownHostException
	{
		return _getSSLContext().getSocketFactory().createSocket(host, port);
	}

	/**
	 * 
	 */
	public Socket createSocket(
		InetAddress inetAddress,
		int i) throws IOException
	{
		return _getSSLContext().getSocketFactory().createSocket(inetAddress, i);
	}

	/**
	 * 
	 */
	public Socket createSocket(
		Socket socket,
		String host,
		int port,
		boolean autoClose) throws IOException, UnknownHostException
	{
		return _getSSLContext().getSocketFactory().createSocket(socket, host, port,
			autoClose);
	}
}
