package org.eclipse.hyades.logging.core;

/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/

import org.eclipse.hyades.internal.execution.remote.AgentConfiguration;
import org.eclipse.hyades.internal.execution.remote.AgentControllerListener;
import org.eclipse.hyades.internal.execution.remote.MonitorListener;
import org.eclipse.hyades.internal.execution.remote.QueueManager;
import org.eclipse.hyades.internal.execution.remote.RemoteComponentSkeleton;

/**
 * An agent of type 'logging' used to queue logged records until the Agent Controller
 * is available and the Logging Agent is being monitored at which time all queued and
 * future logged records are written to the Agent Controller.
 * <p>
 * A Logging Agent instance is created via the constructor.
 * For example,
 * <p>
 * <code>
 * LoggingAgent loggingAgent = new LoggingAgent("My Logging Agent");
 *
 * loggingAgent.write("My log message.");
 * </code>
 *
 *
 * @author		Paul Slauenwhite
 * @author		Richard Duggan
 * @version	March 7, 2003
 */
public class LoggingAgent {

    /**
     * The type of the Logging Agent.
     */
    public static final String LOGGING_AGENT_TYPE = "Logging";

    /**
     * The skeleton to which to write log records.
     */
    private RemoteComponentSkeleton delegate;

    /**
     * The queue for log records until the Agent Controller is available and the Logging Agent is being monitored.
     */
    private QueueManager queue = new QueueManager();

    /**
     * If the Agent Controller is available.
     */
    private boolean controllerAvailable = false;

    /**
     * If the Logging Agent is being monitored.
     */
    private boolean isMonitored = false;

    /**
     * Current thread lock.
     */
    private Object currentThreadLock = new Object();

    /**
     * Constructor to create and initialize a Logging Agent instance using
     * the parameter name.
     *
     * @param name The name of the newly created Logging Agent.
     */
    public LoggingAgent(String name) {

        try {
            delegate = new RemoteComponentSkeleton(name, LOGGING_AGENT_TYPE);

            delegate.addAgentControllerListener(new AgentControllerListener() {

                public void agentControllerActive() {
                    controllerAvailable = true;
                }

                public void agentControllerInactive() {
                    controllerAvailable = false;
                }
            });

			delegate.addMonitorListener(new MonitorListener() {

				public void monitorActive() {

					synchronized (currentThreadLock) {
						isMonitored = true;
						currentThreadLock.notify();
					}

					flush();
				}

				public void monitorInactive() {
					isMonitored = false;
					flush();
				}
			});

            delegate.initializeFast();
        }
        catch (Throwable e) {
            controllerAvailable = false;
        }
    }

    /**
     * Retrieves the default configuration for the Logging Agent.
     *
     * @return The Logging Agent's default configuration.
     */
    public AgentConfiguration getLoggingAgentDefaultConfiguration() {
        return (delegate.getDefaultConfiguration());
    }

    /**
     * Checks if the Logging Agent is logging (e.g. Agent Controller is available and the Logging Agent is being monitored).
     *
     * @return True if the Logging Agent is logging, otherwise false.
     */
    public boolean isLogging() {
        return ((controllerAvailable) && (isMonitored));
    }

    /**
     * Writes the parameter log record to the Agent Controller
     *
     * If the Agent Controller is available and  the Logging Agent is being
     * monitored, otherwise the parameter log record is added to the Logging
     * Agent's queue of log records waiting to be written to the Agent Controller.
     *
     * @param msg The log record to be written to the Agent Controller.
     */
    public void write(String msg) {

        if (isLogging())
            delegate.logMessageUTF8(msg);

        else
            enQueue(msg);
    }

    /**
     * Adds the parameter log record to the Logging Agent's queue of log
     * records waiting to be written to the Agent Controller.
     *
     * @param msg The log record to be added Logging Agent's queue of log records.
     */
    public void enQueue(String msg) {
        queue.insertInQueue(msg);

    }

    /**
     * Writes any queued log record to the Agent Controller.
     */
    public void flush() {

        if (queue.isFlushNeeded())
            queue.flushCurrentQueue(delegate);
    }

    /**
     * Deregisters this Logging Agent instance from the Agent Controller.
     */
    public void deregister() {
        delegate.deregister();
    }

    /**
     * Retrieves the name of this Logging Agent.
     *
     * @return The name of this Logging Agent.
     */
    public String getName() {
        return delegate.getName();
    }

    /**
     * Suspends the current thread until the logging agent is monitored or the
     * the parameter maximum time (milliseconds) has expired.
     *
     * @param maxTime the maximum amount of time (milliseconds) to suspend the current thread
     * @return true if the logger is being monitored
     */
    public boolean waitUntilLogging(long maxTime) {

        synchronized (currentThreadLock) {

            try {
                if (!isLogging())
                    currentThreadLock.wait(maxTime);
            }
            catch (InterruptedException e) {
            }
        }

        return isLogging();
    }



	/**
	 * The following method is used to only log error messages to an active
	 * logging agent.
	 *
	 * @param message	-	The message to be logged
	 *		  messageId -	The message ID corresponding to the error message
	 *		  severity	-	The level of severity of the message (e.g. default = DEBUG, 1 = INFORMATION, 2 = WARNING, 3 = SEVERE, 4 = MAX ERROR LEVEL)
	 */
    public void logErrorMessage(String message, String messageId, int severity) {    	
    	delegate.logErrMsg(message, messageId, severity);
    }
}
