/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.adapter;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.ui.views.properties.IPropertyDescriptor;
import org.eclipse.ui.views.properties.IPropertySource;
import org.eclipse.ui.views.properties.PropertyDescriptor;

import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.util.INamedElement;

/**
 * Property source for {@link org.eclipse.hyades.ui.util.INamedElement} instances.
 * 
 * @author marcelop
 * @since 0.0.1 
 */
public class NamedElementPropertySource 
implements IPropertySource
{
	/**
	 * Info category.
	 * @see IPropertySource
	 */
	public final static String PROP_INFO = HyadesUIPlugin.getString("W_INFO");
	
	/**
	 * File category.
	 * @see IPropertySource
	 */
	public final static String PROP_RESOURCE = HyadesUIPlugin.getString("W_RESOURCE");

	/**
	 * Name property constant
	 * @see IPropertySource
	 */
	protected final static String PROP_NAME = HyadesUIPlugin.getString("W_NAME");

	/**
	 * Description property constant
	 * @see IPropertySource
	 */
	protected final static String PROP_DESCRIPTION = HyadesUIPlugin.getString("W_DESCRIPTION");

	/**
	 * The named element that this property source applies to.
	 */
	private INamedElement namedElement;
	
	/**
	 * The property source of the file associated to the named element.
	 */
	private IPropertySource filePropertySource;
	
	/**
	 * Constructor for NamedElementPropertySource
	 * 
	 * <p>This constructor should be used only by subclasses that must to ensure 
	 * that the named element is set after the instantiation.
	 */
	public NamedElementPropertySource()
	{
	}

	/**
	 * Constructor for NamedElementPropertySource
	 * @param namedElement
	 */
	public NamedElementPropertySource(INamedElement namedElement)
	{
		setNamedElement(namedElement);
	}
	
	/**
	 * Sets the named element.
	 * @param namedElement
	 */
	protected void setNamedElement(INamedElement namedElement)
	{
		this.namedElement = namedElement;
		if(namedElement == null)
			return;
			
		IFile file = (IFile)namedElement.getAdapter(IFile.class);
		if(file != null)
			filePropertySource = (IPropertySource)file.getAdapter(IPropertySource.class);
	}
	
	/**
	 * Returns the named element.
	 * @return INamedElement
	 */
	public INamedElement getNamedElement()
	{
		return namedElement;
	}

	/**
	 * @see org.eclipse.ui.views.properties.IPropertySource#getPropertyDescriptors()
	 */
	public IPropertyDescriptor[] getPropertyDescriptors()
	{
		List propertyDescriptors = new ArrayList();
		
		PropertyDescriptor descriptor = new PropertyDescriptor(PROP_NAME, PROP_NAME.toLowerCase());
		descriptor.setCategory(PROP_INFO);
		descriptor.setAlwaysIncompatible(true);
		propertyDescriptors.add(descriptor);

		propertyDescriptors.addAll(Arrays.asList(addPropertyDescriptors()));

		descriptor = new PropertyDescriptor(PROP_DESCRIPTION, PROP_DESCRIPTION.toLowerCase());
		descriptor.setCategory(PROP_INFO);
		descriptor.setAlwaysIncompatible(true);
		propertyDescriptors.add(descriptor);
		
		if(filePropertySource != null)
		{
			IPropertyDescriptor[] filePropertyDescriptors = filePropertySource.getPropertyDescriptors();
			for(int i = 0, maxi = filePropertyDescriptors.length; i < maxi; i++)
			{
				if(filePropertyDescriptors[i] instanceof PropertyDescriptor)
					((PropertyDescriptor)filePropertyDescriptors[i]).setCategory(PROP_RESOURCE);
			}
			propertyDescriptors.addAll(Arrays.asList(filePropertyDescriptors));
		}
					
		return (IPropertyDescriptor[])propertyDescriptors.toArray(new IPropertyDescriptor[propertyDescriptors.size()]);
	}
	
	/**
	 * Adds property descriptors between the name and the description property.
	 * Subclasses may overwrite to provide detailed information.
	 * @return IPropertyDescriptor[]
	 */
	protected IPropertyDescriptor[] addPropertyDescriptors()
	{
		return new IPropertyDescriptor[0];
	}

	/**
	 * @see org.eclipse.ui.views.properties.IPropertySource#getPropertyValue(java.lang.Object)
	 */
	public Object getPropertyValue(Object id)
	{
		if(PROP_NAME.equals(id))
			return namedElement.getName();

		if(PROP_DESCRIPTION.equals(id))
			return namedElement.getDescription();
		
		if(filePropertySource != null)
			return filePropertySource.getPropertyValue(id);
			
		return null;
	}

	/**
	 * @see org.eclipse.ui.views.properties.IPropertySource#isPropertySet(java.lang.Object)
	 */
	public boolean isPropertySet(Object id)
	{
		return false;
	}

	/**
	 * @see org.eclipse.ui.views.properties.IPropertySource#getEditableValue()
	 */
	public Object getEditableValue()
	{
		return null;
	}

	/**
	 * @see org.eclipse.ui.views.properties.IPropertySource#resetPropertyValue(java.lang.Object)
	 */
	public void resetPropertyValue(Object id)
	{

	}

	/**
	 * @see org.eclipse.ui.views.properties.IPropertySource#setPropertyValue(java.lang.Object, java.lang.Object)
	 */
	public void setPropertyValue(Object id, Object value)
	{

	}
}
