/*******************************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.hyades.ui.extension;

import java.util.Collection;
import java.util.List;

import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.viewers.IStructuredSelection;

/**
 * An extension of the base Hyades navigator that provides
 * additional content. This is implemented by clients who
 * wish to contribute item(s) to the Hyades navigator views
 * (e.g. Profiling Monitor, Log Navigator). The built-in
 * Hyades types (e.g.,
 * {@link org.eclipse.hyades.models.hierarchy.TRCAgentProxy})
 * are all EMF objects, but any new types contributed by
 * extensions are not. Instead, they are simply UI
 * representatives (that will probably contain EMF objects).
 * External objects must implement
 * {@link org.eclipse.hyades.ui.extension.INavigatorItem}.
 *
 * @see org.eclipse.hyades.ui.extension.INavigatorItem
 *  
 * @author curtispd
 * @since 3.0
 */
public interface INavigatorContribution {
	
	/**
	 * The context menu group for additional items.
	 */
	public final String GROUP_ADDITIONS = "group.additions";

	/**
	 * The context menu group for creating new items
	 * ("New >")
	 */
	public final String GROUP_NEW = "group.new";

	/**
	 * The context menu group for new wizard shortcut and more generaly for acces the content of the 'New' menu
	 */
	public final String GROUP_SUB_NEW = "group.new.content";

	/**
	 * The context menu group for opening
	 * items ("Open", "Open With >").
	 */
	public final String GROUP_OPEN = "org.eclipse.ui.OpenWithSubMenu";
	
	/**
	 * The context menu group for delete action(s).
	 */
	public final String GROUP_DELETE = "group.delete";

	/**
	 * The context menu group for save action(s).
	 */
	public final String GROUP_SAVE = "group.save";

	/**
	 * The context menu group for importing and exporting.
	 */
	public final String GROUP_IMPORT= "group.import";

	/**
	 * The context menu group for refreshing actions.
	 */
	public final String GROUP_REFRESH = "group.refresh";

	/**
	 * The context menu group for the open properties
	 * action.
	 */
	public final String GROUP_PROPERTIES = "group.properties";
	
	
	/**
	 * Disposes any resources associated with this extension
	 * (e.g. image icons).
	 */
	public void dispose();

	/**
	 * Fills the context menu for the given item. Note that
	 * this is for additions only. The navigator will
	 * automatically add its default items.
	 * 
	 * @param menu the context menu to add actions to.
	 * @param selection the selected items.
	 */
	public void fillContextMenu(IMenuManager menu, IStructuredSelection selection);
	
	/**
	 * Returns the given {@link java.lang.Object}'s children
	 * known by this extension. These items are appended to the
	 * Hyades built-in types (e.g.,
	 * {@link org.eclipse.hyades.models.hierarchy.TRCMonitor}).
	 * Extensions are responsible for maintaining all references
	 * to external (non built-in) items. If the given item is of
	 * type {@link org.eclipse.core.resources.IWorkspaceRoot},
	 * these are the top-level items. The children's types must
	 * be the designated external type,
	 * {@link org.eclipse.hyades.ui.extension.INavigatorItem}.
	 *  
	 * @param item the item whose children we wish to get.
	 * @return the children of the given item.
	 */
	public List getChildren(Object item);
	
	/**
	 * Returns all
	 * {@link org.eclipse.hyades.ui.extension.INavigatorItem}s
	 * known by this extension that require saving. This is
	 * used to determine which items will be saved on exit when
	 * the workbench shuts down, if the user wishes to save
	 * his/her resources.
	 * 
	 * @return all items that need to be saved.
	 */
	public Collection getModifiedItems();
	
	/**
	 * Returns <code>true</code> if the item has children,
	 * otherwise <code>false</code>. This is equivalent to
	 * <code>(getChildren(item).size() > 0)</code>,
	 * but should be more efficient.
	 * 
	 * @return whether or not this item has children.
	 */
	public boolean hasChildren(Object item);
}
