/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.util;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IContentProvider;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Shell;

import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.util.IDisposable;

/**
 * A dialog container for the ChooseElementComposite.
 * 
 * @author marcelop
 * @since 0.0.1
 */
public class ChooseElementDialog 
extends Dialog implements IDisposable, IDoubleClickListener, ISelectionChangedListener
{
	private final static int HEIGHT = 300;
	
	private ChooseElementComposite chooseElementComposite;
	private Object[] elements;
	private IContentProvider contentProvider;
	private ILabelProvider labelProvider;
	private Object initialSelection;
	private Object selection;
	private String selectionWidgetLabel;
	private String descriptionLabel;
	
	private String dialogTitle;
	 
	/**
	 * Constructor for ChooseElementDialog
	 * @param parentShell
	 * @param elements
	 * @param labelProvider
	 * @param contentProvider
	 */
	public ChooseElementDialog(Shell parentShell, Object[] elements, ILabelProvider labelProvider, IContentProvider contentProvider)
	{
		super(parentShell);
		
		setShellStyle(getShellStyle() | SWT.RESIZE);
		this.elements = elements;
		this.labelProvider = labelProvider;
		if(contentProvider == null)
			this.contentProvider = new ArrayContentProvider();
		else
			this.contentProvider = contentProvider;
	}
	
	/**
	 * Opens this dialog. Clients can choose to instantiate this class and open it by
	 * themselves.
	 * 
	 * <p>This is an utility method that opens an instance of this dialog with a
	 * default title and description label.
	 * 
	 * @param parentShell
	 * @param contentProvider
	 * @param labelAndDescriptionProvider
	 * @return Object the table's selected element or null if there was no selection or
	 * if the user canceled the dialog. 
	 */
	public static Object open(Shell parentShell, Object[] elements, ILabelProvider labelProvider)
	{
		return open(parentShell, HyadesUIPlugin.getString("DLG_CHOOSE_TTL"), elements, labelProvider);
	}

	/**
	 * Opens this dialog. Clients can choose to instantiate this class and open it by
	 * themselves.
	 * 
	 * @param parentShell
	 * @param dialogTitle
	 * @param contentProvider
	 * @param labelAndDescriptionProvider
	 * @return Object the table's selected element or null if there was no selection or
	 * if the user canceled the dialog. 
	 */
	public static Object open(Shell parentShell, String dialogTitle, Object[] elements, ILabelProvider labelProvider)
	{
		ChooseElementDialog dialog = new ChooseElementDialog(parentShell, elements, labelProvider, null);
		dialog.setDialogTitle(dialogTitle);
		dialog.setDescriptionLabel(HyadesUIPlugin.getString("LBL_DESC"));
		
		int ret = dialog.open();
		dialog.dispose();
		
		if(ret == IDialogConstants.CANCEL_ID)
			return null;				
		return dialog.getSelection();
	}
	
	/**
	 * @see org.eclipse.hyades.ui.test.util.IDisposable#dispose()
	 */
	public void dispose()
	{
		contentProvider = null;
		labelProvider = null;
		initialSelection = null;
		
		chooseElementComposite.removeSelectionChangedListener(this);
		chooseElementComposite.removeDoubleClickListener(this);
		if(!chooseElementComposite.isDisposed())
			chooseElementComposite.dispose();
		chooseElementComposite = null;
	}

	/**
	 * Sets the object to be selected when this dialog pops up. 
	 * @param selection
	 */
	public void setInitialSelection(Object selection)
	{
		initialSelection = selection;
	}
	
	/**
	 * Sets the text to be presented as the label for the description text
	 * are.  If the dialog was not created and label is <code>null</code> then no 
	 * label is presented.
	 * @param label
	 */
	public void setDescriptionLabel(String label)
	{
		descriptionLabel = label;
	}

	/**
	 * Sets the label of the selection widget.
	 * @param label
	 */
	public void setSelectionWidgetLabel(String label)
	{
		if(chooseElementComposite != null)
			chooseElementComposite.setStructuredViewerLabel(label);
		else
			selectionWidgetLabel = label;	
	}

	/**
	 * Sets the title to be shown in the title area of this dialog.
	 * @param newTitle the title show 
	 */
	public void setDialogTitle(String newTitle)
	{
		if(getShell() == null)
			dialogTitle = newTitle;
		else
			getShell().setText(newTitle);
	}

	/**
	 * Sets the content provided to be used by the table area.  If the content provider is 
	 * null then {@link org.eclipse.jface.viewers.ArrayContentProvider} is used.
	 * @param label
	 */
	public void setContentProvider(IContentProvider contentProvider)
	{
		if(contentProvider != null)
			this.contentProvider = contentProvider;
		else
			this.contentProvider =  new ArrayContentProvider();
	}

	/**
	 * Returns the selection made by the user or null if there was no selection of it
	 * the dialog was canceled.
	 * @return
	 */
	public Object getSelection()
	{
		return selection;
	}
			
	/**
	 * @see org.eclipse.jface.window.Window#createContents(org.eclipse.swt.widgets.Composite)
	 */
	protected Control createContents(Composite parent)
	{
		Control control = super.createContents(parent);
		
		chooseElementComposite.setContentProvider(contentProvider);
		chooseElementComposite.setInput(elements);
		
		if(initialSelection != null)
			chooseElementComposite.setSelection(new StructuredSelection(initialSelection), true, true);
		else
			chooseElementComposite.setSelection(0, true);
		
		if(chooseElementComposite.getSelection().isEmpty())	
			getButton(IDialogConstants.OK_ID).setEnabled(false);
		chooseElementComposite.getControl().setFocus();
						
		return control;
	}

	/**
	 * @see org.eclipse.jface.dialogs.Dialog#createDialogArea(org.eclipse.swt.widgets.Composite)
	 */
	protected Control createDialogArea(Composite parent)
	{
		Composite superComposite = (Composite)super.createDialogArea(parent);
		GridData gridData = GridDataUtil.createFill();
		gridData.heightHint = HEIGHT;
		gridData.widthHint = HEIGHT;
		superComposite.setLayoutData(gridData);
		
		chooseElementComposite = new ChooseElementComposite(superComposite, SWT.NONE);
		chooseElementComposite.setLayoutData(GridDataUtil.createFill());
		chooseElementComposite.createContents(labelProvider);
		chooseElementComposite.addSelectionChangedListener(this);
		chooseElementComposite.addDoubleClickListener(this);
		
		if(descriptionLabel == null)
			descriptionLabel = HyadesUIPlugin.getString("LBL_DESC");
		chooseElementComposite.setDescriptionLabel(descriptionLabel);

		if(selectionWidgetLabel != null)
			chooseElementComposite.setStructuredViewerLabel(selectionWidgetLabel);

		if(dialogTitle != null)
			setDialogTitle(dialogTitle);

		return superComposite;
	}
	
	/**
	 * @see org.eclipse.jface.viewers.IDoubleClickListener#doubleClick(org.eclipse.jface.viewers.DoubleClickEvent)
	 */
	public void doubleClick(DoubleClickEvent event)
	{
		if(chooseElementComposite.fired(event))
		{
			selection = ((IStructuredSelection)event.getSelection()).getFirstElement();
			okPressed();
		}
	}

	/**
	 * @see org.eclipse.jface.viewers.ISelectionChangedListener#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
	 */
	public void selectionChanged(SelectionChangedEvent event)
	{
		if(chooseElementComposite.fired(event))
		{
			getButton(IDialogConstants.OK_ID).setEnabled(true);
			selection = ((IStructuredSelection)event.getSelection()).getFirstElement();
		}
	}
}
