package org.eclipse.hyades.logging.events.cbe.impl.tests;

import junit.framework.TestCase;

import org.eclipse.hyades.logging.events.cbe.ContextDataElement;
import org.eclipse.hyades.logging.events.cbe.EventFactory;
import org.eclipse.hyades.logging.events.cbe.ValidationException;

/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * 
 * @author Denilson Nastacio
 * @version 1.0.1
 * @since 1.0.1
 */
public class ContextDataElementImplTest extends TestCase {
    ContextDataElement cde = null;

    /**
     * Constructor for ContextDataElementImplTest.
     * @param arg0
     */
    public ContextDataElementImplTest(String arg0) {
        super(arg0);
    }

    protected void setUp() throws Exception {
        super.setUp();

        cde = EventFactory.eINSTANCE.createContextDataElement();
    }

    public void testValidate1() throws ValidationException {
        cde = EventFactory.eINSTANCE.createContextDataElement();
        cde.setName("CDE test");
        cde.setType("Type");
        cde.setContextValue("Value");
        cde.validate();

        cde = EventFactory.eINSTANCE.createContextDataElement();
        cde.setName("CDE test");
        cde.setType("Type");
        cde.setContextId(Util.getAlphaString(32));
        cde.validate();
    }

    public void testRequiredAttributes() throws ValidationException {
        cde = EventFactory.eINSTANCE.createContextDataElement();
        Util.assertMissingAttributes(cde, "all attributes missing");

        cde.setName("CDE test");
        cde.setType("Type");
        try {
            cde.validate();
            fail("Missing required value contextId or contextValue " + cde.getClass().getName());
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0204E", ve);
        }

        // Now test that contextId and contextValue are mutually exclusive
        cde.setContextValue("Value");
        cde.setContextId(Util.getAlphaString(32));
        try {
            cde.validate();
            fail("Mutually exclusive attributes contextId and contextValue " + cde.getClass().getName());
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0204E", ve);
        }
    }

    public void testInvalidAttributes() throws ValidationException {
        cde = EventFactory.eINSTANCE.createContextDataElement();
        cde.setName(Util.getAlphaString(65));
        cde.setType("Type");
        cde.setContextValue("Value");
        //cde.setContextId(Util.getAlphaString(32));
        Util.assertInvalidBoundaries(cde, "name");

        cde.setName("CDE test");
        cde.setType(Util.getAlphaString(65));
        Util.assertInvalidBoundaries(cde, "type");

        cde.setType("Type");
        cde.setContextValue(Util.getAlphaString(1025));
        Util.assertInvalidBoundaries(cde, "value");

        cde = EventFactory.eINSTANCE.createContextDataElement();
        cde.setName("CDE test");
        cde.setType("Type");
        cde.setContextId(Util.getAlphaString(15));
        try {
            cde.validate();
            fail("id is too short in " + cde.getClass().getName());
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0208E", ve);
        }

        cde.setContextId(Util.getAlphaString(65));
        try {
            cde.validate();
            fail("id is too long in " + cde.getClass().getName());
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0208E", ve);
        }
    }

    public void testToString() throws ValidationException {
        String name = "my name";
        String type = "my type";
        String value = "my value";
        String id = "my id";

        cde = EventFactory.eINSTANCE.createContextDataElement();
        //System.out.println(cde);
        assertNotNull("empty cde", cde.toString());

        cde.setName(name);
        cde.setType(type);
        cde.setContextValue(value);
        cde.setContextId(id);
        //System.out.println(cde);
        assertNotNull("cde with soma data filled in", cde.toString());
    }

    public void testGetSet() throws ValidationException {
        String name = "my name";
        String type = "my type";
        String value = "my value";
        String id = "my id";

        cde = EventFactory.eINSTANCE.createContextDataElement();
        cde.setName(name);
        cde.setType(type);
        cde.setContextValue(value);
        cde.setContextId(id);

        assertEquals(name, cde.getName());
        assertEquals(type, cde.getType());
        assertEquals(value, cde.getContextValue());
        assertEquals(id, cde.getContextId());
    }

    /**
     * Tests the <code>equals</code> operator.
     */
    public void testEquals() {
        ContextDataElement cde1 = EventFactory.eINSTANCE.createContextDataElement();
        cde1.setName("cde test");
        cde1.setType("Type");
        cde1.setContextId("my id");

        ContextDataElement cde2 = EventFactory.eINSTANCE.createContextDataElement();
        cde2.setName("cde test");
        cde2.setType("Type");
        cde2.setContextId("my other id");

        ContextDataElement cde3 = (ContextDataElement) Util.clone(cde1);
        assertEquals(cde1, cde3);

        ContextDataElement cde4 = (ContextDataElement) Util.clone(cde2);
        assertEquals(cde2, cde4);

        assertFalse(cde1.equals(cde2));
    }
}
