package org.eclipse.hyades.logging.events.cbe.util;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.hyades.logging.core.XmlUtility;
import org.eclipse.hyades.logging.events.cbe.AssociationEngine;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.EventFactory;
import org.eclipse.hyades.logging.events.cbe.EventPackage;
import org.eclipse.hyades.logging.events.cbe.OtherSituation;
import org.eclipse.hyades.logging.events.cbe.Situation;
import org.xml.sax.Attributes;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.helpers.DefaultHandler;

/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * The <code>SAXEventHandler</code> class defines a utility object used for
 * de-serializing <code>CommonBaseEvent</code> XML documents and fragments
 * using a Simple API for XML (SAX) parser.
 * <p>
 * 
 * @author Denilson Nastacio
 * @author Paul E. Slauenwhite
 * @version 1.0.1
 * @since 1.0.1
 * @see org.xml.sax.helpers.DefaultHandler
 */
public final class SAXEventHandler extends DefaultHandler {

    /**
     * Map of element names and respective structural feature.
     * <p>
     * The key is a <code>String</code> object representing the element name
     * and the value is a <code>EClass</code> object required to create the
     * Java instance through the <code>EventFactory</code> object.
     */
    private HashMap elementMap = new HashMap();

    /**
     * Map of XML element names and its EMF structural features.
     * <p>
     * The key is a <code>String</code> object representing the element name
     * and the value is another <code>HashMap</code> representing the mapping
     * between the feature name and the feature itself.
     * <p>
     * For this map, the key is a <code>String</code> object representing the
     * feature name and the value is a <code>EStructuralFeature</code> object
     * required to create the Java instance through the
     * <code>EventFactory</code> object.
     */
    private HashMap featureMap = new HashMap();

    /**
     * Current document column position.
     */
    private int column = 0;

    /**
     * Current document line position.
     */
    private int line = 0;

    /**
     * Buffer for any element XML fragments (e.g. CommonBaseEvent.otherData or
     * OtherSituation.anyData).
     */
    private StringBuffer anyElementXMLFragment = null;

    /**
     * Buffer for Parsed Character DATA (PCDATA) (e.g.
     * ExtendedDataElement.values) since some SAX parsers do not return all
     * contiguous PCDATA in a single chunk.
     */
    private StringBuffer charactersBuffer = null;

    /**
     * Buffer for name space mapping attributes.
     */
    private StringBuffer nameSpaceAttribute = null;

    /**
     * Current nesting depth of any element XML fragments (e.g.
     * CommonBaseEvent.otherData or OtherSituation.anyData).
     */
    private int anyElementNestingDepth = 0;

    /**
     * List of parsed CommonBaseEvents.
     */
    private List commonBaseEvents = null;

    /**
     * List of parsed AssociationEngines.
     */
    private List associationEngines = null;

    /**
     * Stack of event objects currently parsed.
     */
    private List stack = null;

    /**
     * Situation type needs special handling during parsing.
     */
    private final static String SITUATION_TYPE_CLASS = EventPackage.eINSTANCE.getSituationType().getName();

    /**
     * CommonBaseEvent needs special handling during parsing.
     */
    private final static String COMMON_BASE_EVENT_CLASS = EventPackage.eINSTANCE.getCommonBaseEvent().getName();

    /**
     * OtherSituation needs special handling during parsing.
     */
    private final static String OTHER_SITUATION_CLASS = EventPackage.eINSTANCE.getOtherSituation().getName();

    /**
     * AssociationEngine needs special handling during parsing.
     */
    private final static String ASSOCIATION_ENGINE_CLASS = EventPackage.eINSTANCE.getAssociationEngine().getName();

    /**
     * No argument constructor.
     */
    public SAXEventHandler() {
        init();
    }

    /**
     * Initialized the SAX parser to a newly constructed state.
     */
    public void init() {

        reset();

        associationEngines = new ArrayList();

        commonBaseEvents = new ArrayList();
    }

    /**
     * Resets the SAX parser for parsing the next element.
     */
    private void reset() {

        column = 0;

        line = 0;

        anyElementNestingDepth = 0;

        stack = new ArrayList();

        anyElementXMLFragment = new StringBuffer();

        charactersBuffer = new StringBuffer();

        nameSpaceAttribute = null;

        Iterator iterator = EventPackage.eINSTANCE.eContents().iterator();

        while (iterator.hasNext()) {

            EClass eClass = (EClass) iterator.next();

            elementMap.put(eClass.getName(), eClass);

            List features = eClass.getEAllStructuralFeatures();

            for (int counter = 0; counter < features.size(); counter++) {

                EStructuralFeature structuralFeature = (EStructuralFeature) features.get(counter);

                //Remove the 'any' structural feature for special processing:
                if (!structuralFeature.getName().equals("any")) {

                    HashMap featureSubMap = (HashMap) featureMap.get(eClass.getName());

                    if (featureSubMap == null) {
                        featureSubMap = new HashMap();
                        featureMap.put(eClass.getName(), featureSubMap);
                    }

                    featureSubMap.put(structuralFeature.getName(), structuralFeature);
                }
            }
        }
    }

    /**
     * Returns the list of <code>CommonBaseEvent</code> objects created when
     * de-serializing a Common Base Event XML document or fragment.
     * <p>
     * This API will return null if no <code>CommonBaseEvent</code> objects
     * have been parsed.
     * <p>
     * 
     * @return List of <code>CommonBaseEvent</code> objects created when
     *         de-serializing a Common Base Event XML document or fragment,
     *         otherwise null if no <code>CommonBaseEvent</code> objects have
     *         been parsed.
     */
    public CommonBaseEvent[] getCommonBaseEvents() {

        if ((commonBaseEvents != null) && (!commonBaseEvents.isEmpty())) { return (((CommonBaseEvent[]) (commonBaseEvents.toArray(new CommonBaseEvent[commonBaseEvents.size()])))); }

        return null;
    }

    /**
     * Returns the list of <code>AssociationEngine</code> objects created when
     * de-serializing a Common Base Event XML document.
     * <p>
     * This API will return null if no <code>AssociationEngine</code> objects
     * have been parsed.
     * <p>
     * 
     * @return List of <code>AssociationEngine</code> objects created when
     *         de-serializing a Common Base Event XML document or fragment,
     *         otherwise null if no <code>AssociationEngine</code> objects
     *         have been parsed.
     */
    public AssociationEngine[] getAssociationEngines() {

        if ((associationEngines != null) && (!associationEngines.isEmpty())) { return (((AssociationEngine[]) (associationEngines.toArray(new AssociationEngine[associationEngines.size()])))); }

        return null;
    }

    /**
     * @see org.xml.sax.ContentHandler#setDocumentLocator(org.xml.sax.Locator)
     */
    public void setDocumentLocator(Locator loc) {

        super.setDocumentLocator(loc);

        column = loc.getColumnNumber();
        line = loc.getLineNumber();
    }

    /**
     * @see org.xml.sax.ContentHandler#characters(char[], int, int)
     */
    public void characters(char ch[], int start, int length) throws SAXException {

        super.characters(ch, start, length);

        charactersBuffer.append(new String(ch, start, length));
    }

    /**
     * @see org.xml.sax.ContentHandler#startPrefixMapping(java.lang.String,
     *      java.lang.String)
     */
    public void startPrefixMapping(String prefix, String uri) throws SAXException {

        super.startPrefixMapping(prefix, uri);

        //NOTE: Xerces fires namespace mapping events for all elements.
        //Do not capture namespace mapping events for the 'xmlns' and 'xml'
        // namespace:
        if ((!prefix.equals("xmlns")) && (!prefix.equals("xml"))) {

            nameSpaceAttribute = new StringBuffer("xmlns");

            if ((prefix != null) && (prefix.trim().length() > 0)) {
                nameSpaceAttribute.append(':');
                nameSpaceAttribute.append(prefix);
            }

            nameSpaceAttribute.append("=\"");
            nameSpaceAttribute.append(uri);
            nameSpaceAttribute.append('\"');
        }
    }

    /**
     * @see org.xml.sax.ContentHandler#startElement(java.lang.String,
     *      java.lang.String, java.lang.String, org.xml.sax.Attributes)
     */
    public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException {

        super.startElement(uri, localName, qName, attributes);

        String elementName = getLocalElementName(localName, qName);

        if (!isRestrictedElementName(elementName)) {

            //The EMF class and object for this XML element:
            EObject emfObj = null;
            EClass emfObjClass = null;

            //If the stack is empty, this element should only be either a
            // <CommonBaseEvent> or <AssociationEngine> element:
            if (stack.isEmpty()) {

                if ((!elementName.equals(COMMON_BASE_EVENT_CLASS)) && (!elementName.equals(ASSOCIATION_ENGINE_CLASS))) { 
                    throw new SAXException(EventHelpers.getString(EventMessages.LOG_EVENT_SAX_PARSER_INCORRECT_XML_ELEMENT_EXC_, new String[] { COMMON_BASE_EVENT_CLASS, elementName, String.valueOf(line), String.valueOf(column)})); 
                }

                emfObjClass = (EClass) elementMap.get(elementName);
                emfObj = EventFactory.eINSTANCE.create(emfObjClass);

                stack.add(emfObj);
            } else {

                EObject container = (EObject) stack.get(stack.size() - 1);
                String containerName = container.eClass().getName();

                HashMap featureSubMap = (HashMap) featureMap.get(containerName);
                EStructuralFeature objStructuralFeature = (EStructuralFeature) featureSubMap.get(elementName);

                //This element is assumed to be an any element if no structural
                // feature was found:
                if (objStructuralFeature == null) {

                    if ((containerName.equals(COMMON_BASE_EVENT_CLASS)) || (containerName.equals(OTHER_SITUATION_CLASS))) {

                        if (anyElementNestingDepth > 0) {

                            String anyElementCharacters = charactersBuffer.toString();

                            //Do not persist white space (e.g. formatting
                            // characters) between elements:
                            if (anyElementCharacters.trim().length() > 0) {
                                anyElementXMLFragment.append(XmlUtility.normalize(anyElementCharacters));
                            }
                        }

                        anyElementNestingDepth++;

                        anyElementXMLFragment.append('<');
                        anyElementXMLFragment.append(getQualifiedElementName(uri, localName, qName));

                        //Add the name space mapping attribute:
                        if (nameSpaceAttribute != null) {

                            anyElementXMLFragment.append(' ');
                            anyElementXMLFragment.append(nameSpaceAttribute.toString());
                        }

                        for (int counter = 0; counter < attributes.getLength(); counter++) {

                            anyElementXMLFragment.append(' ');
                            anyElementXMLFragment.append(getQualifiedElementName(attributes.getURI(counter), attributes.getLocalName(counter), attributes.getQName(counter)));
                            anyElementXMLFragment.append("=\"");
                            anyElementXMLFragment.append(XmlUtility.normalize(attributes.getValue(counter)));
                            anyElementXMLFragment.append("\"");
                        }

                        anyElementXMLFragment.append('>');
                    } else {
                        throw new SAXException(EventHelpers.getString(EventMessages.LOG_EVENT_SAX_PARSER_UNEXPECTED_XML_ELEMENT_EXC_, new String[] { elementName, String.valueOf(line), String.valueOf(column)}));
                    }
                } else {

                    //Special case the situation type:
                    String className = objStructuralFeature.getEType().getName();

                    if (className.equals(SITUATION_TYPE_CLASS)) {

                        className = ((Situation) (container)).getCategoryName();
                        
                        //The situation's category is required to identify the class of situation type:
                        if((className == null) || (className.trim().length() == 0)){
                            throw new SAXException(EventHelpers.getString(EventMessages.LOG_EVENT_SAX_PARSER_INVALID_XML_ATTRIBUTE_EXC_, new String[] {"categoryName", "<empty>", "0", "0"}));
                        }
                    }

                    //Create instance:
                    if (className.equals("EString")) {

                        if (objStructuralFeature.isMany()) {
                            stack.add(((List) (container.eGet(objStructuralFeature))));
                        } else {
                            //This structural feature contains PCDATA to be set
                            // in the characters() API:
                            stack.add(objStructuralFeature);
                        }
                    } else {

                        emfObjClass = (EClass) elementMap.get(className);
                        emfObj = EventFactory.eINSTANCE.create(emfObjClass);

                        //Setting the object inside the container object:
                        if (objStructuralFeature.isMany()) {
                            ((List) (container.eGet(objStructuralFeature))).add(emfObj);
                        } else {
                            container.eSet(objStructuralFeature, emfObj);
                        }

                        stack.add(emfObj);
                    }
                }
            }

            //Fill instance attributes:
            if (emfObj != null) {

                HashMap featureSubMap = (HashMap) featureMap.get(emfObjClass.getName());
                EAttribute feature = null;
                String attributeTypeName = null;

                for (int counter = 0; counter < attributes.getLength(); counter++) {

                    //Find structural feature:
                    feature = (EAttribute) featureSubMap.get(getLocalElementName(attributes.getLocalName(counter), attributes.getQName(counter)));

                    //The attribute could not be in the featureSubMap because
                    //it is an xmlns declaration or some other xml directive
                    // (e.g. <SituationType ... xsi:type="..."/>):
                    if (feature != null) {

                        attributeTypeName = feature.getEAttributeType().getName();

                        if (attributeTypeName.equals("ELong")) {

                            try {
                                emfObj.eSet(feature, Long.valueOf(attributes.getValue(counter)));
                            } catch (NumberFormatException n) {

                                //If the attribute value is not white space
                                // (e.g. empty), throw an exception:
                                if (attributes.getValue(counter).trim().length() > 0) { throw new SAXException(EventHelpers.getString(EventMessages.LOG_EVENT_SAX_PARSER_INVALID_XML_ATTRIBUTE_EXC_, new String[]{getQualifiedElementName(attributes.getURI(counter), attributes.getLocalName(counter), attributes.getQName(counter)), attributes.getValue(counter), String.valueOf(line), String.valueOf(column)})); }
                            }
                        } else if (attributeTypeName.equals("EShort")) {

                            try {
                                emfObj.eSet(feature, Short.valueOf(attributes.getValue(counter)));
                            } catch (NumberFormatException n) {

                                //If the attribute value is not white space
                                // (e.g. empty), throw an exception:
                                if (attributes.getValue(counter).trim().length() > 0) { throw new SAXException(EventHelpers.getString(EventMessages.LOG_EVENT_SAX_PARSER_INVALID_XML_ATTRIBUTE_EXC_, new String[]{getQualifiedElementName(attributes.getURI(counter), attributes.getLocalName(counter), attributes.getQName(counter)), attributes.getValue(counter), String.valueOf(line), String.valueOf(column)})); }
                            }
                        } else {
                            emfObj.eSet(feature, attributes.getValue(counter));
                        }
                    }
                }
            }

            //Empty the buffer of characters (e.g. white space):
            charactersBuffer = new StringBuffer();

            //Release the name space mapping attribute since
            // #startPrefixMapping(String, String) is <i>always</i> called
            // before #startElement(String, String, String, Attributes):
            nameSpaceAttribute = null;
        }
    }

    /**
     * @see org.xml.sax.ContentHandler#endElement(java.lang.String,
     *      java.lang.String, java.lang.String)
     */
    public void endElement(String uri, String localName, String qName) throws SAXException {

        super.endElement(uri, localName, qName);

        String elementName = getLocalElementName(localName, qName);

        if (!isRestrictedElementName(elementName)) {

            if (stack.isEmpty()) {
                throw new SAXException(EventHelpers.getString(EventMessages.LOG_EVENT_SAX_PARSER_UNEXPECTED_XML_ELEMENT_EXC_, new String[] { elementName, String.valueOf(line), String.valueOf(column)}));
            } else {

                if (anyElementNestingDepth > 0) {

                    String anyElementCharacters = charactersBuffer.toString();

                    //Do not persist white space (e.g. formatting characters)
                    // between elements:
                    if (anyElementCharacters.trim().length() > 0) {
                        anyElementXMLFragment.append(XmlUtility.normalize(anyElementCharacters));
                    }

                    anyElementXMLFragment.append("</");
                    anyElementXMLFragment.append(getQualifiedElementName(uri, localName, qName));
                    anyElementXMLFragment.append('>');

                    if (--anyElementNestingDepth == 0) {

                        Object object = stack.get(stack.size() - 1);

                        if (object instanceof CommonBaseEvent) {
                            ((CommonBaseEvent) (object)).addAny(anyElementXMLFragment.toString());
                        } else if (object instanceof OtherSituation) {

                            OtherSituation otherSituation = ((OtherSituation) (object));
                            String currentAnyElement = otherSituation.getAny();

                            if ((currentAnyElement != null) && (currentAnyElement.trim().length() > 0)) {
                                otherSituation.setAny(currentAnyElement.concat(anyElementXMLFragment.toString()));
                            } else {
                                otherSituation.setAny(anyElementXMLFragment.toString());
                            }
                        }

                        anyElementXMLFragment = new StringBuffer();
                    }
                } else {

                    Object object = stack.remove(stack.size() - 1);

                    if (object instanceof CommonBaseEvent) {

                        String anyElementCharacters = charactersBuffer.toString();
                        
                        //Do not persist white space (e.g. formatting characters)
                        // between elements:
                        if (anyElementCharacters.trim().length() > 0) {
                            ((CommonBaseEvent) (object)).addAny(XmlUtility.normalize(anyElementCharacters));
                        }
                        
                        commonBaseEvents.add(((CommonBaseEvent) (object)));

                        reset();

                    }
					else if (object instanceof OtherSituation) {

					    String anyElementCharacters = charactersBuffer.toString();
                        
                        //Do not persist white space (e.g. formatting characters)
                        // between elements:
                        if (anyElementCharacters.trim().length() > 0) {

                            OtherSituation otherSituation = ((OtherSituation) (object));
                            String currentAnyElement = otherSituation.getAny();

                            if ((currentAnyElement != null) && (currentAnyElement.trim().length() > 0)) {
                                otherSituation.setAny(currentAnyElement.concat(XmlUtility.normalize(anyElementCharacters)));
                            } 
                            else {
                                otherSituation.setAny(XmlUtility.normalize(anyElementCharacters));
                            }
				        }					
					}
                    //NOTE: Must check for <AssociationEngine .../> elements
                    // since the object may be an <AssociationEngineInfo .../>
                    // element:
                    else if ((elementName.equals(ASSOCIATION_ENGINE_CLASS)) && (object instanceof AssociationEngine)) {

                        associationEngines.add((AssociationEngine) object);

                        reset();
                    }
                    //Persist PCDATA:
                    else{

                        if (stack.size() > 1) {

                            if (object instanceof List) {
                                ((List) (object)).add(charactersBuffer.toString());
                            }

                            if (object instanceof EStructuralFeature) {
                                ((EObject) (stack.get(stack.size() - 1))).eSet(((EStructuralFeature) (object)), charactersBuffer.toString());
                            }
                        }
                    }
                }
            }

            //Empty the buffer of characters (e.g. PCDATA):
            charactersBuffer = new StringBuffer();
        }
    }

    public void warning(SAXParseException exception) throws SAXException {
        throw exception;
    }

    public void error(SAXParseException exception) throws SAXException {
        throw exception;
    }

    public void fatalError(SAXParseException exception) throws SAXException {
        throw exception;
    }

    private String getLocalElementName(String localName, String qName) {

        if ((localName == null) || (localName.trim().length() == 0)) {

            if (qName != null) {

                int lastColonIndex = qName.lastIndexOf(':');

                if (lastColonIndex != -1) {
                    return (qName.substring(lastColonIndex + 1).trim());
                } else {
                    return (qName.trim());
                }
            } else {
                return (qName.trim());
            }
        }

        return (localName.trim());
    }

    private String getQualifiedElementName(String uri, String localName, String qName) {

        if ((qName == null) || (qName.trim().length() == 0)) {

            if (localName != null) {

                if ((uri != null) && (uri.trim().length() > 0)) {
                    return (uri.concat(":").concat(localName));
                } else {
                    return (localName.trim());
                }
            } else {
                return (localName.trim());
            }
        }

        return (qName.trim());
    }

    private boolean isRestrictedElementName(String elementName) {

        if (elementName != null) {

            String trimmedElementName = elementName.trim();

            return (trimmedElementName.equalsIgnoreCase("CommonBaseEvents")) || (trimmedElementName.equalsIgnoreCase("TemplateEvent"));
        }

        return false;
    }
}

