/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.uml2sd.trace.loaders;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.models.trace.TRCFullMethodInvocation;
import org.eclipse.hyades.models.trace.TRCThread;
import org.eclipse.hyades.trace.internal.ui.PDProjectExplorer;
import org.eclipse.hyades.trace.ui.HyadesUtil;
import org.eclipse.hyades.trace.ui.IProfileEventListener;
import org.eclipse.hyades.trace.ui.ProfileEvent;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.uml2sd.trace.TraceSDPlugin;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.ContextIds;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.DateComparator;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceCallStack;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceInteractionUpdate;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceInteractionUtils;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceLifelineDraft;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceProcesses;
import org.eclipse.hyades.uml2sd.trace.selection.IDateSelection;
import org.eclipse.hyades.uml2sd.ui.actions.provider.ISDFindProvider;
import org.eclipse.hyades.uml2sd.ui.actions.widgets.Criteria;
import org.eclipse.hyades.uml2sd.ui.core.BasicExecutionOccurrence;
import org.eclipse.hyades.uml2sd.ui.core.Frame;
import org.eclipse.hyades.uml2sd.ui.core.GraphNode;
import org.eclipse.hyades.uml2sd.ui.load.BackgroundLoader;
import org.eclipse.hyades.uml2sd.ui.load.IUml2SDLoader;
import org.eclipse.hyades.uml2sd.ui.view.SDView;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.help.WorkbenchHelp;

/**
 * The abstract trace loader.
 * It is based on the parsing of MethodInvocations from the model.
 * Lifelines attribution is made by the concrete classes.
 */
public abstract class BaseTraceInteractions
	implements IUml2SDLoader, ISelectionListener, ISDFindProvider, IProfileEventListener {

	/**
	 * With this we know how to implement IUML2SDLoader.setViewer()
	 */
	protected SDView view;
	
	/**
	 * The frame containing the current page
	 */
	protected Frame frame;
	
	protected TraceProcesses currentProcesses;

	/**
	 * Implementation of IUml2SDLoader
	 */
	public void setViewer(SDView view_) {
		view = view_;
	    //listen to selection from other part.
	    view.getSite().getWorkbenchWindow().getSelectionService().addSelectionListener(this);
	    view.setEnableAction("org.eclipse.hyades.uml2sd.ui.actions.ShowStatTable",true);//$NON-NLS-1$
		UIPlugin.getDefault().addProfileEventListener(this);		
		view.setSDFindProvider(this);
	    onSetViewer();
		WorkbenchHelp.setHelp(this.view.getSDWidget(), ContextIds.INTERACTION_VIEW);
		parseModelBackground();
	}
	
	abstract public void onSetViewer();
	
	public void aboutToBeReplaced() {
		view.setEnableAction("org.eclipse.hyades.uml2sd.ui.actions.ShowStatTable",false);//$NON-NLS-1$
		UIPlugin.getDefault().removeSelectionListener(this);
		UIPlugin.getDefault().removeProfileEventListener(this);		
		clearInternals();
		view = null;
	}
	
	protected EObject currentMofObjectFromPDProjectExplorer;
	private Cursor waitCursor;
	
	/**
	 * 
	 */
	protected void clearInternals() {
		currentMofObjectFromPDProjectExplorer = null;
		currentProcesses = null;
		frame = null;
		if (waitCursor != null && !waitCursor.isDisposed()) {
			waitCursor.dispose();
		}
		waitCursor = null;
		clearFindInternals();
	}
	
	/**
	 * 
	 * @todo Generated comment
	 */
	protected void clearFindInternals() {
		findResults = null;
		findCriteria = null;
		currentFindIndex = 0;
	}

	/**
	 * Let's parse the model for the translation
	 * @param eObject
	 */
	protected synchronized void parseModel(EObject eObject) {
		long start = 0;
		if (TraceSDPlugin.debugEvents) {
			start = System.currentTimeMillis();
		}
		if (TraceSDPlugin.debugEvents) {
			TraceSDPlugin.debugTraceEvents("Start parseModel()"); //$NON-NLS-1$
		}
		clearInternals();
		currentMofObjectFromPDProjectExplorer = eObject;
		createFrame();
		if (currentProcesses == null) {
			if (view != null && !view.getSDWidget().isDisposed()) {
				setFrameName();
				view.setFrameSync(frame);
			}
			if (TraceSDPlugin.debugEvents) {
				TraceSDPlugin.debugTraceEvents("Abort parseModel()"); //$NON-NLS-1$
			}
			return;
		}
		computeModel();
		try {
			fillPage(new TraceInteractionUpdate());
			if (view != null && !view.getSDWidget().isDisposed() && frame != null && !isCurrentLoadCanceled()) {
				setFrameName();
				view.setFrameSync(frame);
			}
		} catch (Throwable e) {
			e.printStackTrace();
		}
		if (TraceSDPlugin.debugEvents) {
			if (isCurrentLoadCanceled()) {
				TraceSDPlugin.debugTraceEvents("Cancelled parseModel() after "+(System.currentTimeMillis()-start)+"ms"); //$NON-NLS-1$ //$NON-NLS-2$
			} else {
				TraceSDPlugin.debugTraceEvents("End parseModel() after "+(System.currentTimeMillis()-start)+"ms"); //$NON-NLS-1$ //$NON-NLS-2$
			}
		}
	}
	
	abstract protected void computeModel();
	
	abstract protected void updateSD(TraceInteractionUpdate update);
	
	/**
	 * monitor is the IProgressMonitor to be checked not cancelled during long tasks
	 */
	private IProgressMonitor monitor;
	
	protected boolean isCurrentLoadCanceled() {
		return monitor != null && monitor.isCanceled();
	}
	
	/**
	 * @param monitor_
	 * @param taskName
	 */
	private void beforeLongTask(IProgressMonitor monitor_, String taskName) {
		if (view == null || view.getSDWidget().isDisposed()) {
			monitor_.setCanceled(true);
			return;
		}
		view.toggleWaitCursorSync(true);
		monitor = monitor_;
		monitor.beginTask(taskName, 100/*???*/);
	}
	
	/**
	 * 
	 */
	private void afterLongTask() {
		monitor.done();
		monitor = null;
		if (view != null && !view.getSDWidget().isDisposed()) {
			view.toggleWaitCursorSync(false);
		}
	}
	
	/**
	 * parseModel() in a background task
	 */
	protected void parseModelBackground() {
		final EObject eObject = HyadesUtil.getMofObject(); // Must be taken in the main loop thread
		if (eObject == null) {
			return;
		}
		BackgroundLoader.getInstance().newTask(new IRunnableWithProgress() {
			public void run(IProgressMonitor monitor_) throws InterruptedException {
				beforeLongTask(monitor_, TraceSDPlugin.getResourceString("TASK_NAME_CREATING_SD")); //$NON-NLS-1$
				if (!isCurrentLoadCanceled()) {
					parseModel(eObject);
				}
				afterLongTask();
			}
		});
	}
	
	/**
	 * updateSD() in a background task
	 */
	protected void updateSDBackground(final TraceInteractionUpdate update) {
		BackgroundLoader.getInstance().newTask(new IRunnableWithProgress() {
			public void run(IProgressMonitor monitor_) throws InterruptedException {
				beforeLongTask(monitor_, TraceSDPlugin.getResourceString("TASK_NAME_UPDATING_SD")); //$NON-NLS-1$
				if (!isCurrentLoadCanceled()) {
					updateSD(update);
				}
				afterLongTask();
			}
		});
	}
	
	/**
	 * @param eObject
	 * @return
	 */
	protected void createFrame() {
		if (currentProcesses == null) {
			currentProcesses = TraceProcesses.getTraceProcesses(currentMofObjectFromPDProjectExplorer, monitor);
		}
		frame = new Frame();
		if (currentProcesses == null) {
			frame.setName(TraceSDPlugin.getResourceString("STR_NO_PROCESSES_HERE")); //$NON-NLS-1$
		} else {
			frame.setTimeUnitName(TraceSDPlugin.getResourceString("STR_TIME_UNIT_IS_SECOND")); //$NON-NLS-1$
		}
	}

	/**
	 * @param process
	 */
	abstract public void setFrameName();

	/**
	 *
	 */
	abstract public void fillPage(TraceInteractionUpdate update);
	
	protected HashMap traceThreadMap;
	protected ArrayList traceThreadList;

	/**
	 * @param eThread
	 * @return
	 */
	protected TraceCallStack getTraceCallStack(TRCThread eThread) {
		TraceCallStack ret = (TraceCallStack)traceThreadMap.get(eThread);
		if (ret == null) {
			ret = new TraceCallStack(this, eThread);
			traceThreadMap.put(eThread, ret);
			traceThreadList.add(ret);
		}
		return ret;
	}
	
	/**
	 * @param tcs
	 */
	public void sortTraceThreadList(TraceCallStack tcs) {
		traceThreadList.remove(tcs);
		for (int i = 0; i < traceThreadList.size(); i++) {
			TraceCallStack tcs2 = (TraceCallStack)traceThreadList.get(i);
			if (tcs.getCurrentExitTime() < tcs2.getCurrentExitTime()) {
				traceThreadList.add(i, tcs);
				return;
			}
		}
		traceThreadList.add(tcs);
	}
	
	/**
	 * @return
	 */
	protected double getTraceThreadListFirstTime() {
		double ret = (traceThreadList != null && traceThreadList.size() > 0) ?
				((TraceCallStack)traceThreadList.get(0)).getCurrentExitTime() :
				Double.MAX_VALUE;
		return ret;
	}
	
	/**
	 * @return
	 */
	protected long getTraceThreadListFirstTicket() {
		long ret = (traceThreadList != null && traceThreadList.size() > 0) ?
				((TraceCallStack)traceThreadList.get(0)).getCurrentTicket() :
				Long.MAX_VALUE;
		return ret;
	}
	
	/**
	 * Place to create the execution occurence for the lifeline
	 * @param current
	 */
	protected void setExecutionOccurence(TraceLifelineDraft draft) {
		if (draft != null) {
			if (draft.getStart() == Integer.MAX_VALUE) {
				draft.setStart(0);
			}
			BasicExecutionOccurrence occ = new BasicExecutionOccurrence();
			occ.setStartOccurrence(draft.getStart());
			occ.setEndOccurrence(draft.getEnd());
			draft.getLifeline().addExecution(occ);
		}
	}

	/**
	 * Get the lifeline to be used for this message
	 * @param mi
	 * @return
	 */
	protected abstract EObject getLifelineEObjectFromMethodInvocation(TRCFullMethodInvocation mi);

	/**
	 * Get the name of the class
	 * @param mi
	 * @return
	 */
	protected abstract String getLifeLineTitle(EObject to, boolean long_);

	/**
	 * Get the category of the lifeline corresponding to this message
	 * @param mi
	 * @return
	 */
	protected abstract int getLifeLineCategory(EObject to);

	
	/**
	 * Implementation of ISelectionListener
	 */
	public void selectionChanged(IWorkbenchPart part, ISelection selection) {
		if (TraceSDPlugin.debugEvents) {
			TraceSDPlugin.debugTraceEvents("selectionChanged("+part+")="+selection); //$NON-NLS-1$ //$NON-NLS-2$
		}
	    if( part instanceof PDProjectExplorer ) {
			if (!((PDProjectExplorer)part).isLinkingEnabled()) {
				return;
			}
			if (view != null &&
				view.getSDWidget() != null &&
				!view.getSDWidget().isDisposed()) {
				EObject mofObject = HyadesUtil.getMofObject();
				if (mofObject instanceof TRCProcessProxy) {
					List agents = TraceInteractionUtils.getProfileAgents((TRCProcessProxy)mofObject);
					if (agents != null && agents.size() == 1) {
						mofObject = (EObject)agents.get(0);
					}
				}
				if (currentMofObjectFromPDProjectExplorer == mofObject) {
					return;
				}
				parseModelBackground();
			}
	    } else if (part instanceof SDView) {
    		if (TraceSDPlugin.debugEvents) {
    			showInternalSelection(selection);
    		}
	    } else {
			if (frame == null) {
				return;
			}
			TraceInteractionUpdate update;
			update = new TraceInteractionUpdate();
    		externalSelectionChanged(part, selection, update);
			finishExternalExtendedSelection(update);
    		if (update.needsUpdate()) {
        		updateSDBackground(update);
    		}
	    }
	}
	
	/**
	 * Implementation of IProfileEventListener
	 */
	public void handleProfileEvent(ProfileEvent event) {
		if (TraceSDPlugin.debugEvents) {
			TraceSDPlugin.debugTraceEvents("handleProfileEvent("+event+")="/*+event.getSource()*/+" "+event.getType()); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		}
		if (event.getSource() instanceof EObject &&
			(event.getType() & (ProfileEvent.REFRESH_VIEWS |
								ProfileEvent.STOP_COLLECTING |
								ProfileEvent.UPDATE_MODEL)) != 0) {
			if (view != null &&
					view.getSDWidget() != null &&
					!view.getSDWidget().isDisposed()) {
					if (event.getType() == ProfileEvent.REFRESH_VIEWS &&
						currentMofObjectFromPDProjectExplorer == HyadesUtil.getMofObject()) {
						return;
					}
					parseModelBackground();
				}
		}
	}
	
	/**
	 * @param selection
	 */
	private void showInternalSelection(Object selection) {
		if (TraceSDPlugin.debugEvents) {
			TraceSDPlugin.debugTraceEvents("showInternalSelection("+selection+")"); //$NON-NLS-1$ //$NON-NLS-2$
		}
		// Do nothing
		if (selection instanceof IDateSelection) {
			IDateSelection ds = (IDateSelection)selection;
			if (TraceSDPlugin.debugEvents) {
				TraceSDPlugin.debugTraceEvents(Double.toString(ds.getStartDate()));
			}
		} else if (selection instanceof IStructuredSelection) {
			IStructuredSelection iss = (IStructuredSelection)selection;
			for (Iterator i = iss.iterator(); i.hasNext(); ) {
				Object o = i.next();
				if (TraceSDPlugin.debugEvents) {
					TraceSDPlugin.debugTraceEvents("IStructuredSelection:showInternalSelection("+o+")"); //$NON-NLS-1$ //$NON-NLS-2$
				}
				showInternalSelection(o);
			}
		}
	}

	/**
	 * @param part
	 * @param selection
	 * @param update
	 */
	protected void externalSelectionChanged(IWorkbenchPart part,
										    Object selection,
										    TraceInteractionUpdate update) {
		if (TraceSDPlugin.debugEvents) {
			TraceSDPlugin.debugTraceEvents("External selection from "+part+" is "+selection); //$NON-NLS-1$ //$NON-NLS-2$
		}
		if (selection instanceof IStructuredSelection) {
			if (TraceSDPlugin.debugEvents) {
				TraceSDPlugin.debugTraceEvents("IStructuredSelection"); //$NON-NLS-1$
			}
			IStructuredSelection iss = (IStructuredSelection)selection;
			for (Iterator i = iss.iterator(); i.hasNext(); ) {
				Object o = i.next();
				externalSelectionChanged(part, o, update);
			}
    	} else if (externalExtendedSelectionChanged(part, selection, update)) {
    		// done by concrete class
    	}
	}

	abstract protected boolean externalExtendedSelectionChanged(IWorkbenchPart part,
															    Object selection,
															    TraceInteractionUpdate update);

	abstract protected void finishExternalExtendedSelection(TraceInteractionUpdate update);

	/**
	 * Implementation of ISDGraphNodeSupporter
	 */
	public boolean isLifelineSupported() {
		return true;
	}
	public boolean isSyncMessageSupported() {
		return false;
	}
	public boolean isSyncMessageReturnSupported() {
		return false;
	}
	public boolean isAsyncMessageSupported() {
		return false;
	}
	public boolean isAsyncMessageReturnSupported() {
		return false;
	}
	public boolean isStopSupported() {
		return false;
	}
	
	protected ArrayList findResults;
	protected Criteria findCriteria;
	protected int currentFindIndex;
	
	/**
	 * implementation of ISDFindProvider
	 */
	public boolean find(Criteria toSearch_) {
		if (frame == null) {
			return false;
		}
		if (findResults == null ||
			findCriteria == null ||
			!findCriteria.equals(toSearch_)) {
			findResults = new ArrayList();
			findCriteria = toSearch_;
			// search in the current page:
			if (findCriteria.isLifeLineSelected()) {
				for (int i = 0; i < frame.lifeLinesCount(); i++) {
					if (TraceInteractionUtils.matchCriteria(frame.getLifeline(i).getName(), findCriteria)) {
						findResults.add(frame.getLifeline(i));
					}
				}
			}
			ArrayList msgs = new ArrayList();
			if (findCriteria.isSyncMessageSelected()) {
				for (int i = 0; i < frame.syncMessageCount(); i++) {
					if (TraceInteractionUtils.matchCriteria(frame.getSyncMessage(i).getName(), findCriteria)) {
						msgs.add(frame.getSyncMessage(i));
					}
				}
				for (int i = 0; i < frame.syncMessageReturnCount(); i++) {
					if (TraceInteractionUtils.matchCriteria(frame.getSyncMessageReturn(i).getName(), findCriteria)) {
						msgs.add(frame.getSyncMessageReturn(i));
					}
				}
			}
			if (msgs.size() > 0) {
				Object[] temp = msgs.toArray();
				Arrays.sort(temp, new DateComparator());
				findResults.addAll(Arrays.asList(temp));
			}
			List selection = view.getSDWidget().getSelection();
			if (selection != null && selection.size() == 1) {
				currentFindIndex = findResults.indexOf(selection.get(0))+1;
			} else {
				currentFindIndex = 0;
			}
		} else {
			currentFindIndex++;
		}
		if (findResults.size() > currentFindIndex) {
			GraphNode current = (GraphNode)findResults.get(currentFindIndex);
			view.getSDWidget().moveTo(current);
			return true;			
		}
		
		return notFoundYet(findCriteria);
	}
	
	/**
	 * @see org.eclipse.hyades.uml2sd.ui.actions.provider.ISDFindProvider#cancel()
	 */
	public void cancel() {
		clearFindInternals();
	}
	
	/**
	 * Called when find in the current shown graph nodes returned an empty strings.<br>
	 * It's the moment to search in other pages if any.
	 * @return
	 */
	abstract public boolean notFoundYet(Criteria toSearch);

}
