package org.eclipse.hyades.logging.events;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.ObjectStreamField;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.StringTokenizer;

/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * Taken from "Canonical Situation Data Format: The Common Base Event V1.0.1 
 * ACAB.BO0301.0.1"
 *
 * The AssociatedEvent type allows for associated events to be grouped.  It allows for 
 * identifying associated events and its associationEngine.  The associationEngine MAY 
 * be the reference to the application that created the association.
 * 
 * 
 * @author Richard K. Duggan
 * @author Paul E. Slauenwhite
 * @version 1.0.1
 * @since 1.0
 * @see IAssociatedEvent
 * @deprecated This class is deprecated as of 2004-05-17. Use the corresponding class available in org.eclipse.hyades.logging.events.cbe*
 */
public class AssociatedEventImpl implements IAssociatedEvent, Serializable, Cloneable {

    //Define the serialiVersionUID for serialization:
    //NOTE: If this object is ever changed which breaks backward compatibility,
    //this value MUST be changed.  Every effort should be made not to break 
    //backward compatibility.
    static final long serialVersionUID = 7484637542553047973L;

    //Define the fields that will be serialized: 
    //NOTE: If any new variables are added to the class this table should be 
    //updated along with the readObject and writeObject methods.
    private static final ObjectStreamField[] serialPersistentFields = { new ObjectStreamField("resolvedEvents", String[].class), new ObjectStreamField("associationEngine", String.class), new ObjectStreamField("associationEngineInfo", IAssociationEngine.class)};

    //Class instance variable(s):
    protected ArrayList resolvedEvents = null;
    protected IAssociationEngine associationEngineInfo = null;
    protected String associationEngine = null;

    /**
     * No-argument protected constructor for AssociatedEventImpl.
     * 
     * @since 1.0
     */
    protected AssociatedEventImpl() {
        super();
    }

    /**
     * Adds a Common Base Event reference to the array of resolved events.
     * 
     * Array of globalInstanceIds corresponding to the events that are associated with this event. 
     * This is a REQUIRED property, an arry with at least one element.
     * 
     * @param globalInstanceId The new Common Base Event globalInstanceId.
     * @see ICommonBaseEvent
     * @see org.eclipse.hyades.logging.events.IAssociatedEvent#addResolvedEvent(String)
     * @since 1.0
     */
    public void addResolvedEvent(String globalInstanceId) {

        if (resolvedEvents == null) {
            resolvedEvents = new ArrayList();
        }

        this.resolvedEvents.add(globalInstanceId);
    }

    /**
     * @see org.eclipse.hyades.logging.events.IAssociatedEvent#setResolvedEvents(java.lang.String)
     */
    public void setResolvedEvents(String value) {

        clearResolvedEvents();

        if (value != null) {

            StringTokenizer tokens = new StringTokenizer(value, " ");

            while (tokens.hasMoreTokens()) {

                addResolvedEvent(tokens.nextToken());
            }
        }
    }

    /**
     * @see org.eclipse.hyades.logging.events.IAssociatedEvent#setResolvedEventsAsArray(java.lang.String[])
     */
    public void setResolvedEventsAsArray(String[] value) {

        clearResolvedEvents();

        if (value != null) {

            for (int counter = 0; counter < value.length; counter++) {

                addResolvedEvent(value[counter]);
            }
        }
    }

    /**
     * Returned an array of Common Base Event references of resolved events.
     * 
     * Array of globalInstanceIds corresponding to the events that are associated with this event. 
     * This is a REQUIRED property, an arry with at least one element.
     * 
     * @return String array of Common Base Event references of resolved events.
     * @see ICommonBaseEvent
     * @see org.eclipse.hyades.logging.events.IAssociatedEvent#getResolvedEventsAsArray()
     * @since 1.0
     */
    public String[] getResolvedEventsAsArray() {

        if (resolvedEvents != null) {
            return (((String[]) (resolvedEvents.toArray(new String[resolvedEvents.size()]))));
        }

        return null;
    }

    /**
     * Returned a space delimited string of Common Base Event references of resolved events.
     * 
     * Space delimited string of Common Base Event globalInstanceIds corresponding to the 
     * event(s) that are associated with this event.
     * This is a REQUIRED property.
     * 
     * @return Space delimited string of Common Base Event references of resolved events.
     * @see ICommonBaseEvent
     * @see org.eclipse.hyades.logging.events.IAssociatedEvent#getResolvedEvents()
     * @since 1.0.1
     */
    public String getResolvedEvents() {

        if (resolvedEvents != null) {

            StringBuffer buffer = new StringBuffer();

            for (int counter = 0; counter < resolvedEvents.size(); counter++) {
                buffer.append(resolvedEvents.get(counter));
                buffer.append(" ");
            }

            return (buffer.toString().trim());
        }

        return null;
    }

    /**
     * Removes all Common Base Event references in the array of resolved events.
     * 
     * Array of globalInstanceIds corresponding to the events that are associated with this event. 
     * This is a REQUIRED property, an arry with at least one element.
     * 
     * @see ICommonBaseEvent
     * @see org.eclipse.hyades.logging.events.IAssociatedEvent#clearResolvedEvents()
     * @since 1.0
     */
    public void clearResolvedEvents() {
        resolvedEvents = null;
    }

    /**
     * Removes a named Common Base Event reference in the array of resolved events.
     * 
     * Array of globalInstanceIds corresponding to the events that are associated with this event. 
     * This is a REQUIRED property, an arry with at least one element.
     * 
     * @param globalInstanceId The removed Common Base Event globalInstanceId.
     * @see ICommonBaseEvent
     * @see org.eclipse.hyades.logging.events.IAssociatedEvent#removeResolvedEvent(String)
     * @since 1.0
     */
    public void removeResolvedEvent(String globalInstanceId) {

        if (resolvedEvents != null) {

            //Remove all duplicates:
            for (int counter = 0; counter < resolvedEvents.size(); counter++) {

                if (resolvedEvents.get(counter).equals(globalInstanceId))
                    resolvedEvents.remove(counter);
            }
        }
    }

    /**
     * Sets the assocationEngine.
     * 
     * Identifies the application that establishes association among related or 
     * associated events.  In addition, it provides properties to describe the 
     * type of the association.  The associationEngine and associationEnginInfo 
     * properties are mutually exclusive.  One of these properties MUST be defined.
     * 
     * @param name The assocationEngine to be set.
     * @see IAssociationEngine
     * @since 1.0
     */
    public void setAssociationEngine(String associationEngine) {
        this.associationEngine = associationEngine;
    }

    /**
     * Returns the associationEngine.
     * 
     * Reference to the AssocationEngine that created this AssociatedEvent.
     * The associationEngine and associationEnginInfo properties are mutually 
     * exclusive.  One of these properties MUST be defined.
     * 
     * @return String The returned associationEngine.
     * @see IAssociationEngine
     * @since 1.0
     */
    public String getAssociationEngine() {
        return associationEngine;
    }

    /**
     * Sets the associationEngineInfo.
     * 
     * Identifies the application that establishes association among related or 
     * associated events.  In addition, it provides properties to describe the 
     * type of the association.  The associationEngine and associationEnginInfo 
     * properties are mutually exclusive.  One of these properties MUST be defined.
     * 
     * @param name The associationEngineInfo to be set.
     * @since 1.0
     * @see IAssociationEngine
     */
    public void setAssociationEngineInfo(IAssociationEngine associationEngineInfo) {
        this.associationEngineInfo = associationEngineInfo;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IAssociatedEvent#setAssociationEngineInfo(java.lang.String, java.lang.String, java.lang.String)
     */
    public void setAssociationEngineInfo(String associationEngineName, String associationEngineType, String associationEngineId) {

        this.associationEngineInfo = new AssociationEngineImpl();
        this.associationEngineInfo.setName(associationEngineName);
        this.associationEngineInfo.setType(associationEngineType);
        this.associationEngineInfo.setId(associationEngineId);
    }

    /**
     * Returns the associationEngineInfo.
     * 
     * Identifies the application that establishes association among related or 
     * associated events.  In addition, it provides properties to describe the 
     * type of the association.  The associationEngine and associationEnginInfo 
     * properties are mutually exclusive.  One of these properties MUST be defined.
     * 
     * @return IAssociationEngine The returned associationEngineInfo.
     * @since 1.0
     * @see IAssociationEngine
     */
    public IAssociationEngine getAssociationEngineInfo() {
        return associationEngineInfo;
    }

    /**
     * Overrides the <code>java.lang.Object</code>'s equals() API to 
     * determine if the parameter <code>object</code> is equal to this object.
     * 
     * @param object The <code>java.lang.Object</code> to be compared to this object.
     * @return true if this object is the same as the parameter <code>object</code>, false otherwise.
     * @since 1.0
     * @see java.lang.Object#equals(java.lang.Object)
     */
    public boolean equals(Object object) {

        //Check if the parameter object is the same object reference as this object (e.g. equal):
        if (this == object)
            return true;

        else {

            //Check if the parameter object is null and is the a different type as this object(e.g. not equal):
            if ((object != null) && (getClass() == object.getClass())) {

                //Compare the value(s) of all object properties:
                Object thisObjectsProperty = getAssociationEngine();
                Object parameterObjectsProperty = ((AssociatedEventImpl) (object)).getAssociationEngine();

                if (((thisObjectsProperty != null) && (thisObjectsProperty.equals(parameterObjectsProperty))) || ((thisObjectsProperty == null) && (parameterObjectsProperty == null))) {

                    thisObjectsProperty = getAssociationEngineInfo();
                    parameterObjectsProperty = ((AssociatedEventImpl) (object)).getAssociationEngineInfo();

                    if (((thisObjectsProperty != null) && (thisObjectsProperty.equals(parameterObjectsProperty))) || ((thisObjectsProperty == null) && (parameterObjectsProperty == null))) {

                        thisObjectsProperty = getResolvedEvents();
                        parameterObjectsProperty = ((AssociatedEventImpl) (object)).getResolvedEvents();

                        if (((thisObjectsProperty != null) && (thisObjectsProperty.equals(parameterObjectsProperty))) || ((thisObjectsProperty == null) && (parameterObjectsProperty == null))) {
                            return true;
                        }
                    }
                }
            }
        }

        return false;
    }

    private void writeObject(ObjectOutputStream out) throws IOException {

        //Set the values of the serializable field(s):
        ObjectOutputStream.PutField fields = out.putFields();

        if ((resolvedEvents != null) && (!resolvedEvents.isEmpty())) {
            fields.put("resolvedEvents", ((String[]) (resolvedEvents.toArray(new String[resolvedEvents.size()]))));
        }

        fields.put("associationEngine", associationEngine);
        fields.put("associationEngineInfo", associationEngineInfo);

        //Write the values of the serializable field(s):
        out.writeFields();
    }

    private void readObject(ObjectInputStream in) throws IOException, ClassNotFoundException {

        //Prepare and read the alternate persistent field(s):
        ObjectInputStream.GetField fields = null;

        fields = in.readFields();

        //Retrieve the alternate persistent field(s):
        Object tempArray[] = null;
        tempArray = (Object[]) fields.get("resolvedEvents", null);
        if (tempArray != null) {
            int arraySize = tempArray.length;
            for (int i = 0; i < arraySize; i++) {
                addResolvedEvent((String) tempArray[i]);
            }
        }

        associationEngine = ((String) (fields.get("associationEngine", associationEngine)));
        associationEngineInfo = ((IAssociationEngine) (fields.get("associationEngineInfo", associationEngineInfo)));
    }

    /**
     * Return a clone (copy by value) of this object. This is a deep copy version, 
     * in which all the objects within this object will be copied by value.
     *  
     * @return Object The clone of this object.
     * @since 1.0
     */
    public Object clone() throws CloneNotSupportedException {

        AssociatedEventImpl copy = ((AssociatedEventImpl) (super.clone()));

        //Clone the resolvedEvents array list:
        if (this.resolvedEvents != null) {
            copy.resolvedEvents = ((ArrayList) (this.resolvedEvents.clone()));
        }

        if (this.associationEngineInfo instanceof AssociationEngineImpl) {
            copy.associationEngineInfo = ((IAssociationEngine) ((AssociationEngineImpl) (this.associationEngineInfo)).clone());
        }

        return copy;
    }

    /**
     * Resets the object's properties to their initial (e.g. null) state.
     * @since 1.0
     */
    public void init() {

        resolvedEvents = null;
        associationEngineInfo = null;
        associationEngine = null;
    }
}
