/*******************************************************************************
 * Copyright (c) 2003, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.hyades.trace.sample.loganalyzer;

import java.lang.reflect.InvocationTargetException;
import java.net.UnknownHostException;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.Hashtable;
import java.util.Iterator;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.hyades.internal.execution.local.control.Node;
import org.eclipse.hyades.internal.execution.local.control.NodeFactory;
import org.eclipse.hyades.loaders.util.XMLLoader;
import org.eclipse.hyades.log.ui.internal.wizards.LocalLogParserLoader;
import org.eclipse.hyades.logging.parsers.LogParserException;
import org.eclipse.hyades.logging.parsers.Parser;
import org.eclipse.hyades.models.hierarchy.HierarchyFactory;
import org.eclipse.hyades.models.hierarchy.TRCAgent;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.trace.internal.ui.TraceConstants;
import org.eclipse.hyades.trace.ui.HyadesConstants;
import org.eclipse.hyades.trace.ui.HyadesUtil;
import org.eclipse.hyades.trace.ui.ProfileEvent;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.util.PDCoreUtil;
import org.eclipse.hyades.ui.internal.navigator.INavigator;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.StructuredSelection;


/**
 * @author sleeloy
 *
 * This is an operation class that parsers a log file using the apache log parser.  The apache version
 * is V1.3.20.  It will also create a process and agent that represents
 * the log file that is parsed.
 */
public class AnalyzeCorrelateOperation implements IRunnableWithProgress {

	private String projectName;
	protected INavigator fViewer;
	private String tcMonitor = null;
	private TRCMonitor tRCMonitor;
	private TRCAgent _agent;
	private String logFilePath = null;
	private static final String APACHE_VERSION = "V1.3.20";
	private Parser parser;
	private String name;
	
	public AnalyzeCorrelateOperation(Parser parser, String name){
		this.parser = parser;
		this.name = name;
	}
	
	/* This is where all the work is done.  This method will take the currently set file log and parse the 
	 * file using the apache version 1.3.20 parser.  This method will create a process and agent that
	 * represents the log file in the Log Navigator viewer. 
	 * 
	 * @see org.eclipse.jface.operation.IRunnableWithProgress#run(org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void run(IProgressMonitor monitor)
		throws InvocationTargetException, InterruptedException {
			Node node;

			try {
				node = NodeFactory.createNode("localhost");
				
			} catch (UnknownHostException e) {
				throw new InterruptedException(e.getMessage());
			}
			TRCProcessProxy _process = null;
			
			IContainer container = PDCoreUtil.createContainer(new Path(projectName));

			tRCMonitor = PDCoreUtil.createMonitor(container, tcMonitor);

			String portNb = UIPlugin.getDefault().getPreferenceStore().getString(HyadesConstants.LOCALHOST_PORT);
			TRCNode trcNode = PDCoreUtil.createNode(tRCMonitor, node.getName(), portNb);

			_process = createProcess(container, trcNode);

			_agent = createAgent(container, _process, name);
			
			final TRCAgent trcAgent = _agent;
			final TRCProcessProxy trcProcess = _process;

			monitor.worked(1);
			
			try {
			
				XMLLoader xmlLoader = new XMLLoader(trcAgent);
				xmlLoader.loadEvent("<CommonBaseEvents>".getBytes(), "<CommonBaseEvents>".getBytes().length);
				Hashtable input = new Hashtable();
				input.put("file_path", logFilePath);
				input.put("APACHE_VERSION", APACHE_VERSION);
				
				LocalLogParserLoader parserLoader = new LocalLogParserLoader(parser, input);						
				parserLoader.setXMLLoader(xmlLoader);
				parserLoader.startParsing();
				xmlLoader.loadEvent("</CommonBaseEvents>".getBytes(), "</CommonBaseEvents>".getBytes().length);
			
				try {
					xmlLoader.cleanUp();				
				}
				catch(Exception exc)
				{
				}
						
	
			} catch (LogParserException l) {
				throw new InterruptedException(l.getMessage());
			}

			monitor.worked(1);
			
			
			fViewer.selectReveal(new StructuredSelection(trcAgent.getAgentProxy()));

			monitor.done();

	}
	
	/**
	 * Cleans up all resources. 
	 *
	 */
	public void dispose()
	{
	}

	/**
	 * Retuns the project name that contains the log file.
	 * 
	 * @return String that represents the project name that contains the log file.
	 */
	public String getProjectName() {
		return projectName;
	}

	/**
	 * Sets the project name that contains the log file resource.
	 * 
	 * @param string - represents the project name that contains the log file resource.
	 */
	public void setProjectName(String string) {
		projectName = string;
	}

	/**
	 * Returns the log navigator.
	 * 
	 * @return INavigator
	 */
	public INavigator getFViewer() {
		return fViewer;
	}

	/**
	 * Sets the Problem Determinisitic project explorer that contains the process and agents.
	 *  
	 * @param explorer - represents the viewer that contains the monitors and agents.
	 */
	public void setFViewer(INavigator explorer) {
		fViewer = explorer;
	}

	/**
	 * Creates a process in the PDProjectExplorer that represents the log file.
	 * 
	 */
	private TRCProcessProxy createProcess(IContainer container, TRCNode node) {
		int pID = 0;

		String logPath = "";
		Hashtable elems = new Hashtable();
		elems.put("file_path", logFilePath);
		elems.put("APACHE_VERSION", APACHE_VERSION);
		ArrayList index = new ArrayList();
		index.add("file_path");
		index.add("APACHE_VERSION");

		Iterator iterator = index.iterator();

		if (iterator.hasNext()) {
			logPath += elems.get(iterator.next()).toString();
		}

		while (iterator.hasNext()) {
			logPath += (" " + elems.get(iterator.next()).toString());
		}

		TRCProcessProxy process = null;
		EList processes = node.getProcessProxies();
		Iterator i = processes.iterator();
		int counter = 0;

		while (i.hasNext()) {
			process = (TRCProcessProxy) i.next();
			counter++;

			if ((process.getName() != null) && process.getName().equals(logPath)) {
				return process;
			}
		}

		TRCMonitor monitor = node.getMonitor();

		String rID = (new Date()).toString();

		rID = HyadesUtil.change(rID, " ", "");
		rID = HyadesUtil.change(rID, ":", "");

		String processName = node.getMonitor().getName() + "_" + node.getName() + "_" + counter + "_" + pID;

		String fileName = processName + "." + TraceConstants.PROCESS_EXT;
		IPath path = container.getFullPath().append(fileName);
		URI uri = URI.createURI("platform:/resource" + path.toString());

		Resource pDoc = Resource.Factory.Registry.INSTANCE.getFactory(uri).createResource(uri);
		pDoc.setModified(true);
		EList pExt = pDoc.getContents();

		UIPlugin.getDefault().getResourceSet().getResources().add(pDoc); // prevents reloading later

		HierarchyFactory factory = UIPlugin.getDefault().getPerftraceFactory();

		process = factory.createTRCProcessProxy();
		process.setPid(pID);

		/* Ensure the runtime UUID is set on the process object */
		process.setRuntimeId(String.valueOf(pID));

		process.setName(logPath);
		process.setLaunchMode(0); //attach                       

		process.setNode(node);
		pExt.add(process);

		ProfileEvent event = UIPlugin.getDefault().getProfileEvent();

		event.setSource(process);
		event.setType(ProfileEvent.UNSPECIFIED);
		UIPlugin.getDefault().notifyProfileEventListener(event);

		return process;
	}


	/*
	 * Creates an agent that represents the log file.
	 */
	private TRCAgent createAgent(IContainer container, TRCProcessProxy process, String name) {
		TRCAgentProxy agent = null;
		String type = HyadesConstants.LOG_AGENT_TYPE;

		EList agents = process.getAgentProxies();
		Iterator i = agents.iterator();

		while (i.hasNext()) {
			agent = (TRCAgentProxy) i.next();

			if (agent.getName().equals(name)) {
				agent.getAgent().getDefaultEvents().clear();
				return agent.getAgent();
			}
		}

		String timestamp = DateFormat.getDateTimeInstance(DateFormat.FULL, DateFormat.FULL).format(new Date());
		String rID = timestamp;

		rID = HyadesUtil.change(rID, " ", "");
		rID = HyadesUtil.change(rID, ":", "");

		String agentName = new StringBuffer(process.getNode().getMonitor().getName()).append("_").append(process.getNode().getName()).append("_").append(process.getPid()).append("_").append(rID).append("_").append(name).toString();

		agentName = HyadesUtil.change(agentName, " ", "");

		String pPath = process.eResource().getURI().toString();
		IPath path = new Path(pPath);

		if (path.segmentCount() > 1) {
			pPath = path.removeLastSegments(1).toString();
		}

		String fileName = agentName + "." + TraceConstants.AGENT_EXT;
		IPath filePath = new Path(pPath).append(fileName);

		URI uri = URI.createURI(filePath.toString());

		Resource agDoc = Resource.Factory.Registry.INSTANCE.getFactory(uri).createResource(uri);
		agDoc.setModified(true);
		EList agExt = agDoc.getContents();

		UIPlugin.getDefault().getResourceSet().getResources().add(agDoc); // prevents reloading later

		HierarchyFactory factory = UIPlugin.getDefault().getPerftraceFactory();

		//create process element
		agent = factory.createTRCAgentProxy();
		agent.setName(name);
		agent.setType(type);

		agent.setProcessProxy(process);
		process.getAgentProxies().add(agent);

		TRCAgent ag = factory.createTRCAgent();

		ag.setAgentProxy(agent);
		ag.setType(type);
		agExt.add(ag);

		return ag;
	}

	/**
	 * Returns a TRCMonitor that represents the monitor that contains the log file.
	 * @return TRCMonitor that represnets the monitor that contains the log file. the monitor that contains the log file.
	 */
	public TRCMonitor getTRCMonitor(){
		return tRCMonitor;
	}
	
	/**
	 * Returns a String that represents the monitor that contains the log file.
	 * @return String that represnets the monitor that contains the log file. the monitor that contains the log file.
	 */
	public String getTcMonitor() {
		return tcMonitor;
	}

	/**
	 * Sets a String that represents the monitor that contains the log file.
	 * 
	 * @param string - represents the monitor that contains the log file.
	 */
	public void setTcMonitor(String string) {
		tcMonitor = string;
	}

	/**
	 * Returns the location of the log file to parse.
	 * @return String that represents the location of the log file to parse.
	 */
	public String getLogFilePath() {
		return logFilePath;
	}

	/**
	 * Sets the location of the log file to parse.
	 * @param string - represents the location of the log file to parse.
	 */
	public void setLogFilePath(String string) {
		logFilePath = string;
	}

	/**
	 * Returns the agent that contains the log file.
	 * 
	 * @return TRCAgent that represents  the agent that contains the log file. 
	 */
	public TRCAgent getAgent() {
		return _agent;
	}

	/**
	 * Sets the agent that contains the log file.
	 * 
	 * @param agent - represents  the agent that contains the log file. 
	 */
	public void setAgent(TRCAgent agent) {
		_agent = agent;
	}

}
