/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v0.5
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v05.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.loaders.util;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Set;


/**
 * @author slavescu
 */
public class LoaderExtensions extends HashMap {
    //~ Static fields/initializers -----------------------------------------------------------------

    private static LoaderExtensions instance;

    //~ Constructors -------------------------------------------------------------------------------

    private LoaderExtensions() {
        super();
    }

    //~ Methods ------------------------------------------------------------------------------------

    public static LoaderExtensions getInstance() {
        if (instance == null) {
            instance = new LoaderExtensions();
            new LoaderRegistryReader().readRegistry();
            new LoadersFactoryRegistryReader().readRegistry();
        }

        return instance;
    }
}


/**
 * A plugin extension reader that populates the
 * loaders registry.
 * Clients are not expected to use this class directly.
 */
class LoaderRegistryReader extends RegistryReader {
    //~ Static fields/initializers -----------------------------------------------------------------

    static final String TAG_LOADER = "loader";
    static final String ATT_ELEMENT_NAME = "elementName";
    static final String ATT_CLASS = "class";

    //~ Constructors -------------------------------------------------------------------------------

    public LoaderRegistryReader() {
        super(RegistryReader.LOADER_PPID);
    }

    //~ Methods ------------------------------------------------------------------------------------

    protected boolean readElement(HyadesConfigurationElement element) {
        if (element.getName().equals(TAG_LOADER)) {
            String elementName = element.getAttribute(ATT_ELEMENT_NAME);

            if (elementName == null) {
                logMissingAttribute(element, ATT_ELEMENT_NAME);
            } else if (element.getAttribute(ATT_CLASS) == null) {
                logMissingAttribute(element, ATT_CLASS);
            } else {
                if (LoaderExtensions.getInstance().containsKey(elementName)) {
                    logError(element, "A loader for this element already exists, this loader will be ignored.");
                } else {
                    try {
                        IHyadesPluginClassDescriptor classDescriptor = createPluginClassDescriptor(element, ATT_CLASS);

                        LoaderExtensions.getInstance().put(elementName, (XMLFragmentLoader) classDescriptor.createInstance());
                    } catch (Exception e) {
                        logError(element, e.getLocalizedMessage());
                    }

                    return true;
                }
            }
        }

        return false;
    }
}


class LoadersFactoryRegistryReader extends RegistryReader {
    //~ Static fields/initializers -----------------------------------------------------------------

    static final String TAG_LOADERS_FACTORY = "loadersFactory";
    static final String ATT_CLASS = "class";

    //~ Constructors -------------------------------------------------------------------------------

    public LoadersFactoryRegistryReader() {
        super(RegistryReader.LOADERS_FACTORY_PPID);
    }

    //~ Methods ------------------------------------------------------------------------------------

    protected boolean readElement(HyadesConfigurationElement element) {
        if (element.getName().equals(TAG_LOADERS_FACTORY)) {
            if (element.getAttribute(ATT_CLASS) == null) {
                logMissingAttribute(element, ATT_CLASS);
            } else {
                try {
                    IHyadesPluginClassDescriptor factoryClassDescriptor = createPluginClassDescriptor(element, ATT_CLASS);
                    XMLFragmentLoadersFactory factory = (XMLFragmentLoadersFactory) factoryClassDescriptor.createInstance();
                    Set elements = factory.getSupportedElements();

                    for (Iterator iter = elements.iterator(); iter.hasNext();) {
                        String elementName = (String) iter.next();

                        if (LoaderExtensions.getInstance().containsKey(elementName)) {
                            logError(element, "A loader for this element already exists, this loader will be ignored.");
                        } else {
                            //							PluginClassDescriptor classDescriptor = new PluginClassDescriptor(element, factory.getLoaderClassName(elementName));
                            //							
                            //							XMLFragmentLoader loader = (XMLFragmentLoader)classDescriptor.createInstance();
                            try {
                                XMLFragmentLoader loader = (XMLFragmentLoader) factory.getClass().getClassLoader().loadClass(factory.getLoaderClassName(elementName)).newInstance();

                                if (loader != null) {
                                    LoaderExtensions.getInstance().put(elementName, loader);
                                }
                            } catch (Exception e) {
                                logError(element, e.getLocalizedMessage());
                            }
                        }
                    }
                } catch (Exception e) {
                    logError(element, e.getLocalizedMessage());
                }

                return true;
            }
        }

        return false;
    }
}
