/**********************************************************************
 * Copyright (c) 2003,2004 Scapa Technologies Limited and others
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.statistical.ui.editor.internal;

import org.eclipse.hyades.statistical.ui.widgets.grapher.internal.*;

import org.eclipse.hyades.statistical.ui.EditorPlugin;

import org.eclipse.hyades.model.statistical.*;

import org.eclipse.hyades.statistical.ui.widgets.spinner.internal.*;

import org.eclipse.hyades.statistical.ui.variableloader.internal.*;

import org.eclipse.core.runtime.*;

import org.eclipse.hyades.statistical.ui.widgets.zoomslider.internal.*;

import org.eclipse.jface.dialogs.*;

import org.eclipse.swt.*;
import org.eclipse.swt.widgets.*;
import org.eclipse.swt.layout.*;
import org.eclipse.swt.graphics.*;
import org.eclipse.swt.events.*; 
import org.eclipse.swt.custom.*;

import java.text.*;
import java.util.*;
import java.io.*;

import java.lang.reflect.*;

public class GraphWindow extends Composite implements PaintListener, GraphTableEditorListener, ControlTableEditorListener, SelectionListener {

public static final int TAB_GRAPHS = 0;
public static final int TAB_CONTROLS = 1;
public static final int TAB_LOG = 2;

private static final boolean RIGHT_DRAG_RELEASES_BUTTON = false;

public static final double YSLIDER_DEFMAX = 100000000;
public static final double YSLIDER_DEFMIN = -YSLIDER_DEFMAX;

private static final int XSLIDER_HEIGHT = 35;
private static final int YSLIDER_WIDTH = 50;

	ArrayList disposables = new ArrayList();

SashForm sash;

Composite topsash;
Composite bottomsash;

Composite graph_panel;

	Composite nw_panel;
	GraphCanvas ne_panel;
	Composite sw_panel;
	Composite se_panel;
	
	ReferenceLineGraph ref_min_graph;
	ReferenceLineGraph ref_maj_graph;

	IndicatorLineGraph indicator_graph;

	boolean ref_min_on = true;
	boolean ref_maj_on = true;

boolean sliders_match_graph = false;

Composite modify_panel;
	Button add_vert;
	Button del_vert;
	Button add_horz;
	Button del_horz;

TabFolder config_panel;
	TabItem config_tab;
	GraphTableEditor config_editor;
	TabItem control_tab;
//	ControlTableEditor control_editor;
	ControlTable control_editor;
	TabItem cbe_tab;
	MessagePanel cbe_text;

ArrayList colours = new ArrayList();

ArrayList ysliders = new ArrayList();
ArrayList xsliders = new ArrayList();

HashMap button_map = new HashMap();
HashMap menu_map = new HashMap();

GraphPopupListener graph_popup;

final int GRAPH_COLUMNS  = 9 + 1;//N columns plus the graph itself
Object[][] graph_values = new Object[0][GRAPH_COLUMNS];

final int CONTROL_COLUMNS  = 6 + 1;//N columns plus the control itself
Object[][] control_values = new Object[0][CONTROL_COLUMNS];

boolean constant_update = false;
long constant_update_ms = 1000;
UpdateThread update_thread = null;

	public static void disposeObject(Object o) throws Throwable {
		Class c = o.getClass();

		Throwable error = null;
		
		while (c != null) {				
			try {
				Method o_dispose = c.getDeclaredMethod("dispose",new Class[]{});
				o_dispose.setAccessible(true);
				o_dispose.invoke(o,new Object[]{});
				error = null;
				break;
			} catch (Throwable e) { 
				EditorPlugin.DBG.warning("failed to dispose object "+o.getClass()+" / "+o);
				error = e;
			}
			c = c.getSuperclass();
		}
		
		if (error != null) throw error;
	}
	
	public class GraphCanvasUpdate extends Thread {
		public void run() {

			//
			// Follow the graph data if we've been asked to
			//
			try {
				for (int i = 0; i < xsliders.size(); i++) {
					TimeZoomSlider slider = (TimeZoomSlider)xsliders.get(i);
					Button button = (Button)button_map.get(slider);
					if (button != null) {
						if (button.getSelection()) {
							jumpToNow(slider);
						} else {
//							System.out.println("not following slider "+slider.getTitle());	
						}
					} else {
//						System.out.println("no button for slider "+slider.getTitle());	
					}
				}
			} catch (Throwable e) {
				EditorPlugin.DBG.warning("Graph Canvas Update - problem updating",e);
			}

			//
			// redraw the graphs
			//
			try { 
				ne_panel.redraw();
			} catch (Throwable e) {
				EditorPlugin.DBG.warning("Graph Canvas Update - problem redrawing",e);
			}	
			
		}
	}

ArrayList update_runnables = new ArrayList();

	public void addUpdateRunnable(Runnable r) {
		update_runnables.add(r);	
	}

	public void removeUpdateRunnables(Runnable r) {
		update_runnables.remove(r);
	}

	public class UpdateThread extends Thread {
		public void run() {
			GraphCanvasUpdate update = new GraphCanvasUpdate();

			EditorPlugin.DBG.info("Graph Canvas Update - starting updates");

			while (constant_update) {				
				try {
					Thread.sleep(constant_update_ms);
				} catch (Throwable e) {
				}
				if (!constant_update) break;

				if (!isDisposed()) {
					if (!getDisplay().isDisposed()) {
						getDisplay().syncExec(update);
					}
				}
				
				for (int i = 0; i < update_runnables.size(); i++) {
					try {
						Runnable r = (Runnable)update_runnables.get(i);
						getDisplay().syncExec(r);
					} catch (Throwable t) {
					}	
				}
			}

			EditorPlugin.DBG.info("Graph Canvas Update - stopping updates");
		}	
	}

	/**
	 * Show a particular tab (TAB_GRAPH, TAB_CONTROLS, TAB_LOG ...)
	 * @param t the tab to show
	 */
	public void showTab(int t) {
		config_panel.setSelection(t);
	}

	/**
	 * Set the message log of this GraphWindow
	 * @param txt the text to set the message log to
	 */
	public void setCBEText(String txt) {
		cbe_text.setMsgText(txt);
	}

	/**
	 * Append to the message log of this GraphWindow
	 * @param txt the message to append
	 */
	public void appendCBEText(String txt) {
		cbe_text.appendMsgText(txt);
	}

	/**
	 * Get whether the sliders use the same background colour as the graph area
	 * @return whether the sliders use the same background colour as the graph area
	 */	
	public boolean getSlidersMatchGraph() {
		return sliders_match_graph;
	}

	/**
	 * Set whether the sliders use the same background colour as the graph area
	 * @param b whether the sliders use the same background colour as the graph area
	 */	
	public void setSlidersMatchGraph(boolean b) {
		sliders_match_graph = b;	
		updateSliderColors();
		setDirty(true);
	}

	private void updateSliderColors() {
		if (sliders_match_graph) {

			Color tmp;
			Color newcolor = ne_panel.getBackground();

			RGB rgb = newcolor.getRGB();
			
			int r = rgb.red;
			int g = rgb.green;
			int b = rgb.blue;
			
			int avg = (r + g + b) / 3;

			//calculate slider foreground color
			RGB slidercol;
			
//					slidercol = new RGB(
//							(int) ( (r > 128) ? 0 : 255 ),
//							(int) ( (g > 128) ? 0 : 255 ),
//							(int) ( (b > 128) ? 0 : 255 )
//							);

			slidercol = new RGB(
					(int) ( (avg > 100) ? 0 : 255 ),
					(int) ( (avg > 100) ? 0 : 255 ),
					(int) ( (avg > 100) ? 0 : 255 )
					);

			tmp = ((ZoomSlider)ysliders.get(0)).getForeground();
			Color newslidercolor = new Color(getDisplay(),slidercol);

			disposables.add(newslidercolor);
			if (disposables.remove(tmp)) tmp.dispose();	//dispose only if we created it
	

			//set slider background + foreground color
			for (int i = 0; i < xsliders.size(); i++) {
				ZoomSlider slider = (ZoomSlider)xsliders.get(i);
				slider.setBackground(newcolor);		
				slider.setForeground(newslidercolor);	
				slider.setTitleColor(newslidercolor);
				slider.updateScale();			
			}

			//set slider background + foreground color
			for (int i = 0; i < ysliders.size(); i++) {
				ZoomSlider slider = (ZoomSlider)ysliders.get(i);
				slider.setBackground(newcolor);							
				slider.setForeground(newslidercolor);					
				slider.setTitleColor(newslidercolor);				
				slider.updateScale();			
			}
			
			sw_panel.setBackground(newcolor);

		} else {
			
			Color newcolor = getDisplay().getSystemColor(SWT.COLOR_WIDGET_BACKGROUND);
			Color newslidercolor = getDisplay().getSystemColor(SWT.COLOR_WIDGET_FOREGROUND);
			
			//set slider background + foreground color
			for (int i = 0; i < xsliders.size(); i++) {
				ZoomSlider slider = (ZoomSlider)xsliders.get(i);
				slider.setBackground(newcolor);		
				slider.setForeground(newslidercolor);	
				slider.setTitleColor(newslidercolor);
				slider.updateScale();			
			}
			
			//set slider background + foreground color
			for (int i = 0; i < ysliders.size(); i++) {
				ZoomSlider slider = (ZoomSlider)ysliders.get(i);
				slider.setBackground(newcolor);							
				slider.setForeground(newslidercolor);					
				slider.setTitleColor(newslidercolor);
				slider.updateScale();			
			}
			
			sw_panel.setBackground(newcolor);
		}	
		
	}

	public void setConstantUpdate(boolean on) {
		setConstantUpdate(on,constant_update_ms);
	}

	/**
	 * Set whether to constantly refresh the graph area
	 * @param on whether to refresh or not
	 * @param ms_refresh the period to wait (in milliseconds) between refreshes
	 */
	public void setConstantUpdate(boolean on, long ms_refresh) {

		EditorPlugin.DBG.info("set constant update "+on+" / "+ms_refresh+"ms");

		if (ms_refresh < 500) ms_refresh = 500;
		constant_update_ms = ms_refresh;

		EditorPlugin.DBG.info("actual constant update "+on+" / "+ms_refresh+"ms");

		if (on && !constant_update) {
			constant_update = true;

			update_thread = new UpdateThread();

			update_thread.start();
		} else if (!on && constant_update) {
			constant_update = false;

		}

	}
	
	/**
	 * Jump a all time sliders to the current time
	 * @param slider
	 */
	public void jumpToNow() {
		for (int i = 0; i < xsliders.size(); i++) {
			jumpToNow((TimeZoomSlider)xsliders.get(i));
		}		
	}
	
	/**
	 * Jump a particular time slider to the current time
	 * @param slider
	 */
	public void jumpToNow(TimeZoomSlider slider) {
//		System.out.println("Jumping to end of slider "+slider.getTitle());
		
		indicator_graph.setIndicatorLocation(Double.MAX_VALUE);
		
		double t = System.currentTimeMillis();

		if (t <= slider.getMaxLimit()) {
			try {
			slider.configure(slider.getMinLimit(),
							slider.getMaxLimit() + (2 * 60 * 60 * 1000),
							slider.getMinVisible(),
							slider.getMaxVisible(),
							slider.getResolution());	
			} catch (Throwable e) {
			}
		}

		if (t <= slider.getMaxLimit()) {
			double diff = t-slider.getMaxVisible(); 
			
			try {		
				slider.configure(
					slider.getMinLimit(),
					slider.getMaxLimit(),
					slider.getMinVisible()+diff,
					slider.getMaxVisible()+diff,
					slider.getResolution()
					);
					
				setDirty(true);
			} catch (Throwable e) {
			}
		}
	}

Dirtiable dirtiable = null;

	/**
	 * Mark this graphWindow as being dirty ( something has changed )
	 * @param b whether this graph window is dirty or not
	 */
	public void setDirty(boolean b) {
		if (dirtiable != null) {
			dirtiable.setDirty(b);	
		}	
	}

	public GraphWindow(Composite parent, int style, Dirtiable d) {
		super(parent,style);

		dirtiable = d;
		initGraphics();
	}

	public GraphWindow(Composite parent, int style) {
		super(parent,style);

		initGraphics();
	}

	/**
	 * Get a reference to the graph which draws the indicator position and value
	 * @return the graph which draws the indicator position and value
	 */
	public Graph getIndicatorLineGraph() {
		return indicator_graph;	
	}

	/**
	 * Get a reference to the graph which draws the minor tick lines
	 * @return the graph which draws the minor tick lines
	 */
	public Graph getMinorReferenceLineGraph() {
		return ref_min_graph;	
	}

	/**
	 * Get a reference to the graph which draws the major tick lines
	 * @return the graph which draws the major tick lines
	 */
	public Graph getMajorReferenceLineGraph() {
		return ref_maj_graph;	
	}

	public void updateSliders() {
		for (int i = 0; i < ysliders.size(); i++) {
			ZoomSlider slider = (ZoomSlider)ysliders.get(i);	
			slider.updateScale();
		}
		for (int i = 0; i < xsliders.size(); i++) {
			ZoomSlider slider = (ZoomSlider)xsliders.get(i);	
			slider.updateScale();
		}
//		rebuildSliderChoices();
	}	

	/**
	 * Update all the tables to reflect any changes
	 *
	 */
	public void updateTables() {
		updateGraphTable();
		updateControlTable();	
	}

	/**
	 * Redraw the graphs in the graph area
	 *
	 */
	public void updateGraphs() {
		ne_panel.redraw();	
	}

	/**
	 * Update the graphs table to reflect any changes
	 *
	 */
	public void updateGraphTable() {
		
		Object[][] tmp = graph_values;
		
		for (int i = 0; i < tmp.length; i++) {
			Graph g = (Graph)tmp[i][0];
			//dont update these - they never change
//			tmp[i][0] = g;
//			tmp[i][1] = name;
//			tmp[i][2] = description;
			tmp[i][3] = g.getForeground().getRGB();
			tmp[i][4] = new Integer(g.getLineWidth());
			tmp[i][5] = new Integer(g.getLineStyle());
			tmp[i][6] = new Integer(xsliders.indexOf(g.getXSlider()));
			tmp[i][7] = new Integer(ysliders.indexOf(g.getYSlider()));
			tmp[i][8] = new Double(g.getStaticScaling());
//			tmp[i][9] = descriptor;
		}

		config_editor.setValues(graph_values);

	}
	
	/**
	 * Update the cotrol table to reflect any changes
	 *
	 */
	public void updateControlTable() {

		Object[][] tmp = control_values;
		
		for (int i = 0; i < tmp.length; i++) {
			ZoomControlBar bar = (ZoomControlBar)tmp[i][0];
			
//			tmp[i][0] = bar;
//			tmp[i][1] = name;
//			tmp[i][2] = description;
//			tmp[i][3] = image;
			if (bar != null) {
				tmp[i][4] = new Integer(ysliders.indexOf(bar.getZoomSlider()));
			}
		}

		rebuildSliderChoices();
		control_editor.setControlValues(control_values);
	}

	/**
	 * Add a graph to this GraphWindow
	 * @param name the name of the graph
	 * @param description the description of the graph
	 * @param g the graph to add
	 */
	public void addGraph(String name, String description, Graph g, SDDescriptor descriptor) {
		try {
			if (name == null) name = "none";
			if (description == null) description = "none";
	
			g.setIndicatorSource(indicator_graph);
	
			int len = graph_values.length+1;
			int index = graph_values.length;
			
			Object[][] tmp = new Object[len][GRAPH_COLUMNS];		
			
			for (int i = 0; i < graph_values.length; i++) {
				tmp[i] = graph_values[i];	
			}
					
			tmp[index][0] = g;
			tmp[index][1] = name;
			tmp[index][2] = description;
			tmp[index][3] = g.getForeground().getRGB();
			tmp[index][4] = new Integer(g.getLineWidth());
			tmp[index][5] = new Integer(g.getLineStyle());
			tmp[index][6] = new Integer(xsliders.indexOf(g.getXSlider()));
			tmp[index][7] = new Integer(ysliders.indexOf(g.getYSlider()));
			tmp[index][8] = new Double(g.getStaticScaling());
			tmp[index][9] = descriptor;
	
			graph_values = tmp;
			config_editor.setValues(graph_values);
	
			ne_panel.addGraph(g);	
			ne_panel.redraw();
			
			setDirty(true);
		} catch (Throwable t) {
			EditorPlugin.DBG.warning("problem adding control to GraphWindow",t);
		}
	}
	
	/**
	 * Add a control to this GraphWindow
	 * @param name the name of the control
	 * @param description the description of the control
	 * @param image an image representing the control (may be null)
	 * @param bar a bar associated with this control (may be null)
	 * @param rep a SDModifiableVariableRepresentation used to set the value of this control
	 */
	public void addControl(String name, String description, Image image, ZoomControlBar bar, SDModifiableVariableRepresentation rep) {
		try {
			if (name == null) name = "none";
			if (description == null) description = "none";
			
			int len = control_values.length+1;
			int index = control_values.length;
			
			Object[][] tmp = new Object[len][CONTROL_COLUMNS];
			
			for (int i = 0; i < control_values.length; i++) {
				tmp[i] = control_values[i];	
			}
			
			tmp[index][0] = bar;
			tmp[index][1] = name;
			tmp[index][2] = description;
			tmp[index][3] = image;
			if (bar == null) {
				tmp[index][4] = new Integer(0);
			} else {
				tmp[index][4] = new Integer(ysliders.indexOf(bar.getZoomSlider()));
			}
			tmp[index][5] = rep;
	
			control_values = tmp;
			rebuildSliderChoices();
			control_editor.setControlValues(control_values);
	
			setDirty(true);
		} catch (Throwable t) {
			EditorPlugin.DBG.warning("problem adding control to GraphWindow",t);
		}
	}

	private int getGraphIndex(Object[][] graph_values, Graph graph) {
		int len = graph_values.length;
		for (int i = 0; i < len; i++) {
			if 	(graph_values[i][0] == graph) return i;
		}
		return -1;
	}

	private int getControlIndex(Object[][] control_values, ZoomControlBar bar) {
		int len = control_values.length;
		for (int i = 0; i < len; i++) {
			if 	(control_values[i][0] == bar) return i;
		}
		return -1;
	}

	private int getControlIndex(Object[][] control_values, SDModifiableVariableRepresentation rep) {
		int len = control_values.length;
		for (int i = 0; i < len; i++) {
			if 	(control_values[i][5] == rep) return i;
		}
		return -1;
	}
		
	/**
	 * Remove the specified graph from this GraphWindow
	 * @param graph the graph to remove
	 */
	public void removeGraph(Graph graph) {

		int index = getGraphIndex(graph_values, graph);

		if (index == -1) {
			EditorPlugin.DBG.warning("attempt to remove non-existent graph (-1)");
			return; 	
		}
		if (index >= graph_values.length) {
			EditorPlugin.DBG.warning("attempt to remove non-existent graph > array length");
			return; 	
		}

		Object[][] tmp = new Object[graph_values.length-1][GRAPH_COLUMNS];		
		
		for (int i = 0; i < index; i++) {
			tmp[i] = graph_values[i];	
		}
		for (int i = index+1; i < graph_values.length; i++) {
			tmp[i-1] = graph_values[i];	
		}

		graph_values = tmp;
		config_editor.setValues(graph_values);

		ne_panel.removeGraph(graph);
		ne_panel.redraw();

		setDirty(true);
	}

	/**
	 * Remove the specified control
	 * @param rep representation associated with the control to remove
	 */
	public void removeControl(SDModifiableVariableRepresentation rep) {

		int index = getControlIndex(control_values, rep);

		if (index == -1) {
			EditorPlugin.DBG.warning("attempt to remove non-existent control (-1)");
			return; 	
		}
		if (index >= control_values.length) {
			EditorPlugin.DBG.warning("attempt to remove non-existent control > array length");
			return; 	
		}
	
		ZoomControlBar bar = (ZoomControlBar)control_values[index][0];

		Object[][] tmp = new Object[control_values.length-1][CONTROL_COLUMNS];		
		
		for (int i = 0; i < index; i++) {
			tmp[i] = control_values[i];	
		}
		for (int i = index+1; i < control_values.length; i++) {
			tmp[i-1] = control_values[i];	
		}

		control_values = tmp;
		rebuildSliderChoices();
		control_editor.setControlValues(control_values);

		if (bar != null) {
			try {
				bar.getZoomSlider().removeZoomControlBar(bar);
			} catch (Throwable e) {
				EditorPlugin.DBG.warning("problem removing control",e);
			}
		}

		setDirty(true);
	}
	
	/**
	 * remove all graphs from this GraphWindow
	 *
	 */
	public void removeAllGraphs() {

		graph_values = new Object[0][GRAPH_COLUMNS];
		config_editor.setValues(graph_values);

		ne_panel.removeAllGraphs();	
		
		ne_panel.addGraph(ref_min_graph);
		ne_panel.addGraph(ref_maj_graph);
		ne_panel.addGraph(indicator_graph);

		setDirty(true);
	}	
	
	/**
	 * Set the background colour of the graph
	 * @param rgb the background colour to set the graph area to
	 */
	public void setGraphBackground(Color col) {
		if (col != null) {
			setGraphBackground(col.getRGB());
		}
//		ne_panel.setBackground(col);
//		setDirty(true);
	}

	/**
	 * Set the background colour of the graph
	 * @param rgb the background colour to set the graph area to
	 */
	public void setGraphBackground(RGB rgb) {

		Color tmp;
		Color newcolor;
		
		if (rgb != null) {

			int r = rgb.red;
			int g = rgb.green;
			int b = rgb.blue;
			
			int avg = (r + g + b) / 3;

//					graph_canvas.setBackground(rgb);

			tmp = ne_panel.getBackground();
			newcolor = new Color(getDisplay(),rgb);

			disposables.add(newcolor);
			if (disposables.remove(tmp)) tmp.dispose();	//dispose only if we created it

			ne_panel.setBackground(newcolor);

			updateSliderColors();
				
			
			RGB mincol;
			RGB majcol;
			RGB indcol;
	
			double contrast = 40.0d;
	
			//calculate minor reference line graph colour
			mincol = new RGB(
					(int) ( (r > 128) ? r - contrast : r + contrast ),
					(int) ( (g > 128) ? g - contrast : g + contrast ),
					(int) ( (b > 128) ? b - contrast : b + contrast )
					);
	
			contrast = 60.0d;

			//calculate major reference line graph colour
			majcol = new RGB(
					(int) ( (r > 128) ? r - contrast : r + contrast ),
					(int) ( (g > 128) ? g - contrast : g + contrast ),
					(int) ( (b > 128) ? b - contrast : b + contrast )
					);

			//calculate indicator line graph colour
			indcol = new RGB(
					(int) ( (avg > 100) ? 0 : 255 ),
					(int) ( (avg > 100) ? 0 : 255 ),
					(int) ( (avg > 100) ? 0 : 255 )
					);

			//apply minor reference line graph colour
			tmp = ref_min_graph.getForeground();
			newcolor = new Color(getDisplay(),mincol);

			disposables.add(newcolor);
			if (disposables.remove(tmp)) tmp.dispose();	//dispose only if we created it
	
			ref_min_graph.setForeground(newcolor);

			//apply major reference line graph colour
			tmp = ref_maj_graph.getForeground();
			newcolor = new Color(getDisplay(),majcol);

			disposables.add(newcolor);
			if (disposables.remove(tmp)) tmp.dispose(); //dispose only if we created it

			ref_maj_graph.setForeground(newcolor);

			//apply minor reference line graph colour
			tmp = indicator_graph.getForeground();
			newcolor = new Color(getDisplay(),indcol);

			disposables.add(newcolor);
			if (disposables.remove(tmp)) tmp.dispose();	//dispose only if we created it
	
			indicator_graph.setForeground(newcolor);

			setDirty(true);
		}

	}
	
	/**
	 * Get the background colour of the graph
	 * @return the background colour of the graph canvas
	 */
	public Color getGraphBackground() {
		return ne_panel.getBackground();	
	}

	/**
	 * Set the background colour of all y sliders
	 * @param col the background colour to set to
	 */
	public void setYSlidersBackground(Color col) {
		nw_panel.setBackground(col);
		setDirty(true);
	}
	
	/**
	 * Set the background colour of all time sliders
	 * @param col the background colour to set to
	 */
	public void setXSlidersBackground(Color col) {
		se_panel.setBackground(col);
		setDirty(true);
	}
	
	/**
	 * Set the colour of the space left of the time sliders
	 * @param col the colour to set the space to
	 */
	public void setDeadSpaceBackground(Color col) {
		sw_panel.setBackground(col);
		setDirty(true);
	}
		
	private Color getColor(int r, int g, int b) {
		Color mycol = new Color(getShell().getDisplay(),r,g,b);
		colours.add(mycol);

		return mycol;
	}
	
	private void initGraphics() {
		GridLayout glayout = new GridLayout(1,false);
		GridData gdata = null;

		//
		// Set up the Sash (perhaps I could use sashForms to have more graphCanvases too?)
		//
		sash = new SashForm(this,0);
		sash.setOrientation(SWT.VERTICAL);

		disposables.add(sash);

		gdata = new GridData();
		gdata.horizontalAlignment = GridData.FILL;
		gdata.verticalAlignment = GridData.FILL;
		gdata.grabExcessHorizontalSpace = true;
		gdata.grabExcessVerticalSpace = true;

		sash.setLayoutData(gdata);

		topsash = new Group(sash,0);
		bottomsash = new Composite(sash,0);

		disposables.add(topsash);
		disposables.add(bottomsash);

		topsash.setLayout(new FillLayout());
		bottomsash.setLayout(new FillLayout());

		graph_panel = new Composite(topsash,0);

		org.eclipse.ui.help.WorkbenchHelp.setHelp(graph_panel,EditorPlugin.getDefault().getInfopopsId()+".statcon_anywhere");

//		modify_panel = new Composite(topsash,0);
		config_panel = new TabFolder(bottomsash,SWT.TOP);

		disposables.add(graph_panel);
		disposables.add(config_panel);

		sash.setWeights(new int[]{75,25});


		glayout.marginWidth = 0;
		glayout.marginHeight = 0;
		glayout.horizontalSpacing = 0;
		glayout.verticalSpacing = 0;

		setLayout(glayout);		

//		gdata = new GridData();
//		gdata.horizontalAlignment = GridData.FILL;
//		gdata.verticalAlignment = GridData.FILL;
//		gdata.grabExcessHorizontalSpace = true;
//		gdata.grabExcessVerticalSpace = true;

//		graph_panel.setLayoutData(gdata);

//		gdata = new GridData();
//		gdata.horizontalAlignment = GridData.FILL;
//		gdata.verticalAlignment = GridData.FILL;
//		gdata.grabExcessHorizontalSpace = true;
//		gdata.grabExcessVerticalSpace = false;

//		modify_panel.setLayoutData(gdata);

//		gdata = new GridData();
//		gdata.horizontalAlignment = GridData.FILL;
//		gdata.verticalAlignment = GridData.FILL;
//		gdata.grabExcessHorizontalSpace = true;
//		gdata.grabExcessVerticalSpace = true;

//		config_panel.setLayoutData(gdata);
		
		
		
		nw_panel = new Composite(graph_panel,0);
		ne_panel = new GraphCanvas(graph_panel,0);
		sw_panel = new Composite(graph_panel,0);
		se_panel = new Composite(graph_panel,0);

		sw_panel.setBackground(getDisplay().getSystemColor(SWT.COLOR_WIDGET_BACKGROUND));

		graph_popup = new GraphPopupListener(ne_panel);
		ne_panel.addMouseListener(graph_popup);
		
		disposables.add(nw_panel);
		disposables.add(ne_panel);
		disposables.add(sw_panel);
		disposables.add(se_panel);

		glayout = new GridLayout(2,false);
		gdata = null;

		glayout.marginWidth = 0;
		glayout.marginHeight = 0;
		glayout.horizontalSpacing = 0;
		glayout.verticalSpacing = 0;

		graph_panel.setLayout(glayout);		
		
		//northwest
		gdata = new GridData();
		gdata.horizontalAlignment = GridData.FILL;
		gdata.verticalAlignment = GridData.FILL;
		gdata.grabExcessHorizontalSpace = false;
		gdata.grabExcessVerticalSpace = true;
//		gdata.horizontalIndent = 0;
		nw_panel.setLayoutData(gdata);

		//northeast		
		gdata = new GridData();
		gdata.horizontalAlignment = GridData.FILL;
		gdata.verticalAlignment = GridData.FILL;
		gdata.grabExcessHorizontalSpace = true;
		gdata.grabExcessVerticalSpace = true;
//		gdata.horizontalIndent = 0;
		ne_panel.setLayoutData(gdata);

		//southwest		
		gdata = new GridData();
		gdata.horizontalAlignment = GridData.FILL;
		gdata.verticalAlignment = GridData.FILL;
		gdata.grabExcessHorizontalSpace = false;
		gdata.grabExcessVerticalSpace = false;
		gdata.heightHint = 30;
//		gdata.horizontalIndent = 0;
		sw_panel.setLayoutData(gdata);

		//southeast		
		gdata = new GridData();
		gdata.horizontalAlignment = GridData.FILL;
		gdata.verticalAlignment = GridData.FILL;
		gdata.grabExcessHorizontalSpace = true;
		gdata.grabExcessVerticalSpace = false;
		gdata.heightHint = 30;
//		gdata.horizontalIndent = 0;
		se_panel.setLayoutData(gdata);

		//modify panel
		glayout = new GridLayout(4,false);
		glayout.marginWidth = 0;
		glayout.marginHeight = 0;
		glayout.horizontalSpacing = 0;
		glayout.verticalSpacing = 0;

//		modify_panel.setLayout(glayout);

//		add_vert = new Button(modify_panel,0);
//		del_vert = new Button(modify_panel,0);
//		add_horz = new Button(modify_panel,0);
//		del_horz = new Button(modify_panel,0);

//		add_vert.setText("Add Y");
//		del_vert.setText("Del Y");
//		add_horz.setText("Add X");
//		del_horz.setText("Del X");

//		add_vert.addSelectionListener(this);
//		del_vert.addSelectionListener(this);
//		add_horz.addSelectionListener(this);
//		del_horz.addSelectionListener(this);
	
		
		//config panel
		glayout = new GridLayout(1,false);
		glayout.marginWidth = 0;
		glayout.marginHeight = 0;
		glayout.horizontalSpacing = 0;
		glayout.verticalSpacing = 0;
		
		config_panel.setLayout(glayout);
		
		//config table
		gdata = new GridData();
		gdata.horizontalAlignment = GridData.FILL;
		gdata.verticalAlignment = GridData.FILL;
		gdata.grabExcessHorizontalSpace = true;
		gdata.grabExcessVerticalSpace = true;

		//
		//Create the tables
		//		
		config_editor = new GraphTableEditor(config_panel,0,graph_values,ne_panel);
		config_editor.setLayoutData(gdata);
		config_editor.addListener(this);

		config_tab = new TabItem(config_panel,0);
		config_tab.setControl(config_editor);
		config_tab.setText(EditorPlugin.getString("TABLE_NAME_GRAPHS"));
		config_tab.setToolTipText(EditorPlugin.getString("GRAPHS_TAB_TOOLTIP"));

		ne_panel.setGraphHighlighter(new ThicknessHighlighter());
		ne_panel.setGraphSelectionSource(config_editor);

		disposables.add(config_editor);

//		control_editor = new ControlTableEditor(config_panel,0,graph_values);
		control_editor = new ControlTable(config_panel);
		control_editor.setLayoutData(gdata);
		control_editor.addListener(this);

		control_tab = new TabItem(config_panel,0);
		control_tab.setControl(control_editor);
		control_tab.setText(EditorPlugin.getString("TABLE_NAME_CONTROLS"));
		control_tab.setToolTipText(EditorPlugin.getString("CONTROLS_TAB_TOOLTIP"));

		disposables.add(control_editor);

		cbe_text = new MessagePanel(config_panel,0);
		cbe_tab = new TabItem(config_panel,0);
		cbe_tab.setControl(cbe_text);
		cbe_tab.setText(EditorPlugin.getString("TABLE_NAME_CBELOG"));
		cbe_tab.setToolTipText(EditorPlugin.getString("MESSAGES_TAB_TOOLTIP"));

		org.eclipse.ui.help.WorkbenchHelp.setHelp(config_editor,EditorPlugin.getDefault().getInfopopsId()+".statcon_graphtable");
		org.eclipse.ui.help.WorkbenchHelp.setHelp(control_editor,EditorPlugin.getDefault().getInfopopsId()+".statcon_controlstable");
		org.eclipse.ui.help.WorkbenchHelp.setHelp(cbe_text,EditorPlugin.getDefault().getInfopopsId()+".statcon_cbelog");

		addXSlider();
		addYSlider();

		ref_min_graph = new ReferenceLineGraph(getXSlider(0),getYSlider(0),getShell().getDisplay().getSystemColor(SWT.COLOR_GRAY),false);
		ref_maj_graph = new ReferenceLineGraph(getXSlider(0),getYSlider(0),getShell().getDisplay().getSystemColor(SWT.COLOR_DARK_GRAY),true);

		indicator_graph = new IndicatorLineGraph(getXSlider(0),getYSlider(0),getShell().getDisplay().getSystemColor(SWT.COLOR_BLACK));
		indicator_graph.setSourceType(IndicatorSource.TYPE_PIXEL);

		ne_panel.addGraph(ref_min_graph);
		ne_panel.addGraph(ref_maj_graph);
		ne_panel.addGraph(indicator_graph);

	}
	
	public void dispose() {
		super.dispose();

		disposables.addAll(colours);		
		disposables.addAll(xsliders);
		disposables.addAll(ysliders);
		disposables.addAll(menu_map.values());		

		EditorPlugin.DBG.info(getClass()+":disposed");
		for (int i = 0; i < disposables.size(); i++) {
			try {
			Object o = disposables.get(i);
			if (o != null) {
				if (o instanceof Widget) {
					((Widget)o).dispose();	
				} else if (o instanceof Color) {
					((Color)o).dispose();	
				} else if (o instanceof Image) {
					((Image)o).dispose();	
				} else {
					try {
						disposeObject(o);
					} catch (Throwable e) {
						EditorPlugin.DBG.warning("failed to dispose "+o);
					}
				}
			}
			} catch (Throwable e) {
				EditorPlugin.DBG.error("dispose error ",e);
			}
		}		
	}

	/**
	 * Get the number of time sliders currently in this GraphWindow
	 * @return the number of time sliders in this GraphWindow
	 */
	public int getXSliderCount() {
		return xsliders.size();
	}
	
	/**
	 * Get the number of Y sliders currently in this GraphWindow
	 * @return the number of Y sliders in this GraphWindow
	 */
	public int getYSliderCount() {
		return ysliders.size();
	}

	/**
	 * Get the Y slider at the specified index 
	 * @param index the index of the slider to get (starts at 0)
	 * @return the slider at the specified index or null if not found
	 */
	public ZoomSlider getYSlider(int index) {
		return (ZoomSlider)ysliders.get(index);
	}
	
	/**
	 * Get the index of the specified Y slider.
	 * @param slider the slider to get the index of
	 * @return the index (from 0) of the specified slider or -1 if not found
	 */
	public int getYSlider(ZoomSlider slider) {
		return ysliders.indexOf(slider);	
	}

	/**
	 * Get the time slider at the specified index
	 * @param index the index of the time slider to get
	 * @return the time slider at the specified index or null if not found
	 */
	public TimeZoomSlider getXSlider(int index) {
		return (TimeZoomSlider)xsliders.get(index);
	}

	/**
	 * Get the index of the specified time slider.
	 * @param slider the slider to get the index of
	 * @return the index (from 0) of the specified slider or -1 if not found
	 */
	public int getXSlider(TimeZoomSlider slider) {
		return xsliders.indexOf(slider);	
	}

	public void ensureGraphsAreValid() {
		TimeZoomSlider def_xslider = (TimeZoomSlider)xsliders.get(0);
		ZoomSlider def_yslider = (ZoomSlider)ysliders.get(0);

		if (!xsliders.contains(ref_min_graph.getXSlider())) {
			ref_min_graph.setXSlider(def_xslider);	
		}
		if (!ysliders.contains(ref_min_graph.getYSlider())) {
			ref_min_graph.setYSlider(def_yslider);	
		}

		if (!xsliders.contains(ref_maj_graph.getXSlider())) {
			ref_maj_graph.setXSlider(def_xslider);	
		}
		if (!ysliders.contains(ref_maj_graph.getYSlider())) {
			ref_maj_graph.setYSlider(def_yslider);	
		}

		if (!xsliders.contains(indicator_graph.getXSlider())) {
			indicator_graph.setXSlider(def_xslider);	
		}
		if (!ysliders.contains(indicator_graph.getYSlider())) {
			indicator_graph.setYSlider(def_yslider);	
		}
			
		for (int i = 0; i < graph_values.length; i++) {
			Graph g = (Graph)graph_values[i][0];
			
			ZoomSlider xslider = g.getXSlider();
			ZoomSlider yslider = g.getYSlider();
			
			if (!ysliders.contains(yslider)) {
				g.setYSlider(def_yslider);	
			}
			if (!xsliders.contains(xslider)) {
				g.setXSlider(def_xslider);	
			}
		}
		
	}

	/**
	 * Rebuilds slider choices for the graph and control tables
	 * Ensures that the choices are valid and that non-existent sliders cannot be chosen
	 */
	public void rebuildSliderChoices() {
		
		Integer[] x_indexes = new Integer[xsliders.size()];
		String[] x_names = new String[xsliders.size()];

		Integer[] y_indexes = new Integer[ysliders.size()];
		String[] y_names = new String[ysliders.size()];

		for (int i = 0; i < xsliders.size(); i++) {
			ZoomSlider slider = (ZoomSlider)xsliders.get(i);
			
			x_indexes[i] = new Integer(i);
			x_names[i] = slider.getTitle();
			
			if (x_names[i] == null) {
//				if (i < greek_alphabet.length) {
//					x_names[i] = "Horizontal "+greek_alphabet[i];
//				} else {
					x_names[i] = EditorPlugin.getString("HORIZONTAL_SLIDER_PREFIX")+" "+(i+1);
//				}
				
				slider.setTitle(x_names[i]);
				slider.redraw();
			}
		}

		for (int i = 0; i < ysliders.size(); i++) {
			ZoomSlider slider = (ZoomSlider)ysliders.get(i);
			
			y_indexes[i] = new Integer(i);
			y_names[i] = slider.getTitle();


			if (y_names[i] == null) {
//				if (i < greek_alphabet.length) {
//					y_names[i] = "Vertical "+greek_alphabet[i];
//				} else {
					y_names[i] = EditorPlugin.getString("VERTICAL_SLIDER_PREFIX")+" "+(i+1);
//				}
				slider.setTitle(y_names[i]);
				slider.redraw();
			}
		}

		config_editor.setXSliderChoices(x_indexes,x_names);		
		config_editor.setYSliderChoices(y_indexes,y_names);		

		control_editor.setSliderChoices(y_indexes,y_names);		
		
	}
	
	/**
	 * Shift all the bars from one zoomslider to another
	 * This is used when a zoom slider is removed and the bars must be moved to a valid slider
	 * @param from the slider to remove the bars from
	 * @param to the slider to move the bars to
	 */
	public void shiftBars(ZoomSlider from, ZoomSlider to) {
		ZoomControlBarsCollection bars = from.getZoomControlBarsCollection();	
		
		Vector v = bars.getZoomControlBars();
		for (int i = 0; i < v.size(); i++) {
			ZoomControlBar bar = (ZoomControlBar)v.get(i);
			from.removeZoomControlBar(bar);
			bar.setZoomSlider(to);
			to.addZoomControlBar(bar);
		}

	}

	public void updateSliderLayouts() {

			GridData gdata = (GridData)nw_panel.getLayoutData();
			//northwest
//			gdata = new GridData();
//			gdata.horizontalAlignment = GridData.FILL;
//			gdata.verticalAlignment = GridData.FILL;
//			gdata.grabExcessHorizontalSpace = false;
//			gdata.grabExcessVerticalSpace = true;
			gdata.widthHint = YSLIDER_WIDTH * ysliders.size();
			nw_panel.setLayoutData(gdata);


			gdata = (GridData)se_panel.getLayoutData();
//			gdata = new GridData();
//			gdata.horizontalAlignment = GridData.FILL;
//			gdata.verticalAlignment = GridData.FILL;
//			gdata.grabExcessHorizontalSpace = true;
//			gdata.grabExcessVerticalSpace = false;
			gdata.heightHint = XSLIDER_HEIGHT * xsliders.size();
			se_panel.setLayoutData(gdata);


			se_panel.layout(true);
			se_panel.redraw();

			nw_panel.layout(true);
			nw_panel.redraw();

			graph_panel.layout(true);
			graph_panel.redraw();

//			nw_panel.update();
//			se_panel.update();
			graph_panel.update();
	}

	/**
	 * Remove the specified Y slider
	 * @param slider the Y slider to remove
	 */
	public void removeYSlider(ZoomSlider slider) {
		removeYSlider(ysliders.indexOf(slider));
	}
		
	/**
	 * Remove the Y slider at the specified index (indexes start at 0)
	 * @param index the index of the y slider to remove
	 */
	public void removeYSlider(int index) {

		if (index > -1 && index < ysliders.size() && ysliders.size() > 1) {
		
			ZoomSlider slider = (ZoomSlider)ysliders.get(index);
			ysliders.remove(slider);

			shiftBars(slider,(ZoomSlider)ysliders.get(0));

			SliderPopupListener popup = (SliderPopupListener)menu_map.get(slider);
			if (popup != null) {
				popup.dispose();	
			}
			
			ensureGraphsAreValid();

			slider.dispose();

			GridLayout glayout = new GridLayout(ysliders.size(),false);
			glayout.marginWidth = 0;
			glayout.marginHeight = 0;
			glayout.horizontalSpacing = 0;
			glayout.verticalSpacing = 0;

			nw_panel.setLayout(glayout);

//			nw_panel.layout(true);
//			nw_panel.redraw();
//			nw_panel.update();

			rebuildSliderChoices();		
			
			updateSliderLayouts();

			updateTables();
			setDirty(true);
		}
			
	}

	/**
	 * Remove the specified time slider (if it is a part of this graph)
	 * @param slider the time slider to remove
	 */
	public void removeXSlider(ZoomSlider slider) {
		removeXSlider(xsliders.indexOf(slider));
	}
	
	/**
	 * Remove the time slider with the specified index (indexes start at 0)
	 * @param index the index of the time slider to remove
	 */
	public void removeXSlider(int index) {

		if (index > -1 && index < xsliders.size() && xsliders.size() > 1) {
		
			ZoomSlider slider = (ZoomSlider)xsliders.get(index);
			xsliders.remove(slider);

			shiftBars(slider,(TimeZoomSlider)xsliders.get(0));

			SliderPopupListener popup = (SliderPopupListener)menu_map.get(slider);
			if (popup != null) {
				popup.dispose();	
			}

			Button button = (Button)button_map.get(slider);
			if (button != null) button.dispose();

			ensureGraphsAreValid();

			slider.dispose();
			
			GridLayout glayout = new GridLayout(2,false);
			glayout.marginWidth = 0;
			glayout.marginHeight = 0;
			glayout.horizontalSpacing = 0;
			glayout.verticalSpacing = 0;

			se_panel.setLayout(glayout);

			rebuildSliderChoices();		

			updateSliderLayouts();

			updateTables();
			setDirty(true);
		}

	}

	/**
	 * add a Y slider to this graph window (Y sliders are the vertical ones which values are graphed against)
	 *
	 */
	public void addYSlider() {
		
		GridLayout glayout = new GridLayout(ysliders.size()+1,false);
		glayout.marginWidth = 0;
		glayout.marginHeight = 0;
		glayout.horizontalSpacing = 0;
		glayout.verticalSpacing = 0;

		nw_panel.setLayout(glayout);

		try {
			long t = System.currentTimeMillis();
			ZoomSlider slider = new ZoomSlider(nw_panel);
			org.eclipse.ui.help.WorkbenchHelp.setHelp(slider,EditorPlugin.getDefault().getInfopopsId()+".statcon_anywhere");
//			EditorPlugin.DBG.info("slider constructor took "+(System.currentTimeMillis()-t));

			slider.setOrientation(ZoomSlider.VERTICAL);
			slider.configure(0,YSLIDER_DEFMAX,0,500,0.01);

//			slider.setMaxLimit(1000);
//			slider.setMaxVisible(500);
//			slider.setMinLimit(0);
//			slider.setMinVisible(0);

			GridData gridData = new GridData();
//			gridData.widthHint = 50;
			gridData.heightHint = 50;
			gridData.verticalAlignment = GridData.FILL;
			gridData.horizontalAlignment = GridData.FILL;
			gridData.grabExcessHorizontalSpace = true;
			gridData.grabExcessVerticalSpace = true;

			slider.setLayoutData(gridData);

			int i = ysliders.size();
//			if (i < greek_alphabet.length) {
//				slider.setTitle("Vertical "+greek_alphabet[i]);
//			} else {
//				slider.setTitle("Vertical "+(i+1));
//			}

			
//			slider.addPaintListener(this);
			ysliders.add(slider);

			disposables.add(slider);

			SliderPopupListener slider_popup = new SliderPopupListener(slider,false);
			
			menu_map.put(slider,slider_popup);
			
//			nw_panel.layout(true);
//			nw_panel.redraw();
//			nw_panel.update();

			rebuildSliderChoices();		

			updateSliderColors();

			updateSliderLayouts();

			setDirty(true);

			return;
			
		} catch (Throwable e) {
			EditorPlugin.DBG.error("problem adding Y slider ",e);
		}
		
		updateSliderColors();
		rebuildSliderChoices();		
	}
	
	/**
	 * Add a time slider to this graph window
	 *
	 */
	public void addXSlider() {

		GridLayout glayout = new GridLayout(2,false);
		glayout.marginWidth = 0;
		glayout.marginHeight = 0;
		glayout.horizontalSpacing = 0;
		glayout.verticalSpacing = 0;
		
		se_panel.setLayout(glayout);
		
		try {

			//
			// Set up the slider
			//
			TimeZoomSlider slider = new TimeZoomSlider(se_panel);
			org.eclipse.ui.help.WorkbenchHelp.setHelp(slider,EditorPlugin.getDefault().getInfopopsId()+".statcon_anywhere");

			slider.setOrientation(ZoomSlider.HORIZONTAL);
			slider.setDirection(ZoomSlider.DECREASING);

			slider.configure(0,3600000,0,60000,1000.0d);

			slider.setWallTime(true);

			GridData gridData = new GridData();
			gridData.widthHint = 50;
			gridData.heightHint = 30;
			gridData.verticalAlignment = GridData.FILL;
			gridData.horizontalAlignment = GridData.FILL;
			gridData.grabExcessHorizontalSpace = true;
			gridData.grabExcessVerticalSpace = true;

			slider.setLayoutData(gridData);

			//
			// Set up the "follow most recent data" button
			//
			Button slider_button = new Button(se_panel,SWT.TOGGLE);
			
			slider_button.setText(">");
	
			button_map.put(slider,slider_button);
	
			gridData = new GridData();
			gridData.widthHint = 15;
//			gridData.heightHint = 30;
			gridData.verticalAlignment = GridData.FILL;
			gridData.horizontalAlignment = GridData.FILL;
			gridData.grabExcessHorizontalSpace = false;
			gridData.grabExcessVerticalSpace = true;

			slider_button.setLayoutData(gridData);

	
//			int i = xsliders.size();
//			if (i < greek_alphabet.length) {
//				slider.setTitle("Horizontal "+greek_alphabet[i]);
//			} else {
//				slider.setTitle("Horizontal "+(i+1));
//			}

			
			xsliders.add(slider);

			disposables.add(slider);

			SliderPopupListener slider_popup = new SliderPopupListener(slider,true);

			menu_map.put(slider,slider_popup);

//			se_panel.layout(true);
//			se_panel.redraw();
//			se_panel.update();

			rebuildSliderChoices();		

			updateSliderColors();

			updateSliderLayouts();

			setDirty(true);

			return;
						
		} catch (Throwable e) {
			EditorPlugin.DBG.error("problem adding X slider ",e);
		}

		updateSliderColors();
		rebuildSliderChoices();		
	}
	
	/**
	 * Set whether a particular time slider is following the current time (basically sets the state of
	 * the button to the right of the time slider)
	 * @param slider the slider's button to set
	 * @param b the state to set the button to
	 */
	public void setTimeSliderFollowing(TimeZoomSlider slider, boolean b) {
		Button button = (Button)button_map.get(slider);
		if (button != null) {
			button.setSelection(b);	
		}
	}

	/**
	 * Get whether a particular time slider is following the current time (gets the state of the button to
	 * the right of the time slider)
	 * @param slider the slider's button to get the state of
	 * @return the state of the slider's button
	 */
	public boolean getTimeSliderFollowing(TimeZoomSlider slider) {
		Button button = (Button)button_map.get(slider);
		if (button != null) {
			return button.getSelection();	
		}
		return false;
	}
	
	/**
	 * Snap a slider to its associated graphs.
	 * Y sliders get the max and min value of all graphs associated with them, then set their min and max visibles
	 * accordingly
	 * X slider (time sliders) get the max and min time of all graphs associated with them, then set their min and 
	 * max visibles accordingly
	 * @param slider
	 */
	public void snapSlider(ZoomSlider slider) {
		double yslider_min = -2000000000;
		double yslider_max = 2000000000;
		
		double min = Double.MAX_VALUE;
		double max = Double.MIN_VALUE;
		boolean minset = false;
		boolean maxset = false;

		for (int i = 0; i < graph_values.length; i++) {
			Graph g = (Graph)graph_values[i][0];
			if (slider == g.getXSlider()) {
				double xmin = g.getXMin();
				double xmax = g.getXMax();

//EditorPlugin.DBG.info("Checking graph "+i+" "+xmin+"/"+xmax);
				
				if (xmin != Double.NEGATIVE_INFINITY
					&& xmin != Double.POSITIVE_INFINITY) {
//EditorPlugin.DBG.info("Checking graph "+i+" Xmin ok");
					if (xmin < min) {
						min = xmin;
						minset = true;
					}	
				}

				if (xmax != Double.NEGATIVE_INFINITY
					&& xmax != Double.POSITIVE_INFINITY) {
//EditorPlugin.DBG.info("Checking graph "+i+" Xmax ok");
					if (xmax > max) {
						max = xmax;
						maxset = true;
					}	
				}
				
			} else if (slider == g.getYSlider()) {
				double ymin = g.getYMin();
				double ymax = g.getYMax();

//EditorPlugin.DBG.info("Checking graph "+i+" "+ymin+"/"+ymax);

				if (ymin != Double.NEGATIVE_INFINITY
					&& ymin != Double.POSITIVE_INFINITY) {
//EditorPlugin.DBG.info("Checking graph "+i+" Ymin ok "+(ymin < min));
					if (ymin < min) {
						min = ymin;
						minset = true;
					}	
				}

				if (ymax != Double.NEGATIVE_INFINITY
					&& ymax != Double.POSITIVE_INFINITY) {
//EditorPlugin.DBG.info("Checking graph "+i+" Ymax ok "+(ymax > max));
					if (ymax > max) {
						max = ymax;
						maxset = true;
					}	
				}
					
			}
		}

		

//EditorPlugin.DBG.info("Preliminary Values "+min+"/"+max);

		if (max > min && minset && maxset) {
		
			double fourweeks = 1000d * 60d * 60d * 24d * 7d * 4d;
			if ((max - min) > fourweeks) {
				min = max - fourweeks;
			}
//EditorPlugin.DBG.info("Preliminary Values 2 "+min+"/"+max);
			if ((max - min) < slider.getResolution()) {
				max += (slider.getResolution() * 5);	
				min -= (slider.getResolution() * 5);	
			}
//EditorPlugin.DBG.info("Preliminary Values 3 "+min+"/"+max);
			
			EditorPlugin.DBG.info("Snapping to "+min+" / "+max);
		
			try {
				if (slider instanceof TimeZoomSlider) {

					double twohours = 1000d * 60d * 60d * 2d;
					
					slider.configure(min,max+twohours,min,max,slider.getResolution());
					setDirty(true);
					
				} else {

					double minlim = min;
					double maxlim = max;

					if (min < 0) {
						minlim = YSLIDER_DEFMIN;	
					} else {
						minlim = 0;	
					}
					maxlim = YSLIDER_DEFMAX;
					
					if (min >= 0) {
						min = 0;	
					}
/**
					if (max > 0) {
						maxlim = max * 2d;	
					} else {
						maxlim = max * 0.5d;	
					}

					if (min < 0) {
						minlim = min * 2d;	
					} else {
//						minlim = min * 0.5d;	
						minlim = 0;
					}
**/					

					slider.configure(minlim,maxlim,min,max,slider.getResolution());
					setDirty(true);
				}
				slider.redraw();
			} catch (Throwable x) {
				EditorPlugin.DBG.error("problem snapping slider",x);
			}
		
		} else {
			EditorPlugin.DBG.info("snap slider request is invalid");
//			EditorPlugin.DBG.info("Ignoring");
		}
		
		ne_panel.redraw();
	}
	
	////////////////////////////////////////////
	// PAINT LISTENER
	////////////////////////////////////////////
	
	//listen to the sliders and repaint the graph when necessary
	//FUNCTIONALITY REMOVED - uses too much CPU painting tons of graphs all the time
	public void paintControl(PaintEvent e) {
		ne_panel.redraw();
	}

	////////////////////////////////////////////
	// SELECTION LISTENER
	////////////////////////////////////////////

	public void widgetDefaultSelected(SelectionEvent e) {} 
	public void widgetSelected(SelectionEvent e) {
		Object o = e.getSource();	
		
		if (o == add_vert) {
			addYSlider();
			setDirty(true);
		} else if (o == del_vert) {
			removeYSlider(0);
			setDirty(true);
		} else if (o == add_horz) {
			addXSlider();
			setDirty(true);
		} else if (o == del_horz) {
			removeXSlider(0);
			setDirty(true);
		} else {
			return;	
		}
		
		layout(true);
		redraw();
		update();
	}

	////////////////////////////////////////////
	// CONTROL TABLE EDITOR LISTENER
	////////////////////////////////////////////

	public void barMoved(ZoomControlBar bar, int newindex) {
		if (bar == null) {
			EditorPlugin.DBG.error("zoom control bar is null",new Exception("Null bar being moved?"));
			return;
		}
		
		ZoomSlider old_slider = bar.getZoomSlider();
		ZoomSlider new_slider = null;
		
		if (newindex >= ysliders.size()) {
			newindex -= (ysliders.size()-1);
			new_slider = (ZoomSlider)xsliders.get(newindex);
		} else {
			new_slider = (ZoomSlider)ysliders.get(newindex);
		}
		
		if (old_slider != new_slider) {

			if (!old_slider.isDisposed()) {
				old_slider.removeZoomControlBar(bar);
			}
			bar.setZoomSlider(new_slider);
			new_slider.addZoomControlBar(bar);

		}
		setDirty(true);
	}

	public void barEdited(ZoomControlBar bar) {
		control_values = control_editor.getControlValues();
		
		int index = getControlIndex(control_values,bar);
		
		Object[] vals = control_values[index];
		
		barMoved(bar, ((Integer)vals[ControlTableEditor.VAL_SLIDER]).intValue() );

		Number num = (Number) ((SDModifiableVariableRepresentation)vals[5]).getLastRequestedValue();
		bar.setValue(num.doubleValue());
		bar.getZoomSlider().updateIndicators();
		bar.getZoomSlider().updateScale();
		redraw();
	}

	////////////////////////////////////////////
	// GRAPH TABLE EDITOR LISTENER
	////////////////////////////////////////////

	public void graphEdited(Graph g) {
//		Object[][] data = config_editor.getValues();
		graph_values = config_editor.getValues();
		
		int index = getGraphIndex(graph_values, g);
		
		Object[] vals = graph_values[index];
		
		//set colour
		Color tmp = g.getForeground();
		Color newcolor = new Color(getDisplay(),(RGB)vals[GraphTableEditor.VAL_RGB]);

		disposables.add(newcolor);
		disposables.remove(tmp);
		
		g.setForeground(newcolor);
		tmp.dispose();

		//set line width
		Integer nwidth = (Integer)vals[GraphTableEditor.VAL_WIDTH];
		g.setLineWidth(nwidth.intValue());

		//set line style
		Integer nstyle = (Integer)vals[GraphTableEditor.VAL_STYLE];
		g.setLineStyle(nstyle.intValue());

		//set x slider
		int xsindex = ((Integer)vals[GraphTableEditor.VAL_XSLIDER]).intValue();
		if (xsindex >= 0 && xsindex < xsliders.size()) {
			g.setXSlider((TimeZoomSlider)xsliders.get(xsindex));
		}

		//set y slider
		int ysindex = ((Integer)vals[GraphTableEditor.VAL_YSLIDER]).intValue();
		if (ysindex >= 0 && ysindex < ysliders.size()) {
			g.setYSlider((ZoomSlider)ysliders.get(ysindex));
		}

		//set scaling
		double multiplier = ((Double)vals[GraphTableEditor.VAL_SCALING]).doubleValue();
		g.setStaticScaling(multiplier);
		
		ne_panel.redraw();

		setDirty(true);
	}

	////////////////////////////////////////////
	// INNER CLASSES
	////////////////////////////////////////////

	////////////////////////////////////////////
	// GRAPH POPUP MENU HANDLER
	////////////////////////////////////////////

	class GraphPopupListener implements MouseListener, SelectionListener {

		Menu menu;
		MenuItem set_color;
		MenuItem default_color;
		MenuItem match_color;
		MenuItem msep;
		MenuItem export_csv;

		GraphCanvas graph_canvas;

		public void dispose() {
			menu.dispose();
			set_color.dispose();
			default_color.dispose();
			export_csv.dispose();
			msep.dispose();
			match_color.dispose();
		}

		public GraphPopupListener(GraphCanvas graph_canvas) {
			this.graph_canvas = graph_canvas;

			initMenu();
			initEventHandling();
		}
		
		private void initEventHandling() {
			graph_canvas.addMouseListener(this);
			set_color.addSelectionListener(this);
			default_color.addSelectionListener(this);
			export_csv.addSelectionListener(this);
			match_color.addSelectionListener(this);
		}
		
		private void initMenu() {
			menu = new Menu(graph_canvas);
			
			set_color = new MenuItem(menu,0);
			default_color = new MenuItem(menu,0);
			match_color = new MenuItem(menu,0);
			msep = new MenuItem(menu,SWT.SEPARATOR);
			export_csv = new MenuItem(menu,0);
			
			initMenuText();
		}

		private void initMenuText() {
			export_csv.setText(EditorPlugin.getString("EXPORT_GRAPHS_CSV"));
			set_color.setText(EditorPlugin.getString("MENUITEM_SET_GRAPH_BACKGROUND_COLOR"));
			default_color.setText(EditorPlugin.getString("MENUITEM_DEFAULT_GRAPH_BACKGROUND_COLOR"));
			if (getSlidersMatchGraph()) {
				match_color.setText(EditorPlugin.getString("MENUITEM_SEPARATE_SLIDERS_FROM_GRAPH"));
			} else {
				match_color.setText(EditorPlugin.getString("MENUITEM_MATCH_SLIDERS_TO_GRAPH"));
			}
		}
		
		public void widgetDefaultSelected(SelectionEvent e) {
			widgetSelected(e);
		}
		public void widgetSelected(SelectionEvent e) {
			Object o = e.getSource();
			
			if (o == set_color) {

				EditorPlugin.DBG.info("set graph background colour");

				ColorDialog dialog = new ColorDialog(graph_canvas.getShell());
				dialog.setRGB(graph_canvas.getBackground().getRGB());	
				RGB rgb = dialog.open();

				if (rgb != null) {
					setGraphBackground(rgb);
				}
				
			} else if (o == default_color) {
				
				EditorPlugin.DBG.info("default graph background colour");

				setGraphBackground(getDisplay().getSystemColor(SWT.COLOR_LIST_BACKGROUND));
				
			} else if (o == match_color) {

				EditorPlugin.DBG.info("separate / match sliders to graph");

				setSlidersMatchGraph(!getSlidersMatchGraph());
				updateSliderColors();
				setDirty(true);
				
			} else if (o == export_csv) {
				
				//Start up a dialog and get a time from it
				
				TimeZoomSlider xslider = getXSlider(0);
				ExportTimeDialog etdialog = new ExportTimeDialog(getShell(),(long)xslider.getIncrement(),(long)xslider.getMinVisible(),(long)xslider.getMaxVisible());
				if (etdialog.open() == ExportTimeDialog.OK) {
					long avg = etdialog.getAveragePeriodMS();
					long start = etdialog.getStartTime();
					long finish = etdialog.getMaxTime();
					
					FileDialog fdialog = new FileDialog(getShell(),SWT.SAVE);
					fdialog.setFilterExtensions(new String[]{"*.csv"});
					fdialog.setFilterNames(new String[]{"Comma Separated Values (*.csv)"});
					String file = fdialog.open();
					if (file != null) {
	
						if (file.indexOf('.') == -1) {
							file = file+".csv";	
						}
	
						File f = new File(file);
	
						if (f != null) {
							if (f.exists()) {
								MessageBox mbox = new MessageBox(getShell(),SWT.YES|SWT.NO|SWT.ICON_WARNING);
								mbox.setText(EditorPlugin.getString("FILE_EXISTS"));
								mbox.setMessage(EditorPlugin.getString("FILE_EXISTS_MSG"));	
								if (mbox.open() != SWT.YES) {
									return;
								}
							}
						}
	
						ProgressMonitorDialog pdialog = new ProgressMonitorDialog(getShell());
						pdialog.setCancelable(false);
						try {
							pdialog.run(true,false,new CSVExport(file,avg,start,finish));
						} catch (Throwable x) {
							EditorPlugin.DBG.logVisibleError(x,EditorPlugin.getString("ERROR_CSV_EXPORT_DIALOG"),false);
						}
	
					}
/************
						SimpleDateFormat tformat = new SimpleDateFormat("HH:mm:ss\n");
						SimpleDateFormat dformat = new SimpleDateFormat("dd/MMMM/yyyy\n");

						ColumnPrinter cprinter = new ColumnPrinter();
						cprinter.setCellPrefix("\"");
						cprinter.setCellPostfix("\",");
					
						int COLUMN_DATE = 0;
						int COLUMN_TIME = 1;
						int COL = 2;
						
						cprinter.setHeaderText(COLUMN_DATE,0,"Date");
						cprinter.setHeaderText(COLUMN_TIME,0,"Time");
						
						for (long t = start; t < finish; t+=avg) {
							cprinter.appendColumnText(COLUMN_DATE,dformat.format(new Date(t)));
							cprinter.appendColumnText(COLUMN_TIME,tformat.format(new Date(t)));
						}
						
						for (int i = 0; i < graph_values.length; i++) {
EditorPlugin.DBG.info("Exporting COLUMN "+COL);
							int ROW = 0;
							//description
							cprinter.setHeaderText(COL,ROW++,"Description: "+graph_values[i][2]);
							//name
							cprinter.setHeaderText(COL,ROW++,(String)graph_values[i][1]);
							
							//parent names
							SDDescriptor parent = (SDDescriptor)graph_values[i][9];
							while (parent != null) {
								parent = parent.getParent();
								if (parent != null) {
									cprinter.setHeaderText(COL,ROW++,parent.getName());
								}	
							}
							
							Graph g = (Graph)graph_values[i][0];
							BasicGraphSource source = g.getGraphSource();
							StringBuffer columndata = new StringBuffer();
							if (source instanceof GraphSource) {
								GraphSource gsource = (GraphSource)source;
								for (long t = start; t < finish; t+=avg) {
									columndata.append(gsource.getAverageBetween(t,t+avg));
									columndata.append("\n");
								}
							} else if (source instanceof EnumerationGraphSource) {
								EnumerationGraphSource esource = (EnumerationGraphSource)source;
								for (long t = start; t < finish; t+=avg) {
									columndata.append(esource.getValueAt(t));
									columndata.append("\n");
								}
							} else {
								columndata.append("error - "+source.getClass());	
								columndata.append("\n");
							}
							
							cprinter.setColumnText(COL,columndata.toString());
							
							COL++;
						}
						
						EditorPlugin.DBG.info(cprinter.toString());
						try {
							FileOutputStream fout = new FileOutputStream(file);
							fout.write(cprinter.toString().getBytes());				
							fout.close();		
						} catch (Throwable x) {
							EditorPlugin.DBG.error("error writing CSV file",x);
						}
					}
*******/

				}
				
			}
		}

		public void mouseDown(MouseEvent e) {
		}
		public void mouseUp(MouseEvent e) {
			
			if (e.button > 1) {
				initMenuText();
				menu.setVisible(true);
			}
			ne_panel.redraw();
		}
		public void mouseDoubleClick(MouseEvent e) {
		}
		
	}
	
	class CSVExport implements org.eclipse.jface.operation.IRunnableWithProgress {
		String file;
		long avg, start, finish;
		public CSVExport(String file, long avg, long start, long finish) {
			this.file = file;
			this.avg = avg;
			this.start = start;
			this.finish = finish;
		}
		public void run(IProgressMonitor monitor) { 
			try {
				monitor.beginTask(EditorPlugin.getString("BUILDING_HEADER"),graph_values.length);
				
				OutputStream fout = new FileOutputStream(file);
				fout = new BufferedOutputStream(fout);

				SimpleDateFormat tformat = new SimpleDateFormat("HH:mm:ss");
				SimpleDateFormat dformat = new SimpleDateFormat("dd/MMMM/yyyy");

				ColumnPrinter cprinter = new ColumnPrinter();
				cprinter.setCellPrefix("\"");
				cprinter.setCellPostfix("\",");
			
				int COLUMN_DATE = 0;
				int COLUMN_TIME = 1;
				int COL = 2;
				
				cprinter.setHeaderText(COLUMN_DATE,0,"Date");
				cprinter.setHeaderText(COLUMN_TIME,0,"Time");

				for (int i = 0; i < graph_values.length; i++) {
					int ROW = 0;
					//description
					cprinter.setHeaderText(COL,ROW++,"Description: "+graph_values[i][2]);
					//name
					cprinter.setHeaderText(COL,ROW++,(String)graph_values[i][1]);
					
					//parent names
					SDDescriptor parent = (SDDescriptor)graph_values[i][9];
					while (parent != null) {
						parent = parent.getParent();
						if (parent != null) {
							cprinter.setHeaderText(COL,ROW++,parent.getName());
						}	
					}

					COL++;
					monitor.worked(1);
				}						
				String header = cprinter.toString();
				fout.write(header.getBytes());				

				EditorPlugin.DBG.info("Wrote headers OK");

				int delta = (int)(finish-start);
				int worked = 0;
				monitor.beginTask(EditorPlugin.getString("EXPORTING_DATA"),delta);
				for (long t = start; t < finish; t+=avg) {

					worked += avg;
					if (worked > delta/500) {
						monitor.worked(worked);	
						worked = 0;
					}

					fout.write("\"".getBytes());
					fout.write(dformat.format(new Date(t)).getBytes());
					fout.write("\",".getBytes());
					fout.write("\"".getBytes());
					fout.write(tformat.format(new Date(t)).getBytes());
					fout.write("\",".getBytes());
					for (int i = 0; i < graph_values.length; i++) {
						Graph g = (Graph)graph_values[i][0];
						BasicGraphSource source = (BasicGraphSource)g.getGraphSource();

						String val;
						
						if (source instanceof GraphSource) {
							GraphSource gsource = (GraphSource)source;
							val = Double.toString(gsource.getAverageBetween(t,t+avg));
						} else if (source instanceof EnumerationGraphSource) {
							EnumerationGraphSource esource = (EnumerationGraphSource)source;
							val = esource.getValueAt(t).toString();
						} else {
							val = "error - "+source.getClass();	
						}
						fout.write("\"".getBytes());
						fout.write(val.getBytes());
						fout.write("\",".getBytes());
					}
					fout.write('\n');

				}
				monitor.done();
				EditorPlugin.DBG.info("Wrote data OK");

				fout.close();
			
			} catch (IOException x) {
				EditorPlugin.DBG.error(EditorPlugin.getString("ERROR_CSV_EXPORT_FILE"),x);
				EditorPlugin.DBG.logVisibleError(x,EditorPlugin.getString("ERROR_CSV_EXPORT_FILE"),true);
			}		
			
		}
	}

	////////////////////////////////////////////
	// SLIDER POPUP MENU HANDLER
	////////////////////////////////////////////

	class BasicAction implements SelectionListener {
		String id = "unknown";
		String name;
		Runnable runnable;	
		MenuItem menuitem;
		Image image;
		boolean horizontal;
			
		public void dispose() {
			if (menuitem != null) menuitem.dispose();
		}

		public void widgetSelected(SelectionEvent e) {
			runnable.run();
		}
		public void widgetDefaultSelected(SelectionEvent e) {
			runnable.run();
		}
	}
	

	private BasicAction findAction(String id, ArrayList actions) {
		for (int i = 0; i < actions.size(); i++) {
			BasicAction action = (BasicAction)actions.get(i);
			if (action.id.equals(id)) {
				return action;
			}	
		}
		return null;
	}

	protected ArrayList hslider_actions = new ArrayList();
	protected ArrayList vslider_actions = new ArrayList();

	public void addSliderAction(String id, String name, Image image, Runnable runnable, boolean horizontal) {
		ArrayList slider_actions = hslider_actions;
		if (!horizontal) slider_actions = vslider_actions;
		
		BasicAction action = findAction(id,slider_actions);
		if (action == null) {
			action = new BasicAction();
			slider_actions.add(action);
		}
		action.id = id;
		action.name = name;
		action.image = image;
		action.runnable = runnable;
		action.horizontal = horizontal;
//		action.menuitem = new MenuItem(chooser_menu,0);
//		action.menuitem.setText(name);
//		action.menuitem.setImage(image);
//		action.menuitem.addSelectionListener(action);
	}

	public void removeSliderAction(String id) {
		BasicAction action = findAction(id,hslider_actions);
		if (action == null) {
			action = findAction(id,vslider_actions);
		}
		if (action != null) {
			action.dispose();
			vslider_actions.remove(action);
			hslider_actions.remove(action);
		}	
	}

	ZoomSlider selected_slider;
	double selected_slider_value;

	public ZoomSlider getSelectedSlider() {
		return selected_slider;
	}
	
	public double getSelectedSliderValue() {
		return selected_slider_value;
	}

	class SliderPopupListener implements MouseListener, SelectionListener {
		boolean horizontal;
		ZoomSlider slider;
		Menu menu;
		MenuItem add_slider;
		MenuItem remove_slider;
		MenuItem title_slider;
		MenuItem maxlock_slider;
		MenuItem minlock_slider;
		MenuItem maximum_slider;
		MenuItem minimum_slider;
		MenuItem snap_slider;

//		ZoomControlBar selected_bar;
//		Menu menu_bar;
//		MenuItem moveright_bar;
//		MenuItem moveleft_bar;
		
		public void dispose() {
			menu.dispose();
			add_slider.dispose();
			remove_slider.dispose();
			title_slider.dispose();
			maxlock_slider.dispose();
			minlock_slider.dispose();
			maximum_slider.dispose();
			minimum_slider.dispose();
			snap_slider.dispose();
			
//			menu_bar.dispose();
//			moveright_bar.dispose();
//			moveleft_bar.dispose();	
		}
		
		public SliderPopupListener(ZoomSlider slider, boolean horizontal) {
			this.slider = slider;
			this.horizontal = horizontal;

			initMenu();
			initEventHandling();
		}

		private void initEventHandling() {
			slider.addMouseListener(this);
			
			add_slider.addSelectionListener(this);
			remove_slider.addSelectionListener(this);
			title_slider.addSelectionListener(this);
			maxlock_slider.addSelectionListener(this);
			minlock_slider.addSelectionListener(this);
			maximum_slider.addSelectionListener(this);
			minimum_slider.addSelectionListener(this);
			snap_slider.addSelectionListener(this);
			
//			moveright_bar.addSelectionListener(this);
//			moveleft_bar.addSelectionListener(this);
		}

		private void initMenu() {
			menu = new Menu(slider);

			add_slider = new MenuItem(menu,0);
			title_slider = new MenuItem(menu,0);
			maxlock_slider = new MenuItem(menu,0);
			minlock_slider = new MenuItem(menu,0);
			maximum_slider = new MenuItem(menu,0);
			minimum_slider = new MenuItem(menu,0);
			remove_slider = new MenuItem(menu,0);
			snap_slider = new MenuItem(menu,0);
			
//			menu_bar = new Menu(slider);
//			
//			moveright_bar = new MenuItem(menu_bar,0);
//			moveleft_bar = new MenuItem(menu_bar,0);
			
			initMenuText();
		}

		private void initMenuEnabled() {
			if (horizontal) {
				if (getXSliderCount() <= 1) {
					remove_slider.setEnabled(false);	
				} else {
					remove_slider.setEnabled(true);	
				}	
			} else {
				if (getYSliderCount() <= 1) {
					remove_slider.setEnabled(false);	
				} else {
					remove_slider.setEnabled(true);	
				}	
			}
		}

		ArrayList menu_disposables = new ArrayList();
		
		private void initMenuText() {
			add_slider.setText(EditorPlugin.getString("ADD_SLIDER"));
			
//			SimpleDateFormat sdf = new SimpleDateFormat("EEE, d MMM yyyy HH:mm:ss");
			SimpleDateFormat sdf = new SimpleDateFormat();
			
			if (horizontal) {
				maximum_slider.setText(EditorPlugin.getString("SET_MAXIMUM")+" ("+sdf.format(new Date((long)slider.getMaxLimit()))+") ("+slider.getTitle()+")");
				minimum_slider.setText(EditorPlugin.getString("SET_MINIMUM")+" ("+sdf.format(new Date((long)slider.getMinLimit()))+") ("+slider.getTitle()+")");

			} else {
				maximum_slider.setText(EditorPlugin.getString("SET_MAXIMUM")+" ("+slider.getMaxLimit()+") ("+slider.getTitle()+")");
				minimum_slider.setText(EditorPlugin.getString("SET_MINIMUM")+" ("+slider.getMinLimit()+") ("+slider.getTitle()+")");
			}
			if (slider.isLocked(1)) {
				maxlock_slider.setText(EditorPlugin.getString("UNLOCK_SLIDER_MAX")+" ("+slider.getTitle()+")");
			} else {
				maxlock_slider.setText(EditorPlugin.getString("LOCK_SLIDER_MAX")+" ("+slider.getTitle()+")");
			}
			if (slider.isLocked(0)) {
				minlock_slider.setText(EditorPlugin.getString("UNLOCK_SLIDER_MIN")+" ("+slider.getTitle()+")");
			} else {
				minlock_slider.setText(EditorPlugin.getString("LOCK_SLIDER_MIN")+" ("+slider.getTitle()+")");
			}
			remove_slider.setText(EditorPlugin.getString("REMOVE_SLIDER")+" ("+slider.getTitle()+")");
			title_slider.setText(EditorPlugin.getString("SET_TITLE")+" ("+slider.getTitle()+")");
			snap_slider.setText(EditorPlugin.getString("SNAP_TO_ASSOCIATED_GRAPHS")+" ("+slider.getTitle()+")");
			
//			moveleft_bar.setText("Move To Previous Slider");
//			moveright_bar.setText("Move To Next Slider");
			
			//
			// Add any special actions
			//
			if (menu_disposables != null) {
				for (int i = 0; i < menu_disposables.size(); i++) {
					MenuItem mitem = (MenuItem)menu_disposables.get(i);
					mitem.dispose();	
				}	
			}

			ArrayList slider_actions = hslider_actions;
			if (!horizontal) slider_actions = vslider_actions;

			if (slider_actions.size() > 0) {
				MenuItem sep = new MenuItem(menu,SWT.SEPARATOR);
				menu_disposables.add(sep);	
				
				for (int k = 0; k < slider_actions.size(); k++) {
					BasicAction action = (BasicAction)slider_actions.get(k);
					action.menuitem = new MenuItem(menu,0);
					action.menuitem.setText(action.name);
					action.menuitem.setImage(action.image);
					action.menuitem.addSelectionListener(action);
					menu_disposables.add(action.menuitem);		
				}
			}
		}
		
		public void widgetDefaultSelected(SelectionEvent e) {
		}
		public void widgetSelected(SelectionEvent e) {
			Object o = e.getSource();

			if (o == add_slider) {
				if (horizontal) {
					EditorPlugin.DBG.info("add a horizontal slider");
					addXSlider();
				} else {
					EditorPlugin.DBG.info("add a vertical slider");
					addYSlider();
				}
				setDirty(true);
			} else if (o == remove_slider) {
				if (horizontal) {
					EditorPlugin.DBG.info("remove slider "+slider.getTitle());
					removeXSlider(slider);
				} else {
					EditorPlugin.DBG.info("remove slider "+slider.getTitle());
					removeYSlider(slider);
				}				
				setDirty(true);
			} else if (o == snap_slider) {

				EditorPlugin.DBG.info("snap slider "+slider.getTitle());
				snapSlider(slider);
				setDirty(true);
				
			} else if (o == title_slider) {

				EditorPlugin.DBG.info("set slider "+slider.getTitle()+" title");

				InputDialog dialog = new InputDialog(getShell(),EditorPlugin.getString("SET_SLIDER_TITLE"),EditorPlugin.getString("SLIDER_TITLE_REQUEST")+":",slider.getTitle(),new SliderTitleValidator());
				dialog.open();

				String newtitle = dialog.getValue();
				if (newtitle != null) {
					slider.setTitle(newtitle);
					updateSliders();
					setDirty(true);
				}

				dialog.close();

				slider.redraw();
			} else if (o == maxlock_slider) {
				EditorPlugin.DBG.info("(un)lock max of slider "+slider.getTitle());
				slider.lock(1,!slider.isLocked(1));
				setDirty(true);
			} else if (o == minlock_slider) {
				EditorPlugin.DBG.info("(un)lock in of slider "+slider.getTitle());
				slider.lock(0,!slider.isLocked(0));
				setDirty(true);
			} else if (o == maximum_slider) {

				EditorPlugin.DBG.info("set maximum for slider "+slider.getTitle());
				
				double newlimit = 0;
				try {
					if (horizontal) {
						DateDialog dialog = new DateDialog(getShell(),EditorPlugin.getString("SET_TIME_SLIDER_MAX"));
						dialog.setDate((long)slider.getMaxLimit());
						newlimit = ((Long)dialog.open()).doubleValue();
					} else {
						InputDialog dialog = new InputDialog(getShell(),EditorPlugin.getString("SET_SLIDER_MAX"),EditorPlugin.getString("NEW_SLIDER_MAXIMUM")+":",""+slider.getMaxLimit(),new SliderMaxValidator(slider));
						dialog.open();
						newlimit = Double.parseDouble(dialog.getValue());
					}
				} catch (Throwable x) {
					EditorPlugin.DBG.warning("problem getting slider configuration",x);
					return;
				}
				
				try {
					double limit = newlimit;
					
					double min = slider.getMinLimit();
					double max = limit;
					double minvis = slider.getMinVisible();
					double maxvis = Math.min(limit,slider.getMaxVisible());
					double res = slider.getResolution();

					slider.configure(min,max,minvis,maxvis,res);
					setDirty(true);

				} catch (Throwable x) {
					EditorPlugin.DBG.warning("problem setting slider configuration",x);
				}

				slider.redraw();
			} else if (o == minimum_slider) {

				EditorPlugin.DBG.info("set minimum for slider "+slider.getTitle());


				double newlimit = 0;
				try {
					if (horizontal) {
						DateDialog dialog = new DateDialog(getShell(),EditorPlugin.getString("SET_TIME_SLIDER_MIN"));
						dialog.setDate((long)slider.getMinLimit());
						newlimit = ((Long)dialog.open()).doubleValue();
					} else {
						InputDialog dialog = new InputDialog(getShell(),EditorPlugin.getString("SET_SLIDER_MIN"),EditorPlugin.getString("NEW_SLIDER_MINIMUM")+":",""+slider.getMinLimit(),new SliderMinValidator(slider));
						dialog.open();
						newlimit = Double.parseDouble(dialog.getValue());
					}
				} catch (Throwable x) {
					EditorPlugin.DBG.warning("problem getting slider configuration",x);
					return;
				}

				try {
					double limit = newlimit;
					
					double min = slider.getMinLimit();
					double max = slider.getMaxLimit();
					double minvis = Math.max(limit,slider.getMinVisible());
					double maxvis = slider.getMaxVisible();
					double res = slider.getResolution();
	
					double diff = limit-min;

					slider.configure(limit,max,minvis,maxvis,res);
					setDirty(true);

				} catch (Throwable x) {
					EditorPlugin.DBG.warning("problem setting slider configuration",x);

				}

				slider.redraw();
			}
		}

	////////////////////////////////////
	// Mouse Listener
	////////////////////////////////////

		int dx;

		public void mouseDown(MouseEvent e) {
			setDirty(true);

			dx = e.x;
			
			if (horizontal) {
				ref_min_graph.setXSlider((TimeZoomSlider)slider);
				ref_maj_graph.setXSlider((TimeZoomSlider)slider);
//				indicator_graph.setXSlider((TimeZoomSlider)slider);
				
//				Button button = (Button)button_map.get(slider);
//				if (button != null && update_thread != null) {
//					if (button.getSelection() && update_thread.isAlive()) {
//						indicator_graph.setIndicatorLocation(Double.MAX_VALUE);
//					} else {
////						indicator_graph.setIndicatorLocation(slider.pixel2Value(e.x));
//						indicator_graph.setIndicatorLocation(e.x);
//					}
//				} else {
//					indicator_graph.setIndicatorLocation(slider.pixel2Value(e.x));
//				}
				
			} else {
				ref_min_graph.setYSlider(slider);
				ref_maj_graph.setYSlider(slider);
			}
			ne_panel.redraw();
		}
		public void mouseUp(MouseEvent e) {
			setDirty(true);
			
			//mouse dragged to right
			if (dx < e.x && RIGHT_DRAG_RELEASES_BUTTON) {
				Button button = (Button)button_map.get(slider);
				if (button != null) {
					button.setSelection(false);	
				}
			}
			
			if (horizontal) {
				ref_min_graph.setXSlider((TimeZoomSlider)slider);
				ref_maj_graph.setXSlider((TimeZoomSlider)slider);
				
				if (dx == e.x && e.button <= 1) {
					indicator_graph.setXSlider((TimeZoomSlider)slider);
	
					Button button = (Button)button_map.get(slider);
					if (button != null && update_thread != null) {
						if (button.getSelection() && update_thread.isAlive()) {
							indicator_graph.setIndicatorLocation(Double.MAX_VALUE);
						} else {
//						indicator_graph.setIndicatorLocation(slider.pixel2Value(e.x));
							indicator_graph.setIndicatorLocation(e.x);
						}
					} else {
//						indicator_graph.setIndicatorLocation(slider.pixel2Value(e.x));
						indicator_graph.setIndicatorLocation(e.x);
					}
				}
				
			} else {
				ref_min_graph.setYSlider(slider);
				ref_maj_graph.setYSlider(slider);
			}
			

			//drag means no menu popup
			if (e.button > 1 && dx == e.x) {
//EditorPlugin.DBG.info("Button = "+e.button);
				initMenuEnabled();
				initMenuText();

				selected_slider = (ZoomSlider)e.widget;
				if (horizontal) {
					selected_slider_value = selected_slider.pixel2Value(e.x);
				} else {
					selected_slider_value = selected_slider.pixel2Value(e.y);
				}
								
//no longer any need for the bar menu				
//				ZoomControlBarsCollection collection = slider.getZoomControlBarsCollection();
//				ZoomControlBar bar = collection.getZoomControlBarContaining(e.x,e.y);
				
//				if (bar == null) {
					menu.setVisible(true);
//				} else {
//					selected_bar = bar;
//					menu_bar.setVisible(true);	
//				}
			}
			ne_panel.redraw();
		}
		public void mouseDoubleClick(MouseEvent e) {
		}
	}

	class SliderMinValidator implements IInputValidator {
		ZoomSlider slider;
		public SliderMinValidator(ZoomSlider slider) {
			this.slider = slider;
		}
		public String isValid(String s) {
			if (s == null) {
				return EditorPlugin.getString("NULL_ARGUMENT");	
			}
			if (s.length() == 0) {
				return EditorPlugin.getString("STRING_TOO_SHORT");	
			}
			try {
				double d = Double.parseDouble(s);
				
				if (d >= (slider.getMaxLimit()-slider.getIncrement())) {
					return EditorPlugin.getString("VALUE_TOO_LARGE")+" ("+(slider.getMaxLimit()-slider.getIncrement())+")";
				}
			} catch (Throwable e) {
				return EditorPlugin.getString("NOT_A_VALID_NUMBER");
			}
			
			return null;
		}
	}

	class SliderMaxValidator implements IInputValidator {
		ZoomSlider slider;
		public SliderMaxValidator(ZoomSlider slider) {
			this.slider = slider;
		}
		public String isValid(String s) {
			if (s == null) {
				return EditorPlugin.getString("NULL_ARGUMENT");	
			}
			if (s.length() == 0) {
				return EditorPlugin.getString("STRING_TOO_SHORT");	
			}
			try {
				double d = Double.parseDouble(s);
				
				if (d <= (slider.getMinLimit()+slider.getIncrement())) {
					return EditorPlugin.getString("VALUE_TOO_SMALL")+" ("+(slider.getMinLimit()+slider.getIncrement()+")");
				}
			} catch (Throwable e) {
				return EditorPlugin.getString("NOT_A_VALID_NUMBER");
			}
			
			return null;
		}
	}

	class SliderTitleValidator implements IInputValidator {
		public String isValid(String s) {
			if (s == null) {
				return EditorPlugin.getString("NULL_ARGUMENT");	
			}
			if (s.length() == 0) {
				return EditorPlugin.getString("STRING_TOO_SHORT");	
			}
			
			return null;
		}
	}

}