package org.eclipse.hyades.logging.java;

/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

import java.io.UnsupportedEncodingException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.logging.ErrorManager;
import java.util.logging.Filter;
import java.util.logging.Formatter;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.LogManager;
import java.util.logging.LogRecord;

import org.eclipse.hyades.logging.core.LoggingAgent;

/** 
 * Extension of the Java Logging <code>java.util.logging.Handler</code>
 * class used by <code>java.util.logging.Logger</code>s to publish logged
 * <code>java.util.logging.LogRecord</code>s to a Logging Agent.
 * <p>
 * An instance of this class will be returned from the <code>getHandlers()</code> API on 
 * <code>java.util.logging.Logger</code>s.  
 * The default <code>java.util.logging.LogManager</code> implementation
 * uses the following configuration variable in the default <JRE v1.4.x>/lib/logging.properties configuration
 * file to load configuration for <code>java.util.logging.Logger</code>s:
 * <p>
 * handlers = <existing handlers>, org.eclipse.hyades.logging.java.LoggingAgentHandler
 * <p>
 * Alternatively, an instantiation of this handler class may be set directly to
 * runtime <code>java.util.logging.Logger</code>s by using the <code>addHandler()</code> API.
 * <p>
 * NOTE:  The Java Logging classes (e.g. JDK v1.4.x and above) must be on the CLASSPATH at runtime to 
 * utilize this handler class
 *  
 * 
 * @author		Paul Slauenwhite
 * @author		Richard Duggan
 * @version	    January 29, 2004
 * @see		    java.util.logging.Handler
 * @see		    java.util.logging.LogManager
 */

public class LoggingAgentHandler extends Handler {

    /**
     * Default Logging Agent name.
     */
    public static final String DEFAULT_LOGGER_NAME = "Default Logging IRemoteEntity";

    /**
     * All created Logging Agents persisted for future recovery.
     */
    private HashMap loggingAgents = new HashMap();

    /**
     *  Current thread lock for synchronized operations. 
     */
    private Object currentThreadLock = new Object();

    /**
     * Constructor to create a Logging Agent Handler.  
     * 
     * The configuration for this type of handler is retrieved from the 
     * <code>java.util.logging.LogManager</code> and set to this instance
     * of this handler.  Configuration includes:
     * 
     * -Level
     * -Filter
     * -Formatter
     * -Encoding
     */
    public LoggingAgentHandler() {

        //Retrieve the LogManager:
        LogManager logManager = LogManager.getLogManager();

        //Retrieve the class' name:
        String className = this.getClass().getName();

        //Retrieve the handler's level based on the org.eclipse.hyades.logging.java.LoggingAgentHandler.level property:
        String defaultProperty = logManager.getProperty(className.concat(".level"));

        //If the handlers level has not been set then we will set it to Level.INFO:
        if (defaultProperty == null) {
            setLevel(Level.INFO);
        }
        else {
            //The level is set:
            try {
                setLevel(Level.parse(defaultProperty.trim()));
            }
            catch (Throwable t) {
                setLevel(Level.INFO);
            }
        }

        //Retrieve the handler's filter based on the org.eclipse.hyades.logging.java.LoggingAgentHandler.filter property:
        defaultProperty = logManager.getProperty(className.concat(".filter"));

        //If the filter is not specified then do no filtering.  Otherwise instantiate our filter:
        if (defaultProperty == null) {
            setFilter(null);
        }
        else {
            try {

                Class filterClass = ClassLoader.getSystemClassLoader().loadClass(defaultProperty.trim());

                setFilter((Filter) (filterClass.newInstance()));
            }
            catch (Throwable t) {
                setFilter(null);
            }
        }

        //Retrieve the handler's formatter based on the org.eclipse.hyades.logging.java.LoggingAgentHandler.formatter property:
        defaultProperty = logManager.getProperty(className.concat(".formatter"));

        //If the formatter is not set then we use the XmlFormatter:
        if (defaultProperty == null) {
            setFormatter(new XmlFormatter());
        }
        else {
            try {

                Class formatterClass = ClassLoader.getSystemClassLoader().loadClass(defaultProperty.trim());

                setFormatter((Formatter) (formatterClass.newInstance()));
            }
            catch (Throwable t) {
                setFormatter(new XmlFormatter());
            }
        }

        //Retrieve the handler's encoding based on the org.eclipse.hyades.logging.java.LoggingAgentHandler.encoding property:
        defaultProperty = logManager.getProperty(className.concat(".encoding"));

        //If the encoding is not specified we set it to be null:
        if (defaultProperty == null) {
            try {
                setEncoding(null);
            }
            catch (UnsupportedEncodingException u) {
            }
        }
        else {
            try {
                setEncoding(defaultProperty.trim());
            }
            catch (Throwable t) {
                try {
                    setEncoding(null);
                }
                catch (UnsupportedEncodingException u) {
                }
            }
        }
    }

    /**
     * Publishing a <code>java.util.logging.LogRecord</code> to a Logging Agent.
     * 
     * The <code>java.util.logging.LogRecord</code> is converted to XML using the handler's 
     * formatter (e.g. <code>org.eclipse.hyades.logging.java.XmlFormatter</code>) and 
     * propagated on to the handler's Logging Agent.
     * 
     * The Logging Agent is created or retrieved (e.g. currently exists) by using the 
     * <code>java.util.logging.LogRecord</code> source logger name or a default Logging
     * Agent name if one does not exist.
     * 
     * @param logRecord The <code>java.util.logging.LogRecord</code> to be published to a Logging Agent.
     */
    public void publish(LogRecord logRecord) {

        //Checks if the LogRecord has an appropriate Level and whether it satisfies any Filter:
        if (isLoggable(logRecord)) {

            //We have to log the LogRecord.  Lets call the formatters:
            String formattedLogRecord = null;
            Formatter formatter = getFormatter();

            if (formatter != null) {
                try {
                    formattedLogRecord = formatter.format(logRecord);
                }
                catch (Exception e) {
                    writeError(null, e, ErrorManager.FORMAT_FAILURE);
                    return;
                }
            }
            else {
                formattedLogRecord = logRecord.toString();
            }

            //Find the Logging Agent we will write to, creating one if necessary:
            LoggingAgent logger = null;

            String loggerName = logRecord.getLoggerName();

            if ((loggerName != null) && (loggerName.trim().length() != 0)) {

                synchronized (currentThreadLock) {

                    if (loggingAgents.containsKey(loggerName)) {
                        logger = ((LoggingAgent) (loggingAgents.get(loggerName)));
                    }
                    else {

                        //If we have never seen this Logging Agent before lets register it in the runtime:
                        logger = new LoggingAgent(loggerName);

                        loggingAgents.put(loggerName, logger);
                    }
                }
            }
            else {

                synchronized (currentThreadLock) {

                    if (loggingAgents.containsKey(DEFAULT_LOGGER_NAME)) {
                        logger = ((LoggingAgent) (loggingAgents.get(DEFAULT_LOGGER_NAME)));
                    }
                    else {

                        //If we have never seen this Logging Agent before lets register it in the runtime:
                        logger = new LoggingAgent(DEFAULT_LOGGER_NAME);

                        loggingAgents.put(DEFAULT_LOGGER_NAME, logger);
                    }
                }
            }

            logger.write(formattedLogRecord);
        }
    }

    /**
     * Flush any waiting messages to the Logging Agent.
     */
    public void flush() {

        Iterator agents = loggingAgents.values().iterator();

        while (agents.hasNext()) {
            ((LoggingAgent) (agents.next())).flush();
        }
    }

    /**
     * Flush any waiting messages and de register the Logging Agent.
     */
    public void close() {

        Iterator agents = loggingAgents.values().iterator();

        while (agents.hasNext()) {

            LoggingAgent current = ((LoggingAgent) (agents.next()));

            current.flush();
            current.deregister();
        }
    }

    /**
     * Private utility method for retrieving the <code>java.util.logging.ErrorManager</code>
     * and writing an error message, exception and error code.
     * 
     * If the <code>java.util.logging.ErrorManager</code> can not be retrieved, the 
     * resultant exception and stack trace are written to standard error.
     * 
     * @param message The error message to be written to the <code>java.util.logging.ErrorManager</code>.
     * @param exception The error exception to be written to the <code>java.util.logging.ErrorManager</code>.
     * @param code The error code to be written to the <code>java.util.logging.ErrorManager</code>.
     */
    private void writeError(String message, Exception exception, int code) {

        try {
            getErrorManager().error(message, exception, code);
        }
        catch (Throwable t) {
            System.err.println("Error: " + t.getMessage());
            t.printStackTrace();
        }
    }
}
