package org.eclipse.hyades.logging.java;

/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

import java.util.logging.Formatter;
import java.util.logging.LogRecord;

import org.eclipse.hyades.internal.logging.core.XmlGenerator;
import org.eclipse.hyades.logging.core.IExternalizableToXml;

/** 
 * Extension of the Java Logging <code>java.util.logging.Formatter</code>
 * class used by <code>java.util.logging.Handler</code>s format 
 * <code>java.util.logging.LogRecord</code>s to XML for consumption by a Logging Agent.
 * <p>
 * An instance of this class will be returned from the <code>getFormatters()</code> API on 
 * <code>java.util.logging.Logger</code>s.  
 * The default <code>java.util.logging.LogManager</code> implementation
 * uses the following configuration variable in the default <JRE V1.4.0>/lib/logging.properties configuration
 * file to load configuration for <code>java.util.logging.Logger</code>s:
 * <p>
 * <handler name>.formatter = org.eclipse.hyades.logging.java.XmlFormatter
 * <p>
 * Alternatively, an instantiation of this filter class may be set directly to
 * runtime <code>java.util.logging.Handler</code>s by using the <code>setFormatter()</code> API.
 * <p>
 * NOTE:  The Java Logging classes (e.g. JDK V1.4.0 and above) must be on the CLASSPATH at runtime to utilize this filter class
 *  
 * 
 * @author		Paul Slauenwhite
 * @version	March 7, 2003
 * @see		java.util.logging.Formatter
 * @see		java.util.logging.LogRecord
 */

public class XmlFormatter extends Formatter {

    /**
     * Generates the XML representation of the parameter <code>java.util.logging.LogRecord</code>
     * for consumption by a Logging Agent.
     * 
     * @return The XML representation of the parameter <code>java.util.logging.LogRecord</code>.
     */
    public String format(LogRecord record) {

        if (record instanceof IExternalizableToXml)
            return ((IExternalizableToXml) record).externalizeCanonicalXmlString();

        else {
            XmlGenerator xmlGenerator = new XmlGenerator(record.getLoggerName());
            xmlGenerator.reset(record.getLevel().getName(), false, 4);
            return xmlGenerator.objectToXML(record);
        }
    }

}
