/**********************************************************************
 * Copyright (c) 2003, 2004 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM Corporation - initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.internal.execution.recorder.local.appadapters;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.util.StringTokenizer;


/**
 * This object is the client object for a RecorderApplication extension point (point org.eclipse.hyades.execution.recorder.RecorderApplication)
 * it contains logic for initializing, starting, and stopping Internet Explorer.  In particular, this object is able
 * to modify system registry settings to cause Internet Explorer to use a particular host and port for proxy recording.
 * This adapter only works on Win32.
 * @author Ernest Jessee
 */
public class InternetExplorerAdapter implements IRecorderApplicationAdapter
{	
	/**
	 * the Internet Explorer Process
	 */
	private Process ieProcess;
		
	/**
	 * filename for storing the unmodified registry settings 
	 */
	private String originalSettingsFileName = "IEConfig.reg";//$NON-NLS-1$
	// defect 78895 below - also enable java console
	private String consoleSettingsFileName = "IEConfigConsole.reg";//$NON-NLS-1$

	/**
	 * filname for storing the registry settings to be used while recording
	 */
	private String tempSettingsFileName = "tempIEConfig.reg";//$NON-NLS-1$
	
	private String startupURL = "about:blank";
	private String launchPath = "C:\\Program Files\\Internet Explorer\\iexplore.exe";
	private String portnum = null;
	/**
	 * init routine called by the recorder to have internet explorer prepared for recording
	 * @param String initInfo -- this is a comma delimeted string indicating which port to point IE at. (ie. port,450)
	 */
	// defect 78895 mdd 12/8/2004
	final String ProxySrvStr = "\"ProxyServer";
	final String BadCertStr = "\"WarnonBadCertRecving";
	final String BadCertStr2 = "\"WarnOnBadCertRecving";
	final String ZoneCrossStr = "\"WarnonZoneCrossing";
	final String ZoneCrossStr2 = "\"WarnOnZoneCrossing";
	final String JavaConsoleStr = "\"EnableJavaConsole";
	final String ZonesMixedContentStr = "\"1609";
	
	public void init(String initInfo) throws RecorderApplicationAdapterException
	{
			
		parseInitInfo(initInfo);	
		StringBuffer inputFileContents = getRegistryInfo();
		
		StringBuffer outputFileContents = new StringBuffer();
		// new mdd hyades 1.3 check for http
		boolean isProxyEnabled = checkProxyEnabled(inputFileContents);

		StringTokenizer fileTokenizer = new StringTokenizer(inputFileContents.toString(),"\r\n");//$NON-NLS-1$
		while(fileTokenizer.hasMoreTokens())
		{
			String line=fileTokenizer.nextToken();
			String newLine = line;
			if(line.startsWith("\"ProxyEnable"))//$NON-NLS-1$
			{
				StringTokenizer lineTokenizer = new StringTokenizer(line,":");//$NON-NLS-1$
				newLine=lineTokenizer.nextToken();
				newLine+=":00000001";//$NON-NLS-1$
				if (isProxyEnabled){
					newLine+="\r\n\"ProxyServer\"=\"http=localhost:" + portnum;//$NON-NLS-1$
					newLine += ";https=localhost:" + portnum + "\"";
				}
				else {
					newLine+="\r\n\"ProxyServer\"=\"socks=localhost:";//$NON-NLS-1$
					newLine+=portnum+"\"";//$NON-NLS-1$
				}				
			}
			else if(line.startsWith("\"ProxyHttp1.1"))//$NON-NLS-1$
			{
				StringTokenizer lineTokenizer = new StringTokenizer(line,":");//$NON-NLS-1$
				newLine=lineTokenizer.nextToken();
				newLine+=":00000001";//$NON-NLS-1$
								
			}
			else if(line.startsWith("\"ProxyServer"))//$NON-NLS-1$
			{
				//it has been found that if the target computer has never been pointed to a proxy.  This 
				//entry may not be present.  Therefore, we will ignore it if it is there and always create it from scratch.
				newLine=null;				
			}
			// Defect 82974 - No longer bypass proxy for any host - detect and inform User 
			//                it it not supported (later in code) 
			/*else if (line.startsWith("\"ProxyOverride"))
			{
				//don't want to bypass the proxy for any hosts.  Leave empty
				newLine= "\"ProxyOverride\"=\"\"";
		
			}*/
			else if ( (line.startsWith(BadCertStr)) || (line.startsWith(BadCertStr2)) ) {
				newLine = disableLine(line);
				
			}
			else if ((line.startsWith(ZoneCrossStr))||(line.startsWith(ZoneCrossStr2))){
				newLine = disableLine(line);
			}
			else if (line.startsWith(JavaConsoleStr)){
				newLine = enableConsoleLine(line);
			} 
			else if (line.startsWith(ZonesMixedContentStr))  {
				newLine = enableZonesLine(line);
			}
			else if (line.startsWith("[HKEY_CLASSES_ROOT\\Applications\\iexplore.exe\\shell\\open\\command]"))
			{ //find ie path
				line = fileTokenizer.nextToken();
				if (line.startsWith("@="))
				{
					launchPath = line.substring(2);
					if (launchPath.charAt(0) == '"' && launchPath.charAt(launchPath.length() - 1) == '"')
						launchPath = launchPath.substring(1, launchPath.length()-1);

					launchPath = launchPath.replaceAll("%1", "");
					int n = launchPath.indexOf("\\\\");  //replaceAll doesn't work for these cases
					int m = launchPath.indexOf("\\\"");
					while (n > -1 || m > -1)
					{						
						if (n > -1)
						{
							launchPath = launchPath.substring(0, n) + launchPath.substring(n+1, launchPath.length());
							
						}
						else if (m > -1) //wait until next loop (have to reassign indeces) 
						{ 
							launchPath = launchPath.substring(0, m) + launchPath.substring(m+1, launchPath.length());
							
						}	
						n = launchPath.indexOf("\\\\");
						m = launchPath.indexOf("\\\"");
					}
					System.out.println(launchPath);
				}
				newLine = null; //don't update this
			}
			if(newLine!=null)
				outputFileContents.append(newLine+"\r\n");//$NON-NLS-1$
		}

			
		
		createRegSettingsFile(outputFileContents);
		
		try
		{
			pushRegFile(tempSettingsFileName);
		}
		catch (Exception e)
		{
			throw new RecorderApplicationAdapterException(e.getMessage());
		}
		
	}


	/**
	 * causes a registry file to be pushed into the registry using regedit.
	 * @param String fileName
	 */
	private void pushRegFile(String fileName) throws InterruptedException, IOException
	{
		File file = new File(fileName);
		if (!file.exists())
			return;
		String pushFileCommandLine = "regedit /s "+fileName;//$NON-NLS-1$
		try {
		Process pusher = Runtime.getRuntime().exec(pushFileCommandLine,null);
		pusher.waitFor();
		}
		catch (InterruptedException e) {
			System.out.println("Interrupted");
		}
		
	}

	/**
	 * method which creates a registry settings file (*.reg)
	 * @param StringBuffer ouputFileContents
	 */
	private void createRegSettingsFile(StringBuffer outputFileContents)
	{
		
		File newFile = new File(tempSettingsFileName);		
				
		try
		{
			OutputStreamWriter outputWriter = new OutputStreamWriter(new FileOutputStream(newFile),"UnicodeLittle");//$NON-NLS-1$
			outputWriter.write(outputFileContents.toString());
			outputWriter.close();
		}
		catch (UnsupportedEncodingException e1)
		{
			e1.printStackTrace();
		}
		catch (FileNotFoundException e1)
		{
			e1.printStackTrace();
		} 
		catch (IOException e)
		{
			e.printStackTrace();
		}
		
	}


	/**
	 * method which exports the registry keys of interested into a file using regedit and then reads the file
	 * @return StringBuffer
	 */
	private StringBuffer getRegistryInfo()
	{		
		String[] regInfo = {"\"HKEY_CLASSES_ROOT\\Applications\\iexplore.exe\\shell\\open\\command\"",
				"\"HKEY_CURRENT_USER\\Software\\Microsoft\\Windows\\CurrentVersion\\Internet Settings\"",
				"\"HKEY_CURRENT_USER\\Software\\Microsoft\\Java VM\""};
		String cmdLine = "regedit /e "+originalSettingsFileName+ " ";//$NON-NLS-1$//$NON-NLS-2$
		String consolecmdLine = "regedit /e "+consoleSettingsFileName+ " ";//$NON-NLS-1$//$NON-NLS-2$

		InputStreamReader inputReader=null;
		File file = null;
		File consoleFile = null;
		Process registryChange = null;
		StringBuffer inputFileContents = new StringBuffer();
		try
		{
			file = new File(originalSettingsFileName);
			if (file.exists())
				file.delete();
			consoleFile = new File(consoleSettingsFileName);
			if (consoleFile.exists())
				consoleFile.delete();

			for (int i = 0; i < regInfo.length; i++)
			{
				if (i == 2){
					registryChange = Runtime.getRuntime().exec(consolecmdLine + regInfo[i],null);	
				} 
				else {
					registryChange = Runtime.getRuntime().exec(cmdLine + regInfo[i],null);
				}
				registryChange.waitFor();
				if (i == 2){
					inputReader= new InputStreamReader(new FileInputStream(consoleFile),"UTF-16");//$NON-NLS-1$
				} else 
				{
					inputReader= new InputStreamReader(new FileInputStream(file),"UTF-16");//$NON-NLS-1$
				}
				char[] inputBuf = new char[1000];
				int bytesRead = inputReader.read(inputBuf);
									
				while(bytesRead!=-1)
				{				
					String buf = new String(inputBuf);				
					inputFileContents.append(inputBuf,0,bytesRead);
					bytesRead = inputReader.read(inputBuf);
					
				}	
				inputReader.close();
			}
		}
		catch (IOException e)
		{
			e.printStackTrace();
		} 
		catch (InterruptedException e)
		{
			e.printStackTrace();
		}
		finally
		{
			if(inputReader!=null)
				try
				{
					inputReader.close();
				}
				catch (IOException e1)
				{
					e1.printStackTrace();
				}
		}
		return inputFileContents;
	}

	
	/**
	 * method called by the recorder to start the target application (Internet Explorer)
	 * @return Process
	 */
	public Process start() throws RecorderApplicationAdapterException 	
	{
		ieProcess = null;
		try
		{
			//TODO derive path to iexplore.exe using registry.
			ieProcess = Runtime.getRuntime().exec(launchPath + " " +  startupURL);//$NON-NLS-1$
		}
		catch (IOException e)
		{
			throw new RecorderApplicationAdapterException("unable to start Internet Explorer Process: "+e.getMessage());//$NON-NLS-1$
		}
		return ieProcess;
	}

	
	/**
	 * method called to restore the process back to its original state
	 */
	public void cleanup()throws RecorderApplicationAdapterException
	{
		try
		{
			pushRegFile(originalSettingsFileName);
			pushRegFile(consoleSettingsFileName);
		}
		catch (Exception e)
		{
			throw new RecorderApplicationAdapterException(e.getMessage());
		}
		File origfile = new File(originalSettingsFileName);
		if (origfile.exists())
			origfile.delete();
		File regfile = new File(tempSettingsFileName);
		if (regfile.exists()) 
			regfile.delete();
		File consolefile = new File(consoleSettingsFileName);
		if (consolefile.exists())
			consolefile.delete();
	}

	/**
	 * method called to stop the target process.
	 */
	public void stop() throws RecorderApplicationAdapterException
	{
		ieProcess.destroy();
	}
	
	//init format:
	//param,value;param,value
	//Params:
	//  port - proxy port
	//  startupURL - page to launch IE with (file protocol only)
	private void parseInitInfo(String initInfo)
	{
		portnum = "";
		startupURL = "";
		StringTokenizer tok = new StringTokenizer(initInfo, ";");
		while (tok.hasMoreTokens())
		{
			String curTokn = tok.nextToken();
			StringTokenizer tok2 = new StringTokenizer(curTokn, ",");
			String key = tok2.nextToken();
			String val = tok2.nextToken();
			if (key.equals("port"))
			{
				portnum = val;
				try {
					Integer.parseInt(portnum);
				}
				catch (NumberFormatException e) {
					//error condition, should maybe just exit? 
					portnum = "443"; 
				}
			}
			else if (key.equals("startupURL"))
				startupURL = val; 
		}
	}
	private boolean checkProxyEnabled(StringBuffer inputBuffer) 
	{
			boolean comparesOK = false;
			StringTokenizer fileTokenizer = new StringTokenizer(inputBuffer.toString(),"\r\n");//$NON-NLS-1$
			while(fileTokenizer.hasMoreTokens())
			{
				String line=fileTokenizer.nextToken();
				String newLine = line;
				// example ProxyServer is key, "http=markdunn:1000;socks=localhost:450"
				if(line.startsWith("\"ProxyEnable"))//$NON-NLS-1$
				{
					StringTokenizer lineTokenizer = new StringTokenizer(line,":");//$NON-NLS-1$
					newLine=lineTokenizer.nextToken();  // get ProxyEnable
					newLine=lineTokenizer.nextToken();
					if (newLine.equalsIgnoreCase("00000001"))
						comparesOK = true;
				
					break;				
				}
			}
		return comparesOK;
	}
	private String getProxyInfo(StringBuffer inputBuffer) 
	{
		boolean comparesOK = false;
		String proxyString = "";
		String httpString = "http=";
		StringTokenizer fileTokenizer = new StringTokenizer(inputBuffer.toString(),"\r\n");//$NON-NLS-1$
		while(fileTokenizer.hasMoreTokens())
		{
			String line=fileTokenizer.nextToken();
			String newLine = line;
			// example ProxyServer is key, "http=markdunn:1000;socks=localhost:450"
			if(line.startsWith("\"ProxyServer"))//$NON-NLS-1$
			{
				String tmp = "\"ProxyServer\"=";
				String modifiedLine = line.substring(tmp.length());
				String [] entireString = modifiedLine.split(";");
				int numVars = entireString.length;
				for (int i =0; i < numVars; i++){
					String tmpEntireString = entireString[i].replaceAll("\"","");
					int findIndex = tmpEntireString.indexOf(httpString);
					if (findIndex != -1) {
						// found the revisedString
						String tmpVal = "";
						tmpVal = tmpEntireString.substring(findIndex+httpString.length());
						proxyString = tmpVal;
					}
				}
			}
		}
		return proxyString;
	}
	// defect 78895 mdd 12/08/2004
	private String disableLine(String origLine) {
		String returnLine = "";
		String newLine = "";
		StringTokenizer lineTokenizer = new StringTokenizer(origLine,":");//$NON-NLS-1$
		newLine=lineTokenizer.nextToken();
		newLine+=":00000000";//$NON-NLS-1$
		returnLine = newLine;
		
		return returnLine;
	}
	private String enableConsoleLine(String origLine) {
		String returnLine = "";
		String newLine = "";
		StringTokenizer lineTokenizer = new StringTokenizer(origLine,":");//$NON-NLS-1$
		newLine=lineTokenizer.nextToken();
		newLine+=":01,00,00,00";//$NON-NLS-1$
		returnLine = newLine;
		
		return returnLine;
	}
	private String enableZonesLine(String origLine) {
		String returnLine = "";
		String newLine = "";
		StringTokenizer lineTokenizer = new StringTokenizer(origLine,":");//$NON-NLS-1$
		newLine=lineTokenizer.nextToken();
		newLine+=":00000000";//$NON-NLS-1$
		returnLine = newLine;

		return returnLine;
	}

}
