/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.uml2sd.ui.view;


import java.util.ArrayList;
import java.util.List;

import org.eclipse.hyades.uml2sd.ui.SDViewerPlugin;
import org.eclipse.hyades.uml2sd.ui.core.Frame;
import org.eclipse.hyades.uml2sd.ui.core.GraphNode;
import org.eclipse.hyades.uml2sd.ui.core.Lifeline;
import org.eclipse.hyades.uml2sd.ui.core.Metrics;
import org.eclipse.hyades.uml2sd.ui.core.SyncMessage;
import org.eclipse.hyades.uml2sd.ui.drawings.IColor;
import org.eclipse.hyades.uml2sd.ui.drawings.ISDPreferences;
import org.eclipse.hyades.uml2sd.ui.load.LoadersManager;
import org.eclipse.hyades.uml2sd.ui.preferences.SDViewPref;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.accessibility.ACC;
import org.eclipse.swt.accessibility.Accessible;
import org.eclipse.swt.accessibility.AccessibleAdapter;
import org.eclipse.swt.accessibility.AccessibleControlAdapter;
import org.eclipse.swt.accessibility.AccessibleControlEvent;
import org.eclipse.swt.accessibility.AccessibleEvent;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.ui.part.ViewPart;

/**
 * 
 * @author sveyrier
 *
 */
public class SDWidget extends ScrollView implements SelectionListener, IPropertyChangeListener,
								DisposeListener,ITimeCompressionListener{
	
	private Frame frame;
	private Image overView = null;
	private static Color foreGroundColor 	= null;
	private static Color backGroundColor 	= null;
	private static Display display 			= null;
	private Menu   menu						= null;
	protected MenuItem	zoomIn				= null;
	protected MenuItem 	zoomOut				= null;
	protected SDWidgetSelectionProvider selProvider	=null;
	protected float  zoomValue				= 1;
	protected boolean zoomInMode			= false;
	protected boolean zoomOutMode			= false;
	
	protected List selectedNodeList = null;
	protected boolean multiSelection =false;
	
	private ViewPart site=null;
	
	public GraphNode currentGraphNode = null;
	
	private TimeCompressionBar timeBar = null;
	
	private DiagramToolTip toolTip = null;
	
	private Accessible accessible=null;
	
	protected GraphNode toolTipNode;
	
	protected int focusedWidget = -1;
	
	public SDWidget(Composite c, int s)
	{
		super(c, s | SWT.NO_BACKGROUND);
		display = getDisplay();
		setOverviewEnabled(true);
		selectedNodeList = new ArrayList();
		selProvider=new SDWidgetSelectionProvider();
		SDViewPref.getInstance().addPropertyChangeListener(this);
		toolTip = new DiagramToolTip(getViewControl());
		super.addDisposeListener(this);
		
	    accessible =getViewControl().getAccessible(); 

        accessible.addAccessibleListener(new AccessibleAdapter() { 
                public void getName(AccessibleEvent e) { 
                        if (e.childID==ACC.CHILDID_SELF)
                        {
   //                     	e.result = "";
                        }
                        //Case toolTip
                        else if (e.childID==0)
                        {
                        	if (toolTipNode != null)
                			{
                				if (toolTipNode instanceof Lifeline)
                				{
                					Lifeline lifeline = (Lifeline)toolTipNode;
                					e.result =lifeline.getToolTipText();
                				}
                				else e.result =toolTipNode.getName();
                			}
                        }
                        else
                        {
                        	e.result =currentGraphNode.getName();
                        }
                } 
        }); 
	        
        accessible.addAccessibleControlListener(new AccessibleControlAdapter() { 
        	 public void getFocus(AccessibleControlEvent e) { 
                        if (focusedWidget == -1) e.childID = ACC.CHILDID_SELF; 
                        else e.childID = focusedWidget; 
        } 	
        	
                public void getRole(AccessibleControlEvent e) { 
                        switch (e.childID) { 
                                case ACC.CHILDID_SELF: e.detail = ACC.ROLE_CLIENT_AREA; break; 
                                case 0: e.detail = ACC.ROLE_TOOLTIP; break; 
                                case 1: e.detail = ACC.ROLE_LABEL; break; 
                        } 
                } 

                public void getState(AccessibleControlEvent e) { 
                        e.detail = ACC.STATE_FOCUSABLE; 
                        if (e.childID == ACC.CHILDID_SELF) { 
                                e.detail |= ACC.STATE_FOCUSED; 
                        } else { 
                                e.detail |= ACC.STATE_SELECTABLE; 
                                if (e.childID == focusedWidget) e.detail |= ACC.STATE_FOCUSED | ACC.STATE_SELECTED| ACC.STATE_CHECKED; 
                        } 
                } 
        });
	}
	
	public void setTimeBar (TimeCompressionBar bar)
	{
		if (bar != null)
		{
			timeBar = bar;
			timeBar.addTimeCompressionListener(this);
		}
	}
	
	protected void keyPressedEvent(KeyEvent event) 
	{
		if (event.keyCode == SWT.CTRL)
			multiSelection =true;
		if ((event.character==' ')&&((zoomInMode)||(zoomOutMode)))
		{
			int cx = Math.round((getContentsX()+getVisibleWidth()/2)/zoomValue);
			int cy = Math.round((getContentsY()+getVisibleHeight()/2)/zoomValue);
			if (zoomInMode)
			{
				if (zoomValue<64)
				zoomValue = zoomValue*(float)1.25;				
			}
			else zoomValue = zoomValue/(float)1.25;
			int x = Math.round(cx*zoomValue-getVisibleWidth()/2);
			int y = Math.round(cy*zoomValue-getVisibleHeight()/2);
			setContentsPos(x ,y);
			if (timeBar != null)
				timeBar.setZoom(zoomValue);
			//redraw also resize the scrollView content
			redraw();
		}
		super.keyPressedEvent(event);
	}
					
	protected void keyReleasedEvent(KeyEvent event) 
	{
		setFocus(-1);
		multiSelection = false;
		super.keyReleasedEvent(event);
		setFocus(1);
	}
	
	/**
	 * Resize the contents to insure the frame fit into the view
	 * @param frame the frame which will be drawn int the view
	 */
	public void resizeContents(Frame frame)
	{
		int width = Math.round((frame.getWidth()+2*Metrics.FRAME_H_MARGIN)* zoomValue);
		int height = Math.round((frame.getHeight()+2*Metrics.FRAME_V_MARGIN) * zoomValue);
		resizeContents(width,height);
	}
	
	protected boolean checkFocusOnChilds(Control childs)
	{
		if (childs instanceof Composite)
		{
			Control[] child=((Composite)childs).getChildren();
			for (int i=0; i< child.length;i++)
			{
				if (child[i].isFocusControl())
				{
					return true;
				}
				else checkFocusOnChilds(child[i]);
			}
		}
		return false;
	}
	
	public boolean isFocusControl()
	{
		Control[] child=getChildren();
		for (int i=0; i< child.length;i++)
		{
			if (child[i].isFocusControl())
			{
				return true;
			}
			else checkFocusOnChilds(child[i]);
		}
		return false;
	}
	
	/**
	 * The frame to render (the sequence diagram)
	 * @param theFrame the frame to display
	 */
	protected void setFrame(Frame theFrame, boolean resetPosition)
	{
		selectedNodeList.clear();
		selProvider.setSelection(new StructuredSelection());
		frame = theFrame;
		if (resetPosition)
		{
			setContentsPos(0,0);
			resizeContents(frame);
			redraw();
		}
		//prepare the old overview to be reused
		if (overView != null)
			overView.dispose();
		overView = null;
		resizeContents(frame);
	}
	
	/**
	 * Returns the current Frame (the sequence diagram container)
	 * @return the frame
	 */
	protected Frame getFrame()
	{
		return frame;
	}
	
	/**
	 * Returns the selection provider for the current sequence diagram
	 * @return the selection provider
	 */
	public ISelectionProvider getSelectionProvider()
	{
		return selProvider;
	}
	
	public boolean setContentsPos(int x, int y)
	{
		if (x<0)
			x=0;
		if (y<0)
			y=0;
		if (frame == null)
			return false;
		if (x+getVisibleWidth()>getContentsWidth())
			x=getContentsWidth()-getVisibleWidth();
		if (y+getVisibleHeight()>getContentsHeight())
			y=getContentsHeight()-getVisibleHeight();
		int x1 = Math.round(x / zoomValue);
		int y2 = Math.round(y / zoomValue);
		int width = Math.round(getVisibleWidth()/zoomValue);
		int height = Math.round(getVisibleHeight()/zoomValue);
		frame.updateIndex(x1, y2, width, height);

		return super.setContentsPos(x, y);
	}
	
	
	protected void contentsMouseHover(MouseEvent event)
	{
		GraphNode currentGraphNode = null;
		if (frame != null)
		{
			int x = Math.round(event.x / zoomValue);
			int y = Math.round(event.y / zoomValue);
			currentGraphNode = frame.getNodeAt(x, y);
			if (currentGraphNode != null)
			{
				toolTipNode=currentGraphNode;
				if (currentGraphNode instanceof Lifeline)
				{
					Lifeline lifeline = (Lifeline)currentGraphNode;
					toolTip.showToolTip(lifeline.getToolTipText());
					setFocus(0);
				}
				else 
				{
					toolTip.showToolTip(currentGraphNode.getName());
					setFocus(0);
				}
			}
			else toolTip.hideToolTip();
		}
	}
	
	protected void setFocus(int newFocusShape) { 
	    focusedWidget = newFocusShape; 
	    if (focusedWidget == -1) { 
	    	getViewControl().getAccessible().setFocus(ACC.CHILDID_SELF); 
	    } else { 
	    	getViewControl().getAccessible().setFocus(focusedWidget); 
	    } 
	} 
	
	protected void contentsMouseMoveEvent(MouseEvent e)
	{
		 toolTip.hideToolTip();
		 super.contentsMouseMoveEvent(e);
		 if (!(isFocusControl()||getViewControl().isFocusControl()))
		 {
			 Control[] child=getParent().getChildren();
			 for (int i=0; i< child.length;i++)
				{
					if (child[i].isFocusControl())
					{
						getViewControl().setFocus();
						break;
					}
				}
		 }
		 setFocus(-1); 
	}
	
	protected void contentsMouseUpEvent(MouseEvent event)
	{
		//Just in case the diagram highlight a time compression region
		//this region need to be released when clicking everywhere
		if (frame == null) {
			return;
		}
		frame.resetTimeCompression();
		
		if ((event.stateMask&SWT.CTRL)!=0)
		{
			multiSelection =true;
		}
		else multiSelection = false;
		
		if (((zoomInMode)||(zoomOutMode))&&(event.button ==1))
		{
			int cx = Math.round(event.x / zoomValue);
			int cy = Math.round(event.y / zoomValue);
			if (zoomInMode)
			{
				if (zoomValue<64)
				zoomValue = zoomValue*(float)1.25;				
			}
			else zoomValue = zoomValue/(float)1.25;
			int x = Math.round(cx*zoomValue-getVisibleWidth()/2);
			int y = Math.round(cy*zoomValue-getVisibleHeight()/2);
			setContentsPos(x ,y);
			if (timeBar != null)
				timeBar.setZoom(zoomValue);
			//redraw also resize the scrollView content
			redraw();
		}
		else// if (event.button ==1)
		{
			GraphNode node = null;
			if (frame != null)
			{
				int x = Math.round(event.x / zoomValue);
				int y = Math.round(event.y / zoomValue);
				node = frame.getNodeAt(x, y);
				if ((event.button ==1)||((node!=null)&&!node.isSelected()&&!multiSelection))
				{
					performSelection(node);
					currentGraphNode = node;
				}
//				setFocusNode(node);	
				redraw();
			}
		}
		super.contentsMouseUpEvent(event);
	}

	/**
	 * Highlight the given GraphNode<br>
	 * The GraphNode is then displayed using the system default selection color
	 * @param node the GraphNode to highlight
	 */
	protected void performSelection (GraphNode node)
	{	
		StructuredSelection selection;
		if (multiSelection)
		{
			if (node!=null)
			{
				if (selectedNodeList.contains(node))
				{
					removeSelection(node);
				}
				else 
				{
					addSelection(node);
				}
			}
			else return;
		}
		else
		{
			clearSelection();
			if(node != null)
			{
				addSelection(node);
			}
		} 
	}
	
	public List getSelection() {
		return selectedNodeList;
	}
	
	public void addSelection(GraphNode node)
	{
		if (node == null)
			return;
		selectedNodeList.add(node);
		node.setSelected(true);
		currentGraphNode = node;
		StructuredSelection selection  = new StructuredSelection(selectedNodeList);
		selProvider.setSelection(selection);
	}
	
	public void addSelection(List list)
	{
		selectedNodeList.addAll(list);
		for (int i = 0; i<list.size(); i++)
		{
			((GraphNode)list.get(i)).setSelected(true);
		}
		StructuredSelection selection  = new StructuredSelection(selectedNodeList);
		selProvider.setSelection(selection);
	}
	
//	public void setFocusNode(GraphNode node)
//	{
//		if (frame != null)
//			frame.setFocusNode(node);	
//	}
	
/*	public GraphNode getFocusNode()
	{
		if (frame != null)
			return frame.getFocusNode();
		else return null;
	}*/
	
	public void removeSelection(GraphNode node)
	{
		selectedNodeList.remove(node);
		node.setSelected(false);
		StructuredSelection selection  = new StructuredSelection(selectedNodeList);
		selProvider.setSelection(selection);
	}
	
	public void removeSelection(List list)
	{
		selectedNodeList.removeAll(list);
		for (int i = 0; i<list.size(); i++)
		{
			((GraphNode)list.get(i)).setSelected(false);
		}
		StructuredSelection selection  = new StructuredSelection(selectedNodeList);
		selProvider.setSelection(selection);
	}
	
	/**
	 * Clear the list of GraphNodes which must be drawn selected
	 *
	 */
	public void clearSelection()
	{
		for (int i = 0; i<selectedNodeList.size(); i++)
		{
			((GraphNode)selectedNodeList.get(i)).setSelected(false);
		}
		selectedNodeList.clear();
		selProvider.setSelection(new StructuredSelection());
	}
	
	public void setSite(ViewPart viewSite)
	{
		
		site=viewSite;
		site.getSite().setSelectionProvider(selProvider);
		String[] sdScropes={"org.eclipse.ui.UML2SDScope","org.eclipse.ui.globalScope"};//$NON-NLS-1$ //$NON-NLS-2$
		site.getSite().getKeyBindingService().setScopes(sdScropes);
	}
	
	/**
	 * 
	 * @param gc the context
	 */
	protected void drawContents(GC gc, int clipx, int clipy, int clipw, int cliph) 
	{
		if (frame == null)
		{
			gc.setBackground(getDisplay().getSystemColor(SWT.COLOR_WHITE));
			gc.fillRectangle(0,0,getVisibleWidth(),getVisibleHeight());
			gc.dispose();
			return;
		}
		else 
		{
			Frame.setUserPref(SDViewPref.getInstance());
		}
		update();
		Rectangle area = getClientArea();
		Image dbuffer = null;
		GC gcim = null;
			
		if (!SDViewerPlugin.debugDoubleBuffer())
		{
			try {
				dbuffer = new Image(getDisplay(), area.width, area.height);
			} catch (Exception e) {
				System.out.println(e.toString());
			}
			gcim = new GC(dbuffer);
		}
		else 
			gcim=gc;
		NGC context = new NGC (this, gcim);
		
		//Set the metrics to use for lifeline text and message text
		//using the Graphical Context
		Metrics.setLifelineFontHeight(context.getFontHeight(SDViewPref.getInstance().getFont(SDViewPref.PREF_LIFELINE)));
		Metrics.setLifelineFontWidth(context.getFontWidth(SDViewPref.getInstance().getFont(SDViewPref.PREF_LIFELINE)));
		Metrics.setLifelineWidth(SDViewPref.getInstance().getLifelineWidth());
		Metrics.setFrameFontHeight(context.getFontHeight(Frame.getUserPref().getFont(ISDPreferences.PREF_FRAME_NAME)));
		Metrics.setLifelineHeaderFontHeight(context.getFontHeight(Frame.getUserPref().getFont(ISDPreferences.PREF_LIFELINE_HEADER)));
		
		int syncMessFontH = context.getFontHeight(SDViewPref.getInstance().getFont(SDViewPref.PREF_SYNC_MESS));
		int syncMessRetFontH = context.getFontHeight(SDViewPref.getInstance().getFont(SDViewPref.PREF_SYNC_MESS_RET));
		int asyncMessFontH = context.getFontHeight(SDViewPref.getInstance().getFont(SDViewPref.PREF_ASYNC_MESS));
		int asyncMessRetFontH = context.getFontHeight(SDViewPref.getInstance().getFont(SDViewPref.PREF_ASYNC_MESS_RET));
		
		int messageFontHeight = 0;
		if (syncMessFontH>syncMessRetFontH)
			messageFontHeight = syncMessFontH;
		else messageFontHeight = syncMessRetFontH;
		if (messageFontHeight<asyncMessFontH)
			messageFontHeight=asyncMessFontH;
		if (messageFontHeight<asyncMessRetFontH)
			messageFontHeight=asyncMessRetFontH;
		Metrics.setMessageFontHeight(messageFontHeight);
		context.setFont(SDViewPref.getInstance().getFont(SDViewPref.PREF_LIFELINE));
		
		
		int width = Math.round((frame.getWidth()+2*Metrics.FRAME_H_MARGIN)* zoomValue);
		int height = Math.round((frame.getHeight()+2*Metrics.FRAME_V_MARGIN) * zoomValue);
/*		if (width<area.width)
			width=area.width;
		if (height<area.height)
			height=area.height;*/
		resizeContents(width,height);
		
		context.setBackground(Frame.getUserPref().getBackGroundColor(ISDPreferences.PREF_FRAME));
		context.fillRectangle(0,0,getContentsWidth(),Metrics.FRAME_V_MARGIN);
		context.fillRectangle(0,0,frame.getX(),getContentsHeight());
		context.fillRectangle(frame.getX()+frame.getWidth()+1,0,
								getContentsWidth()-(frame.getX()+frame.getWidth()+1),getContentsHeight());
		context.fillRectangle(0,frame.getY()+frame.getHeight()+1,getContentsWidth(),
							getContentsHeight()-(frame.getY()+frame.getHeight()+1));
		gcim.setLineWidth(1);
		
		frame.draw(context); 
		if (!SDViewerPlugin.debugDoubleBuffer())
		{
			try {
				gc.drawImage(dbuffer, 0, 0, area.width, area.height, 0, 0,  area.width, area.height);
			} catch (Exception e) {
				System.out.println(e.getMessage());
			}
			gcim.dispose();
			dbuffer.dispose();
		}
		gc.dispose();
		context.dispose();
		setHScrollBarIncrement(Math.round(SDViewPref.getInstance().getLifelineWidth()/2*zoomValue));
		setVScrollBarIncrement(Math.round(Metrics.getMessagesSpacing()*zoomValue));
		if ((timeBar != null)&&(frame.hasTimeInfo()))
		{
			timeBar.resizeContents(9,height+getHorizontalBarHeight());
			timeBar.setContentsPos(getContentsX(),getContentsY());
			timeBar.redraw();
			timeBar.update();
		}
		float xRatio=getContentsWidth()/getVisibleWidth();
		float yRatio=getContentsHeight()/getVisibleHeight();
		if (yRatio>xRatio)
		{
			setOverviewSize((int)(getVisibleHeight()*0.75));
		}
		else
		{
			setOverviewSize((int)(getVisibleWidth()*0.75));
		}
	} 
	
	/**
	 * Returns the GraphNode overView the mouse if any
	 * @return the GraphNode
	 */
	public GraphNode getMouseOverNode()
	{
		return currentGraphNode;
	}
	
	public void widgetDefaultSelected(SelectionEvent event)
	{
	}
	
	public void widgetSelected (SelectionEvent event)
	{
		if (event.widget == zoomIn)
			zoomValue = zoomValue*2;
		else if (event.widget == zoomOut)
			zoomValue = zoomValue/2;
		else {
			//SearchFilterDialog tt = new SearchFilterDialog(null);//display.getActiveShell());
		}
		redraw();
	}
	
	public void setZoomInMode(boolean value)
	{
		if (value)
			setZoomOutMode(false);
		zoomInMode = value;
	}
	
	public void setZoomOutMode(boolean value)
	{
		if (value)
			setZoomInMode(false);
		zoomOutMode = value;
	}
	
	/**
	 * Moves the Sequence diagram to ensure the given node 
	 * is visible and draw it selected
	 * @param node the GraphNode to move to
	 */
	public void moveTo(GraphNode node)
	{
		if (node == null)
			return;
		clearSelection();
		addSelection(node);
		ensureVisible(node);
	}
	
	/**
	 * Moves the Sequence diagram to ensure the given node 
	 * is visible
	 * @param node the GraphNode to move to
	 */
	public void ensureVisible(GraphNode node)
	{
		if (node == null)
			return;
		int x = Math.round(node.getX() * zoomValue);
		int y = Math.round(node.getY() * zoomValue);
		int width = node.getWidth();
		int height = node.getHeight();
		if (node instanceof Lifeline)
		{
			x = Math.round(node.getX() * zoomValue)+width/2;
			width=getVisibleWidth()/2;
		}
		if (node instanceof SyncMessage)
		{
			height = Metrics.getMessagesSpacing();
		}
		if (x < getVisibleWidth()) {
			x = 0;
		} else if (x + getVisibleWidth() > getContentsWidth()) {
			x = getContentsWidth() - width;
		}
		if (y < getVisibleHeight()) {
			y = height;
		} else if (y + getVisibleHeight() > getContentsHeight()) {
			y = getContentsHeight() + height;
		}
		if (node instanceof Lifeline)
		{
			y=getContentsY()+Metrics.LIFELINE_VT_MAGIN;
			height=getVisibleHeight()-Metrics.LIFELINE_VT_MAGIN;
		}
		ensureVisible(x,y-Metrics.LIFELINE_VT_MAGIN,width,height+Metrics.LIFELINE_VT_MAGIN,SWT.CENTER);
		redraw();
	}
	
	public float getZoomFactor()
	{
		return zoomValue; 
	}
	
	/**
	 * Called when property changed occurs in the preference page.
	 * "PREFOK" is fired when the user press the ok or apply button
	 * @param e the PropertyChangeEvent
	 */
	public void propertyChange(PropertyChangeEvent e)
	{
		if (e.getProperty().equals("PREFOK")) //$NON-NLS-1$
		{
			//Prepare the overview to be reused for the new
			//settings (especially the colors)
			if (overView!= null)
				overView.dispose();
			overView = null;
			redraw();
		}
	}
	
	public void widgetDisposed(DisposeEvent e)
	{
		if (overView!=null)
			overView.dispose();
		super.removeDisposeListener(this);
		SDViewPref.getInstance().removePropertyChangeListener(this);
		LoadersManager lm = LoadersManager.getLoadersManager();
		if (lm != null) {
			lm.setCurrentLoader(null);
		}
		if (site != null && site instanceof SDView) {
			((SDView)site).resetProviders();
		}
	}
	
	protected void drawOverview( GC gc, Rectangle r) 
	{
      float oldzoom=zoomValue;
      if (getContentsWidth() > getContentsHeight())
	  	zoomValue=(float)r.width/(float)getContentsWidth()*oldzoom;
	  else zoomValue=(float)r.height/(float)getContentsHeight()*oldzoom;
      if ((overView != null)&&((r.width!=overView.getBounds().width)||(r.height!=overView.getBounds().height)))
	  {
		  overView.dispose();
		  overView = null;
	  }
	  if (overView == null)
	  {  
		  int backX = getContentsX();
		  int backY = getContentsY();
		  setContentsPos(0,0);
		  overView = new Image(getDisplay(), r.width, r.height);
		  GC gcim = new GC(overView);
		  NGC context = new NGC (this, gcim);
			context.setBackground(SDViewPref.getInstance().getBackGroundColor(SDViewPref.PREF_FRAME));
		  frame.draw(context);
		  setContentsPos(backX,backY);
		  gcim.dispose();
		  context.dispose();
	  }
	  if ((overView != null)&&(r.width==overView.getBounds().width)&&(r.height==overView.getBounds().height))
	  	gc.drawImage(overView, 0, 0, r.width, r.height, 0, 0,  r.width, r.height);
	  
	  zoomValue=oldzoom;
	  
	  super.drawOverview(gc,r);
	}
	
	public void deltaSelected(Lifeline lifeline, int startEvent, int nbEvent, IColor color)
	{
		frame.highlightTimeCompression(lifeline,startEvent,nbEvent,color);
		ensureVisible(lifeline);
		redraw();
		update();
	}
	
	public void resetZoomFactor()
	{
		int currentX = Math.round(getContentsX()/zoomValue);
		int currentY = Math.round(getContentsY()/zoomValue);
		zoomValue = 1;
		if (timeBar != null && !timeBar.isDisposed()) {
			timeBar.setZoom(zoomValue);
		}
		redraw();
		update();
		setContentsPos(currentX,currentY);
	}
}
