/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui;

import java.util.Enumeration;
import java.util.Hashtable;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.ui.internal.editor.action.ActionHandlerListener;
import org.eclipse.hyades.ui.internal.navigator.INavigator;
import org.eclipse.jdt.ui.IContextMenuConstants;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.IPartListener2;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchPartReference;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.IPage;
import org.eclipse.ui.part.PageBook;
import org.eclipse.ui.part.ViewPart;

public abstract class TraceViewer extends ViewPart implements IMenuListener, ISelectionListener, IProfileEventListener, IDeleteListener {

	protected EObject _mofObject;
	protected boolean _initializedMenu = false;
	protected Hashtable _pages = new Hashtable();
	private PageBook book;
	private IPage currentPage;
	private IPage defaultPage;
	private IDialogSettings settings;

	protected Menu fContextMenu;

	public final String GROUP_ADD_VIEWS = "open.views";
	public final String GROUP_GENERIC = "generic.action";

	/*
	 * Keep track of the refresh events ignored because the page view not visible
	 */
	protected boolean fRefreshView = false;

	/*
	 * Keep track of the selection changed event ignored because the page view not visible
	 */
	//	 protected boolean fSelectionChanged = false;

	/*
	 * Ensure refresh events being processed only if this part is visible.
	 */
	protected boolean fPartVisible = true;

	protected IPartListener2 fPartListener = new IPartListener2() {
		public void partActivated(IWorkbenchPartReference ref) {
		}
		public void partBroughtToTop(IWorkbenchPartReference ref) {
		}
		public void partInputChanged(IWorkbenchPartReference ref) {
		}
		public void partClosed(IWorkbenchPartReference ref) {
		}
		public void partDeactivated(IWorkbenchPartReference ref) {
		}
		public void partOpened(IWorkbenchPartReference ref) {
		}
		public void partVisible(IWorkbenchPartReference ref) {

			if (ref != null && ref.getPart(false) == TraceViewer.this) {
				fPartVisible = true;
				if (fRefreshView) {
					fRefreshView = false;
					refreshVisiblePage();
				}
			}
		}
		public void partHidden(IWorkbenchPartReference ref) {

			if (ref != null && ref.getPart(false) == TraceViewer.this) {
				fPartVisible = false;
			}
		}
	};
	private int previousSelection=0;

	/**
	 * TraceViewer constructor comment.
	 */
	public TraceViewer() {
		super();

	}

	/**
	 * Add a page to the workbook. If the mofObject is null, show default empty page
	 */
	public void addViewPage(EObject object) {
		
//		System.err.println("TraceViewer.addViewPage() " +(new java.util.Date().getTime()));
//		new java.lang.Exception().printStackTrace();
		if (book == null) {
			//the widget was disposed...
			//remove the view from the pd listeners
			UIPlugin.getDefault().removeSelectionListener(this);
			return;
		}

		_mofObject = getObjectToView(object);
		
		setViewTitle(_mofObject);

		if (_mofObject == null) {
			showPage(defaultPage);
			return;
		}

		boolean newPage = false;
		TraceViewerPage page = getPage(_mofObject);
		if (defaultPage == page)
			return;

		if (page == null) {

			//remove existing pages
			Enumeration pages = _pages.elements();
			while (pages.hasMoreElements()) {
				((IPage) pages.nextElement()).dispose();
			}
			_pages.clear();

			makeActions();
			page = createPage(_mofObject);
			page.createControl(book);

			_pages.put(_mofObject, page);
			newPage = true;

		}
		//		page.refreshPage();
		if(newPage)
			page.update(newPage);

		if (page.isEmpty())
			showPage(defaultPage);
		else
			showPage(page);

	}

	/*
	 * Returns the object that should be viewd if true if the selObject is selected in the monitors view For example, if the process node is selected, the view should display the profiling data if trace collects profiling data
	 *  
	 */
	public EObject getObjectToView(EObject selObject) {
		return selObject;
	}

	/**
	 * Returns the default page.
	 * 
	 * @return the default page.
	 */
	protected IPage createDefaultPage(PageBook book) {
		DefaultPage page = new DefaultPage();
		page.createControl(book);
		page.setMessage(UIPlugin.getResourceString("STR_NODATA"));
		return page;
	}
	public abstract TraceViewerPage createPage(EObject mofObject);
	/**
	 * Implementation of IDesktopPart.createPartControl(Composite)
	 */
	public void createPartControl(Composite parent) {
		// Create the page book.
		book = new PageBook(parent, SWT.NONE);

		// Create the default page.
		defaultPage = createDefaultPage(book);

		//ask the view to display the data for the curently selected object
//		INavigator navigator = HyadesUtil.getActiveNavigator(); 
//		if(navigator!=null && navigator.isLinkingEnabled())
		addViewPage(HyadesUtil.getMofObject());
//		else
//			showPage(defaultPage);

		UIPlugin.getDefault().addSelectionListener(this);
		UIPlugin.getDefault().addProfileEventListener(this);
		UIPlugin.getDefault().addDeleteListener(this);

		getViewSite().getPage().addPartListener(fPartListener);

		settings = UIPlugin.getDefault().getDialogSettings();
	}
	/**
	 * Insert the method's description here. Creation date: (2/19/2001 2:52:36 PM)
	 * 
	 * @return com.ibm.eclipse.ui.part.IPage
	 */
	public IPage getCurrentPage() {
		if (currentPage == null || !(currentPage instanceof TraceViewerPage))
			return null;

		return currentPage;
	}
	/**
	 * Insert the method's description here. Creation date: (08/01/2000 4:01:53 PM)
	 */
	public TraceViewerPage getPage(EObject mofObject) {

		if (mofObject != null) {
			Object page = _pages.get(mofObject);
			if (page != null)
				return (TraceViewerPage) page;
		}

		return null;
	}
	/**
	 *  
	 */
	public void dispose() {
		settings = null;
		UIPlugin.getDefault().removeSelectionListener(this);
		UIPlugin.getDefault().removeProfileEventListener(this);
		UIPlugin.getDefault().removeDeleteListener(this);

		getSite().getPage().removePartListener(fPartListener);

		Enumeration pages = _pages.elements();
		while (pages.hasMoreElements()) {
			((IPage) pages.nextElement()).dispose();
		}

		_pages.clear();

		if (book != null) {
			book.dispose();
			book = null;
		}

		if (currentPage != null) {
			currentPage.dispose();
			currentPage = null;
		}

		if (defaultPage != null) {
			defaultPage.dispose();
			defaultPage = null;
		}

		fPartListener = null;

		super.dispose();
	}

	/**
	 * Insert the method's description here. Creation date: (2/19/2001 3:37:02 PM)
	 * 
	 * @param init
	 *            boolean
	 */
	public void initializedMenu(boolean init) {
		_initializedMenu = init;
	}
	/**
	 * Answer true if we want to track the given page.
	 */
	protected boolean isImportant(IWorkbenchPart part) {
		//We only care about editors
		return true;
	}
	/**
	 * Insert the method's description here. Creation date: (2/19/2001 3:37:34 PM)
	 * 
	 * @return boolean
	 */
	public boolean isInitializedMenu() {
		return _initializedMenu;
	}
	/**
	 * isSaveNeeded method comment.
	 */
	public boolean isSaveNeeded() {
		return false;
	}
	/**
	 * Insert the method's description here. Creation date: (1/11/2001 2:54:24 PM)
	 */
	public void makeActions() {
	}
	/**
	 * Insert the method's description here. Creation date: (6/5/2001 2:38:58 PM)
	 * 
	 * @param obj
	 *            java.lang.Object
	 */
	protected void refreshPage(Object obj) {

		if (_pages == null)
			return;

		if (obj != null) {
			Object page = _pages.get(obj);
			if(page == null )
			{
				fRefreshView = true;
				return;
			}
			if (page instanceof TraceViewerPage) {
//				INavigator navigator = HyadesUtil.getActiveNavigator();
//				if (navigator != null && navigator.isLinkingEnabled())
					 ((TraceViewerPage) page).refreshPage();

				if (((TraceViewerPage) page).isEmpty()) {
					showPage(defaultPage);
				} else {
					showPage((TraceViewerPage) page);
				}
			}
		}

	}

	/**
	 * Insert the method's description here. Creation date: (6/5/2001 2:38:58 PM)
	 * 
	 * @param obj
	 *            java.lang.Object
	 */
	protected void selectionChanged(Object obj) {

		if (obj == null)
			return;

		Object page = _pages.get(obj);

		if (page != null && page instanceof TraceViewerPage) {
			((TraceViewerPage) page).selectionChanged();
		}

	}

	/**
	 * Insert the method's description here. Creation date: (08/01/2000 4:01:53 PM)
	 */
	public void removePage(Object obj) {

		if (obj == null || !(obj instanceof EObject))
			return;

		Object page = _pages.get(getObjectToView((EObject) obj));
		_pages.remove(obj);

		if (page != null) {
			if (page == currentPage) {
				showPage(defaultPage);
			}

			((IPage) page).dispose();
		}
	}
	/**
	 * Asks the part to take focus within the desktop.
	 */
	public void setFocus() {
	}
	/**
	 * Insert the method's description here. Creation date: (2/19/2001 2:53:35 PM)
	 * 
	 * @param page
	 *            com.ibm.eclipse.ui.part.IPage
	 */
	public void showPage(IPage page) {
		book.showPage(page.getControl());
		currentPage = page;

	}

	/*
	 * Selection has changed in the Monitors view (PDProjectExplorer)
	 */
	public void selectionChanged(IWorkbenchPart part, ISelection selection) {
		
		fRefreshView = true;
		boolean openView = ((INavigator) part).isLinkingEnabled();
		if (!fPartVisible || part == null || part.getSite().getWorkbenchWindow() != getSite().getWorkbenchWindow()) {
			return; //get notifications only from the current workbench window

		}

		if (!openView) {
			//			fSelectionChanged = true;
			return;
		}
		
		if (selection == null || selection.isEmpty() || !(selection instanceof IStructuredSelection)) {
			//show empty page
			addViewPage((EObject) null);
			return;
		}

		Object obj = ((IStructuredSelection) selection).getFirstElement();
		if (obj == null || !(obj instanceof EObject))
			addViewPage((EObject) null);
		else
		{	
			if(previousSelection != obj.hashCode())
			{
				fRefreshView = true;
				previousSelection = obj.hashCode();
			}
			else
				fRefreshView=false;
			addViewPage((EObject) obj);
		}
	}

	public void init(IViewSite site) throws PartInitException {

		super.init(site);
	}
	public void init(IViewSite site, IMemento memento) throws PartInitException {

		super.init(site, memento);
		enableEditActions();
	}

	/*
	 * Initial view title
	 */
	public abstract String getViewTitle();

	public void handleProfileEvent(ProfileEvent event) {

		int type = event.getType();

		if (!fPartVisible) {
			//ignore the events and keep track that the page need to be updated
			if (type == ProfileEvent.REFRESH_VIEWS || type == ProfileEvent.UPDATE_MODEL)
				fRefreshView = true;

			return;
		} else {
			fRefreshView = false;

			if ((type == ProfileEvent.REFRESH_VIEWS || type == ProfileEvent.UPDATE_MODEL) && event.getSource() instanceof EObject) {
				INavigator nav = HyadesUtil.getActiveNavigator();
				if (nav == null || nav.isLinkingEnabled() || _mofObject == null) {
					refreshPage(getObjectToView((EObject)event.getSource()));
				}
				else {
					refreshPage(getObjectToView(_mofObject));
				}
			}
		}
	}

	protected void setViewTitle(Object selection) {
		if (selection != null) {
			if (selection instanceof TRCAgentProxy) {
				TRCAgentProxy a = (TRCAgentProxy) selection;

				setTitle(getViewTitle() + " - " + HyadesUtil.getProcessName(a.getProcessProxy()));
				return;

			}
		}
		setTitle(getViewTitle());
	}

	public void initializeActionBar() {
	}

	public void setRecordSelection(EObject record, EObject treeSel) {
	};

	/**
	 * @return
	 */
	public boolean isProcessRefreshEvents() {
		return fPartVisible;
	}

	/**
	 * Refresh the visible page when the view has become visible
	 *  
	 */
	private void refreshVisiblePage() {
		EObject obj = HyadesUtil.getMofObject();
		addViewPage(obj);
		EObject sel = getObjectToView(obj);
		refreshPage(sel);

		selectionChanged(sel);
	}

	/**
	 * @see IMenuListener
	 */
	public void menuAboutToShow(IMenuManager menu) {

		if (getCurrentPage() != null && getCurrentPage() instanceof TraceViewerPage) {
			((TraceViewerPage) getCurrentPage()).fillContextMenu(menu);
		}

		menu.add(new Separator(GROUP_ADD_VIEWS));
		menu.add(new Separator(GROUP_GENERIC));

		menu.add(new Separator(IContextMenuConstants.GROUP_ADDITIONS));

	}

	/**
	 * Invoked when a top hierarchy object is deleted from the model.
	 */
	public void deregister(Object obj) {
		removePage(obj);
	}

	protected void enableEditActions() {
		ActionHandlerListener.DEFAULT.connectPart(this);
	}

}
