/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.util;

import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceStatus;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.hyades.models.hierarchy.CorrelationContainerProxy;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.trace.internal.ui.PDContentProvider;
import org.eclipse.hyades.trace.ui.HyadesUtil;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.ui.extension.INavigatorItem;
import org.eclipse.hyades.ui.internal.extension.NavigatorExtensionUtil;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;

/**
 * Utility methods for saving navigator items.
 */
public class SaveUtil {
	
	private static final String MONITOR = UIPlugin.getResourceString("SAVE_MONITOR_TEXT");
	private static final String NODE = UIPlugin.getResourceString("SAVE_NODE_TEXT");
	private static final String PROCESS = UIPlugin.getResourceString("SAVE_PROCESS_TEXT");
	private static final String AGENT = UIPlugin.getResourceString("SAVE_AGENT_TEXT");
	private static final String CORRELATION = UIPlugin.getResourceString("SAVE_CORRELATION_TEXT"); 

	/**
	 * Save all the given resources. Will also recursively
	 * save the parents and children.
	 */
	public static void save(Collection c, IProgressMonitor monitor, String navigatorID) {
		Collection items = findItems(c, navigatorID);
		
		if (monitor != null)
			monitor.beginTask(UIPlugin.getResourceString("SAVE"), items.size());

		if (isValid(items)) {
			Iterator i = items.iterator();
			while (i.hasNext()) {
				saveItem(i.next(), monitor);
				if (monitor != null && monitor.isCanceled())
					break;
			}
		}
		
		if (monitor != null)
			monitor.done();
	}

	/**
	 * Save a single item (non-recursive).
	 */
	public static void saveItem(Object obj, IProgressMonitor monitor) {
		if (obj instanceof EObject) {
			EObject eobj = (EObject)obj;
			
			if (obj instanceof TRCMonitor) {
				try {
					org.eclipse.hyades.models.hierarchy.util.SaveUtil.saveResource(monitor, eobj.eResource());
				}
				catch (Exception e) {
					handleSaveException(MONITOR, ((TRCMonitor)obj).getName(), eobj);
				}
			}
			else if (obj instanceof TRCNode) {
				try {
					org.eclipse.hyades.models.hierarchy.util.SaveUtil.saveResource(monitor, eobj.eResource());
				}
				catch (Exception e) {
					handleSaveException(NODE, ((TRCNode)obj).getName(), eobj);
				}
			}
			else if (obj instanceof TRCProcessProxy) {
				try {
					org.eclipse.hyades.models.hierarchy.util.SaveUtil.saveResource(monitor, eobj.eResource());
				}
				catch (Exception e) {
					handleSaveException(PROCESS, ((TRCProcessProxy)obj).getName(), eobj);
				}
			}
			else if (obj instanceof TRCAgentProxy) {
				TRCAgentProxy agent = (TRCAgentProxy)obj;
				if (agent.getAgent() != null) {
					try {
						org.eclipse.hyades.models.hierarchy.util.SaveUtil.saveResource(monitor, agent.getAgent().eResource());
					}
					catch (Exception e) {
						handleSaveException(AGENT, agent.getName(), agent);
					}
				}
			}
			else if (obj instanceof CorrelationContainerProxy) {
				CorrelationContainerProxy proxy = (CorrelationContainerProxy)obj;
				if (proxy.getCorrelationContainer() != null) {
					try {
						if(!(proxy.getCorrelationContainer().eResource()!=null && proxy.getCorrelationContainer().eResource().getURI().toString().endsWith(".corrdb")))
						{
							org.eclipse.hyades.models.hierarchy.util.SaveUtil.saveResource(monitor, proxy.getCorrelationContainer().eResource());
						}
					}
					catch (Exception e) {
						handleSaveException(CORRELATION, proxy.getName(), eobj);
					}
				}
			}
		}
		else if (obj instanceof INavigatorItem) {
			if (monitor != null) {
				monitor.setTaskName(((INavigatorItem)obj).getText());
				monitor.worked(1);
			}
			INavigatorItem item = (INavigatorItem)obj;
			if (item.isSaveEnabled())
				item.save(true);
		}
	}

	/*
	 * Collect all items related to the given selection. This
	 * collects parents (ancestors) and children (descendants).
	 */
	private static Collection findItems(Collection selection, String navigatorID) {
		Collection c = new HashSet();
		Iterator i = selection.iterator();
		while (i.hasNext())
			findItems(i.next(), c, true, true, navigatorID);
		return c;
	}
	
	private static void findItems(Object obj, Collection c, boolean findAncestors, boolean findDescendants, String navigatorID) {
		if (obj != null && !c.contains(obj)) {
			if (!(obj instanceof IContainer))
				c.add(obj);
			
			if (findAncestors) {
				findItems(getParent(obj), c, true, false, navigatorID);
			}
			if (findDescendants) {
				Iterator i = getChildren(obj, navigatorID).iterator();
				while (i.hasNext())
					findItems(i.next(), c, false, true, navigatorID);
			}
		}
	}

	private static Object getParent(Object obj) {
		if (obj instanceof TRCNode)
			return ((TRCNode)obj).getMonitor();
		else if (obj instanceof TRCProcessProxy)
			return ((TRCProcessProxy)obj).getNode();
		else if (obj instanceof TRCAgentProxy)
			return ((TRCAgentProxy)obj).getProcessProxy();
		else if (obj instanceof CorrelationContainerProxy)
			return ((CorrelationContainerProxy)obj).getMonitor();
		else if (obj instanceof INavigatorItem)
			return ((INavigatorItem)obj).getParent();
		else
			return null;
	}
	
	private static Collection getChildren(Object obj, String navigatorID) {
		Collection c = new HashSet();

		if (obj instanceof IContainer) {
			IContainer container = (IContainer)obj;
			try {
				Object[] members = container.members();
				for (int i=0;i<members.length;++i)
					if (members[i] instanceof IContainer)
						c.add(members[i]);
			}
			catch (CoreException e) {
				e.printStackTrace();
			}

			c.addAll(PDContentProvider.getMonitors(container));
		}
		else if (obj instanceof TRCMonitor)
			c.addAll(((TRCMonitor)obj).getNodes());
		else if (obj instanceof TRCNode)
			c.addAll(((TRCNode)obj).getProcessProxies());
		else if (obj instanceof TRCProcessProxy)
			c.addAll(((TRCProcessProxy)obj).getAgentProxies());
		else if (obj instanceof CorrelationContainerProxy)
			c.addAll(((CorrelationContainerProxy)obj).getCorrelatedAgents());
		
		c.addAll(NavigatorExtensionUtil.getAllChildren(obj, navigatorID));
		return c;
	}

	/*
	 * Is this a valid collection to save?
	 */
	private static boolean isValid(Collection c) {
		Iterator i = c.iterator();
		while (i.hasNext())
			if (!isValid(i.next()))
				return false;
		return true;
	}
	
	/*
	 * Is this a valid item to save?
	 */
	private static boolean isValid(Object obj) {
		if (obj instanceof EObject) {
			if (obj instanceof TRCMonitor) {
				TRCMonitor mon = (TRCMonitor)obj;
				if (isReadOnly(mon.eResource())) {
					handleSaveReadOnlyException(mon.getName(), mon);
					return false;
				}
			}
			else if (obj instanceof TRCNode) {
				TRCNode node = (TRCNode)obj;
				if (!node.eIsProxy()) {
					if (isReadOnly(node.eResource())) {
						handleSaveReadOnlyException(node.getName(), node);
						return false;
					}
				}
			}
			else if (obj instanceof TRCProcessProxy) {
				TRCProcessProxy p = (TRCProcessProxy)obj;
				if (!p.eIsProxy()) {
					if (isReadOnly(p.eResource())) {
						handleSaveReadOnlyException(p.getName(), p);
						return false;
					}
				}
			}
			else if (obj instanceof TRCAgentProxy) {
				TRCAgentProxy a = (TRCAgentProxy)obj;
				if (!a.eIsProxy()) {
					if (isReadOnly(a.getAgent().eResource())) {
						handleSaveReadOnlyException(a.getName(), a);
						return false;
					}
					if (a.isCollectionData()) {
						handleAgentCollectingWhenSave(a);
						return false;
					}
				}
			}
			else if (obj instanceof CorrelationContainerProxy) {
				CorrelationContainerProxy proxy = (CorrelationContainerProxy)obj;
				if (isReadOnly(proxy.getCorrelationContainer().eResource())) {
					handleSaveReadOnlyException(proxy.getName(), proxy);
					return false;
				}
			}
		}
		return true;
	}
	
	/*
	 * Is this resource read-only?
	 */
	private static boolean isReadOnly(Resource rsc) {
		IPath path = new org.eclipse.core.runtime.Path(rsc.getURI().toString());
		IResource foundrsc =
			ResourcesPlugin.getWorkspace().getRoot().findMember(path);
		if ((foundrsc != null) && (foundrsc.isAccessible()))
			return foundrsc.isReadOnly();

		return false;
	}

	private static void handleAgentCollectingWhenSave(TRCAgentProxy a) {
		if (a.getAgent() == null)
			return;
		Resource res = a.getAgent().eResource();
		final String text = HyadesUtil.change(UIPlugin.getResourceString("SAVE_AGENT_COLLECTING_ERROR_"), "%1", a.getName());

		Display.getDefault().asyncExec(new Runnable() {
			public void run() {
				final Shell shell = UIPlugin.getActiveWorkbenchShell();
				if (shell != null) {
					final Status err =
						new Status(
							Status.WARNING,
							ResourcesPlugin.PI_RESOURCES,
							IResourceStatus.INTERNAL_ERROR,
							text,
							null);
					ErrorDialog.openError(
						shell,
						UIPlugin.getResourceString("TRACE_MSG"),
						UIPlugin.getResourceString("SAVE_FILE_ERROR_TITLE"),
						err);
				}
			}
		});
	}

	private static void handleSaveException(String objectType, String objectName, EObject obj) {
		Resource res = obj.eResource();
		String text = UIPlugin.getResourceString("SAVE_FILE_ERROR_");
		text = TString.change(text, "%1", objectType);

		if (res != null)
			text = TString.change(text, "%2", res.getURI().toString());
		else
			text = TString.change(text, "%2", objectName);

		final String finalText = text;
		Display.getDefault().asyncExec(new Runnable() {
			public void run() {
				Shell shell = UIPlugin.getActiveWorkbenchShell();
				if (shell != null) {
					Status err =
						new Status(
							Status.WARNING,
							ResourcesPlugin.PI_RESOURCES,
							IResourceStatus.INTERNAL_ERROR,
							finalText,
							null);
					ErrorDialog.openError(
						shell,
						UIPlugin.getResourceString("TRACE_MSG"),
						UIPlugin.getResourceString("SAVE_FILE_ERROR_TITLE"),
						err);
				}
			}
		});
	}

	private static void handleSaveReadOnlyException(String objectName, EObject obj) {
		Resource res = obj.eResource();
		String text = UIPlugin.getResourceString("SAVE_READ_ONLY_FILE_ERROR_");
		if (res != null)
			text = TString.change(text, "%2", res.getURI().toString());
		if (objectName != null)
			text = TString.change(text, "%1", objectName);

		final String finalText = text;
		Display.getDefault().asyncExec(new Runnable() {
			public void run() {
				Shell shell = UIPlugin.getActiveWorkbenchShell();
				if (shell != null) {
					final Status err =
						new Status(
							Status.WARNING,
							ResourcesPlugin.PI_RESOURCES,
							IResourceStatus.INTERNAL_ERROR,
							finalText,
							null);
					ErrorDialog.openError(
					    shell,
						UIPlugin.getResourceString("TRACE_MSG"),
						UIPlugin.getResourceString("SAVE_FILE_ERROR_TITLE"),
						err);
				}
			}
		});
	}
}
