/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.resources.database.internal;

import java.util.Collection;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
/**
 * This interface can be used to get referenced objects from the database. You
 * must specify an EObject and an EReference. You may specify the database id of
 * the EObject for faster querying, but you are not required to do so. If you do
 * not specify an id, the EMF database searches for an EObject in the database
 * whose attribute values match the attribute values of the given object.
 * <p>
 * If you do not specify a lower bound and upper bound, all referenced objects
 * are returned. You can set the lower and upper bound so the EMF database
 * returns the referenced objects at those positions (they are inclusive, and
 * the first position is 0).
 * <p>
 * If you specify a value for the lower bound and not for the upper bound, all
 * referenced objects are returned whose positions are greater than or equal to
 * the lower bound.
 * <p>
 * If the lower bound is set to a value greater than the number of referenced
 * objects, null is returned.
 * <p>
 * For each referenced object that is returned, all of its attribute values are
 * set. If contained is false, the contained objects of the referenced objects
 * are not restored. If contained is true, the contained objects are restored.
 * By default, contained is true.
 */
public interface ReferenceQuery {
	/**
	 * You must specify the EObject that has the EReference.
	 * 
	 * @return The EObject that has the EReference.
	 */
	EObject getEObject();

	/**
	 * You must set the EObject using this method; otherwise, no objects are
	 * retrieved from the database.
	 * 
	 * @param object
	 *            The EObject that has the reference.
	 */
	void setEObject(EObject object);

	/**
	 * The EReference.
	 * 
	 * @param The
	 *            EReference
	 */
	EReference getEReference();

	/**
	 * You must set the EReference, otherwise, no objects are returned.
	 * 
	 * @param reference
	 *            The EReference
	 */
	void setEReference(EReference reference);

	/**
	 * Referenced objects with this position or higher are returned.
	 * 
	 * @return The lower bound for referenced object positions.
	 */
	int getLower();

	/**
	 * If you do not set lower, referenced objects are returned starting with
	 * position 0.
	 * 
	 * @param lower
	 *            The lower bound for referenced object positions.
	 */
	void setLower(int lower);

	/**
	 * Referenced objects with this position or lower are returned.
	 * 
	 * @return The upper bound for referenced object positions.
	 */
	int getUpper();

	/**
	 * If you do no set upper, all referenced objects in positions greater than
	 * or equal to lower are returned.
	 * 
	 * @param upper
	 *            The upper bound for referenced object positions.
	 */
	void setUpper(int upper);

	/**
	 * This method returns whether references are set for the objects loaded
	 * from the database. This includes loading contained objects. By default,
	 * isSetReferences returns true.
	 */
	boolean isSetReferences();

	/**
	 * Objects loaded from the database will not have their references set (and
	 * contained objects will not be loaded) if setReferences is false.
	 * 
	 * @param setReferences
	 *            Whether to set references for the objects loaded from the
	 *            database.
	 */
	void setReferences(boolean setReferences);

	/**
	 * You can choose to restrict the objects that are loaded from the database.
	 * You might want to do this, for example, if you are using paging lists to
	 * load certain objects on demand. The given collection should be a list of
	 * classes whose instances will not be created to provide values for
	 * multi-valued references.
	 */
	void setNotLoadedClasses(Collection classes);

	/**
	 * Get the classes whose instances are not to be created to provide values
	 * for multi-valued references, because paging lists are used to get those
	 * values.
	 */
	Collection getNotLoadedClasses();
} // ReferenceQuery
