package org.eclipse.hyades.ui.sample.svg.generator;
/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

import java.util.Comparator;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.TreeSet;
import java.util.Vector;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/**
 * Common data retrieval methods which retrieve data of interest to <code>SVGLineChart, 
 * SVGBarChart, SVGStackBarChart, SVGPieChart</code> and <code>SVGMeter</code> from 
 * documents which conform to the documented data input schemas. 
 * 
 * @version 1.25.2.9
 */
public class CommonDataRetriever extends DOMDataRetriever implements IDataInputProcessingExceptionCodes, IDataInputConstants {
	
	/**
	 * Sole constructor.
	 */		
	CommonDataRetriever() {
		super();
	}
		
	/**
	 * Returns the number of data sets found in the data document.
	 * 
	 * @throws DataInputProcessingException for codes <code>NO_DATA_DOCUMENT, 
	 * NO_DATASETS_FOUND</code>.
	 */
	public int getNumberOfDatasets(Document dataDocument) throws DataInputProcessingException {
		if (dataDocument == null) {
			throwException(NO_DATA_DOCUMENT, EMPTY_STRING);
		}	
		
		int number = getNumberOfElements(dataDocument, DATASET);
		return number;
	}
	
	/**
	 * Returns the configuration attributes and values (if any) found in the 
	 * configuration document.
	 * 
	 * @throws DataInputProcessingException for code <code>NO_DATA_DOCUMENT</code>.
	 */
	public Hashtable getConfigurationAttributes(Document configDocument, Document dataDocument) throws DataInputProcessingException {
		
		Document document = (configDocument != null) ? configDocument : dataDocument;	
		if (document == null) {
			throwException(NO_DATA_DOCUMENT, EMPTY_STRING);
		}	
			
		Hashtable properties = getAttributesFromSingleElement(document, CONFIGURATION);
		return properties;
	}
	
	/**
	 * Returns the ordered array of values found in the data document which comprise 
	 * the legend. Values can be ordered by specifying their ordinal position; otherwise,
	 * the retrieved (document) order is used.
	 * 
	 * @throws DataInputProcessingException for codes <code>NO_DATA_DOCUMENT, NO_NAMES_FOUND, MISSING_NAMES_OR_POSITIONS,
	 * POSITION_NUMBER_OUT_OF_BOUNDS, DUPLICATE_POSITION_NUMBER, INVALID_NUMBER_FORMAT</code>
	 */
	public String [] getLegendLabels(Document dataDocument) throws DataInputProcessingException {		
		if (dataDocument == null) {
			throwException(NO_DATA_DOCUMENT, EMPTY_STRING);
		}	
				
		Vector labels = getAttributeFromElements(dataDocument, DATASET, LABEL);				
		if (labels == null) {
			throwException(NO_NAMES_FOUND, DATASET);			
		}	
		
		int labelsSize = labels.size();
		String [] result = new String[labelsSize];
		Vector positions = getAttributeFromElements(dataDocument, DATASET, POSITION);
		
		// use document order
		if (positions == null) {
			for (int i = 0;i < labelsSize; i++) {
				result[i] = (String)labels.get(i);
			}
			return result;
		}
					
		if (labelsSize != positions.size()) {
			throwException(MISSING_LABELS_OR_POSITIONS, DATASET);			
		}			
		
		// put the label in the array based on its position
		for (int i = 0;i < labelsSize; i++) {
			short position = 0;			
			try {
				position = Short.parseShort((String)positions.get(i));					
			} catch (Exception exc) {
				// NumberFormatException
				throwException(INVALID_NUMBER_FORMAT, DATASET);				
			}
			// make sure the position is within bounds						
			if (position > (labelsSize - 1) || position < 0) {
				throwException(POSITION_NUMBER_OUT_OF_BOUNDS, DATASET);
			}				
			// make sure the position is not already taken
			String test = null;		
			try {
				test = result[position];
			}
			catch (Exception e) {}								
						
			if (test != null) {
				throwException(DUPLICATE_POSITION_NUMBER, DATASET);
			}	
			result[position] = (String)labels.get(i);
		}				
		return result;			
	}	
		
	/**
	 * Returns the timestamp found in the data document.
	 * 
	 * @throws DataInputProcessingException for code <code>NO_DATA_DOCUMENT, NO_TIMESTAMP_FOUND</code>.
	 */	
	public String getTimestamp(Document dataDocument) throws DataInputProcessingException { 
		if (dataDocument == null) {
			throwException(NO_DATA_DOCUMENT, EMPTY_STRING);
		}	
		
		String timestamp = getAttributeFromSingleElement(dataDocument, DATAUPDATE, TIMESTAMP);
//		if (timestamp == null) {
//			throwException(NO_TIMESTAMP_FOUND, DATAUPDATE);			
//		}			
		return timestamp;
	}

	/* *
	 * Returns Axis at the specified location (N, E, S, W).
	 * If the specified location is null, the information on the first range is returned.
	 * The returned data is an array of SegmentMarkers in the order of increasing "value".
	 */

	public DataRange getDataRange(Document dataDocument, String location) throws DataInputProcessingException {
        NodeList ranges = dataDocument.getElementsByTagName(DATARANGE);
        Element range = null;
     	int length = ranges.getLength();

        if ((location == null) || (location.equals(""))) {
        	if (length <= 0) {
        		return null;
        	}
        	range = (Element)ranges.item(0);
        }
        else {
        	for(int i=0; i < length; i++ ) {
        		Element r = (Element)ranges.item(i);
        		if (r.getAttribute(LOCATION).equals(location)) {
        			range = r;
        			break;
        		}
        	}
        }

		if (range == null) {
			return null;
		}

		DataRange axis = new DataRange(location);
		if (range.hasAttribute(ID)) {
			axis.setId(range.getAttribute(ID));
		}
		if (range.hasAttribute(LABEL)) {
			axis.setLabel(range.getAttribute(LABEL));
		}
		if (range.hasAttribute(TYPE)) {
			String t = range.getAttribute(TYPE);
			if (t != null) {
				if (t.equals(CONTINUUM)) {
					axis.setType(DataRange.CONTINUUM);
				}
				else if (t.equals(CATEGORIZATION)) {
					axis.setType(DataRange.CATEGORIZATION);
				}
			}
		}
		
        NodeList markers = range.getElementsByTagName(SEGMENT_MARKER);
        int size = markers.getLength();

	    TreeSet sortedMarkers = new TreeSet(new Comparator() {
	                                              public int compare(Object o1, Object o2) {
                                                     SegmentMarker sm1 = (SegmentMarker)o1;
                                                     SegmentMarker sm2 = (SegmentMarker)o2;
                                                     double d1 = sm1.getValue();
                                                     double d2 = sm2.getValue();
                                                     return (d1 == d2) ? 0 : (d1 < d2) ? -1 : 1;
	                                              }
	                                          }
	                                     );

        for(int i=0; i < size; i++) {
	        Element e = (Element)markers.item(i);
	        SegmentMarker sm = new SegmentMarker(e.getAttribute(VALUE));
			if (e.hasAttribute(LABEL)) {
	        	sm.setLabel(e.getAttribute(LABEL));
			}
			if (e.hasAttribute(TYPE)) {
	            String type = e.getAttribute(TYPE);
	        	if (type.equals(TICK)) {
    	            sm.setType(SegmentMarker.TICK);
        	    }
            	else if (type.equals(LINE)) {
                	sm.setType(SegmentMarker.LINE);
	            }
	        }
	        sortedMarkers.add(sm);
	    }

	    // fill in the array
		SegmentMarker sm[] = new SegmentMarker[size];
	    int i=0;
	    for(Iterator j=sortedMarkers.iterator(); j.hasNext(); i++) {
	    	sm[i] = (SegmentMarker) j.next();
	    }

		axis.setSegmentMarkers(sm);
		
	    return axis;
	}


	/**
	 *  Parse the given list of Datapoint elements and return an array of Datapoint objects in ascending xValues
	 */		
	private DataPoint[] getDatapointsFromList(NodeList dataPoints) throws DataInputProcessingException {
		if (dataPoints == null) {
			return null;
		}
		int n = dataPoints.getLength();

	    TreeSet sortedDatapoints =  new TreeSet(new Comparator() {
	                                              public int compare(Object o1, Object o2) {
                                                     DataPoint dp1 = (DataPoint)o1;
                                                     DataPoint dp2 = (DataPoint)o2;
                                                     double d1 = dp1.getValue1();
                                                     double d2 = dp2.getValue1();
                                                     return (d1 == d2) ? 0 : (d1 < d2) ? -1 : 1;
	                                              }
	                                          }
	                                     );

		for(int i=0; i < n; i++) {
			Element dpe = (Element)dataPoints.item(i);
			String value2String = dpe.getAttribute(VALUE2);
			DataPoint dp;
			if (dpe.hasAttribute(VALUE1)) {
			    dp = new DataPoint(dpe.getAttribute(VALUE1), value2String);
			}
			else {
			    dp = new DataPoint(value2String);
			}

			if (dpe.hasAttribute(TYPE)) {
			    String type = dpe.getAttribute(TYPE);
			    if (type.equals(ACTUAL)) {
		    		dp.setType(DataPoint.ACTUAL);
		    	}
	    		else if (type.equals(HOLE)) {
	    			dp.setType(DataPoint.HOLE);
		    	}
			}
		    if (dpe.hasAttribute(LABEL)) {
		    	dp.setLabel(dpe.getAttribute(LABEL));
		    }
		    
		    // any image element?
		    NodeList l = dpe.getElementsByTagName(IMAGE);
		    if (l.getLength() > 0) {
		    	Element image = (Element)l.item(0);
		    	if (image.hasAttribute(TYPE)) {
		    		dp.setImageType(image.getAttribute(TYPE));
		    	}
		    	if (image.hasAttribute(URI)) {
		    		dp.setImageURI(image.getAttribute(URI));
		    	}
		    	if (image.hasAttribute(CODE)) {
		    		dp.setImageCode(image.getAttribute(CODE));
		    	}
		    	if (image.hasAttribute(WIDTH)) {
			    	String s = image.getAttribute(WIDTH);	
		    		int w = Integer.parseInt(s);
		    		if (w < 0) {
					    throwException(NEGATIVE_IMAGE_WIDTH, s);
		    		}
		    		else {
		    		    dp.setImageWidth(w);
		    		}
		    	}
				if (image.hasAttribute(HEIGHT)) {
			    	String s = image.getAttribute(HEIGHT);
		    		int h = Integer.parseInt(s);
		    		if (h < 0) {
					    throwException(NEGATIVE_IMAGE_HEIGHT, s);
		    		}
		    		else {
		    		    dp.setImageHeight(h);
		    		}
				}
		    } // if
   		        sortedDatapoints.add(dp);
		} // for
		
		// fill in the result
		DataPoint [] result = new DataPoint[n];

	    int i=0;
	    for(Iterator j=sortedDatapoints.iterator(); j.hasNext(); i++) {
	    	result[i] = (DataPoint) j.next();
	    }

		return result;
	}


	/**
	 *  Return the data set by parsing the given DOM element
	 */
	private DataSet getDataSet(Element ds, int position) throws DataInputProcessingException  {
		DataSet dataset = new DataSet(position);
		if (ds.hasAttribute(LABEL)) {
			dataset.setLabel(ds.getAttribute(LABEL));
		}
		if (ds.hasAttribute(FLYOVERLABEL)) {
			dataset.setFlyoverLabel(ds.getAttribute(FLYOVERLABEL));
		}
		if (ds.hasAttribute(DATARANGE1REF)) {
			dataset.setDataRange1Ref(ds.getAttribute(DATARANGE1REF));
		}
		if (ds.hasAttribute(DATARANGE2REF)) {
			dataset.setDataRange2Ref(ds.getAttribute(DATARANGE2REF));
		}
		if (ds.hasAttribute(TOTAL)) {
			double total = Double.parseDouble(ds.getAttribute(TOTAL));
			if(total >= 0) 
			dataset.setTotal(total);
			else{
				throw new DataInputProcessingException(INVALID_NUMBER_FORMAT, TOTAL);
			}
		}
		dataset.setDataPoints(getDatapointsFromList(ds.getElementsByTagName(DATAPOINT)));
		
		return dataset;
	}
	
	
	/**
	 *  Returns an array of datapoints of the dataset of the specified position
	 * 
	 * 	@throws DataInputProcessingException for codes <code>NO_DATA_DOCUMENT, NO_VALUES_FOUND, NO_IDS_FOUND, 
	 *  MISSING_CATEGORY_IDS_OR_DATAPOINT_VALUES, CATEGORY_ID_MISMATCH, INVALID_NUMBER_FORMAT</code>
	 */
	public DataSet getDatasetWithPosition(Document dataDocument, String position) throws DataInputProcessingException  {
		if (dataDocument == null) {
			throwException(NO_DATA_DOCUMENT, EMPTY_STRING);
		}

		// first see if the position attribue is specified
		Element ds= getElementWithAttribute(dataDocument, DATASET, POSITION, position);
		if (ds == null) {
			ds = getElementWithDocOrderOrdinal(dataDocument, DATASET, position);					
		}
		// just get the one with the ordinal position
		if (ds == null) {			
			throwException(NO_VALUES_FOUND, DATASET);			
		}

		return getDataSet(ds, Integer.parseInt(position));
	}
		

	/**
	 * Returns the array of data values from the first data set found in the data document.
	 * 
	 * @throws DataInputProcessingException for codes <code>NO_DATA_DOCUMENT, NO_VALUES_FOUND, NO_IDS_FOUND, 
	 * MISSING_CATEGORY_IDS_OR_DATAPOINT_VALUES, CATEGORY_ID_MISMATCH, INVALID_NUMBER_FORMAT</code>
	 */
	public DataSet getFirstDataset(Document dataDocument) throws DataInputProcessingException {
		if (dataDocument == null) {
			throwException(NO_DATA_DOCUMENT, EMPTY_STRING);
		}	
		
		Element ds = getFirstElement(dataDocument, DATASET);	
		if (ds == null) {
			throwException(NO_VALUES_FOUND, DATAPOINT);	
		}
		return getDataSet(ds, 0);
	}
	
	
	/**
	 * Convenience method for throwing exceptions.
	 */
	protected void throwException(short code, String element) throws DataInputProcessingException {
		DataInputProcessingException e = new DataInputProcessingException(code, element);
		throw e;		
	}	
	
}

