package org.eclipse.hyades.ui.sample.svg.generator;
/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

import java.io.Serializable;
import java.util.Date;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * <code>SVGScatterChart</code> generates a Scatter chart graphic DOM using Scalable Vector Graphics (SVG).
 * 
 * @version 1.18
 */

public class SVGScatterChart extends SVGLineChart implements IGraphicDocumentStyle, IDataInputProcessingExceptionCodes, Serializable {
	/**
	 * Sole constructor
	 */ 
	public SVGScatterChart() {
		super();
		graphicType = IGraphicTypeConstants.SCATTER_CHART;
	}
		/**
     * @see IGraphicDocumentGenerator#generateGraphicDocument(GraphicDocumentProperties)
     */       
	public Document generateGraphicDocument(GraphicDocumentProperties input) throws DataInputProcessingException {	
		// get the input
		GraphicAttributes graphicAttrs = createGraphicAttributes(input);	
		Document newDataDocument = graphicAttrs.getDataDocument();		
		int numberOfDatasets = 0;
		String [] legendFlyOverLabels = null;				
		// make sure there is some data, or exit
		try {
			numberOfDatasets = dataRetriever.getNumberOfDatasets(newDataDocument);
		} catch (DataInputProcessingException e) {
			e.setWidgetType(graphicType);
			//System.out.println(e.getWidgetType() + " " + e.getElement() + " " + e.getMessage());
			throw e;		
		}		
		
		String rangeMaxValue = null;
		int gridXOffset = 60;
		int halfGridXOffset = gridXOffset/2;
		int gridYOffset = 50;
		int xAxisLength = 0;
		// grid     
		try {
			xAxis = dataRetriever.getDataRange(newDataDocument, "S");
		} catch (DataInputProcessingException e) {
			e.setWidgetType(getGraphicType());
			//System.out.println(e.getWidgetType() + " " + e.getElement() + " " + e.getMessage());
			throw e;				
		}					

		try {
			yAxis = dataRetriever.getDataRange(newDataDocument, "W");
		} catch (DataInputProcessingException e) {
			e.setWidgetType(getGraphicType());
			//System.out.println(e.getWidgetType() + " " + e.getElement() + " " + e.getMessage());
			throw e;				
		}					

		if ((xAxis == null) || (yAxis == null)) {
			throw new DataInputProcessingException("", NO_RANGE_FOUND, DATARANGE, getGraphicType());
		}
		if ((xAxis.getType() != DataRange.CONTINUUM) || (yAxis.getType() != DataRange.CONTINUUM)) {
			throw new DataInputProcessingException("", UNSUPPORTED_DATA_RANGE_TYPE, DATARANGE, getGraphicType());
		}

		xMarkers = xAxis.getSegmentMarkers();
		yMarkers = yAxis.getSegmentMarkers();

		if (yMarkers != null && yMarkers.length > 0) { 
			rangeMaxValue = (String)yMarkers[yMarkers.length - 1].getValueString();
		}
		
		// The datasets
		dataSets = new DataSet[numberOfDatasets];
		for(int i=0; i < numberOfDatasets; i++) {
            dataSets[i] = dataRetriever.getDatasetWithPosition(newDataDocument, Integer.toString(i));
		}
		//get the Fly over text for the legend item.				
		legendFlyOverLabels = getLegendFlyOverLabels(dataSets);

		//get the max size of Ymarker Width.		
		double maxLengthOfYmarkerWidth = getMaxLengthOfYmarker(yMarkers, graphicAttrs);
	    double maxAllowableYMarkerWidth = graphicAttrs.getGraphicWidth() / 4;
	    if (maxLengthOfYmarkerWidth > maxAllowableYMarkerWidth) {
			maxLengthOfYmarkerWidth = maxAllowableYMarkerWidth;
		}
		
		//get the max size of Legent Lable Width.		
		double maxLengthOfLegendLableWidth = getMaxLengthOfLegendLabel(graphicAttrs, dataSets);
	    double maxAlloableLegendWidth = graphicAttrs.getGraphicWidth() / 3;
   		if (maxLengthOfLegendLableWidth > maxAlloableLegendWidth) {
			maxLengthOfLegendLableWidth = maxAlloableLegendWidth;
		}
		
		// Start calculations
        if (graphicAttrs.isLegendSuppressed()) {
			maxLengthOfLegendLableWidth = 0;
        }
		xAxisLength = (int)(graphicAttrs.getGraphicWidth() - (maxLengthOfLegendLableWidth + maxLengthOfYmarkerWidth + gridXOffset));
   		int xLabelRowCount = getNumberOfAxisLabels(graphicAttrs, xAxisLength, xAxis);									
		int yAxisLength = (int)(graphicAttrs.getGraphicHeight() - (gridYOffset * 2.5) - (xLabelRowCount * 10));			
				
		Document generatedDocument = createSvgDocument(newDataDocument.getImplementation(), Short.toString(graphicAttrs.getGraphicWidth()), Short.toString(graphicAttrs.getGraphicHeight()));
		Element svgRoot = generatedDocument.getDocumentElement();
		registerEventHandler(svgRoot, "onload", "init(evt)");	
		addDefinitions(generatedDocument, graphicAttrs, numberOfDatasets);
		addExtraDefinitions(generatedDocument, graphicAttrs, numberOfDatasets, svgRoot);					
		addJavaScriptFunctions(generatedDocument, graphicAttrs, numberOfDatasets, gridXOffset, gridYOffset, xAxisLength, yAxisLength);
 		
 		// graphic outline 
 		if (!graphicAttrs.isOutlineSuppressed()) {
 			addOutline(generatedDocument, graphicAttrs);
 		}	 		
 		 		
		// timestamp
		if (!graphicAttrs.isTimestampSuppressed()) {
			try {
				String timestamp = dataRetriever.getTimestamp(newDataDocument);
				Date date = GraphicAttributes.parseISO8601Date(timestamp);
	        	addTimeStamp(generatedDocument, graphicAttrs, (date == null) ? timestamp : graphicAttrs.formatDate(date));
			} catch (DataInputProcessingException e) {
				e.setWidgetType(graphicType);
				//System.out.println(e.getWidgetType() + " " + e.getElement() + " " + e.getMessage());
				throw e;
			}
		}
			
		// title bar
 		if (!graphicAttrs.isTitlebarSuppressed()) {	 		
	   		addTitleBar(generatedDocument, graphicAttrs);
 		}
 		// need to adjust graphic vertical position
 		else {
 			gridYOffset = super.resetGridOffsetForCenter(graphicAttrs.getGraphicHeight(), yAxisLength, 3.0);	
 		}	
				
		// preference icon
		if(!graphicAttrs.isUserPreferencesSuppressed() && graphicAttrs.getSvgFormatOnly())
		{
			addPreferenceIcon(generatedDocument, graphicAttrs);
		}
		
		// the actual x offset
		gridXOffset = (halfGridXOffset) + (int)maxLengthOfYmarkerWidth;

		// border, tick marks etc.
        addGrid(generatedDocument, svgRoot, xAxisLength, yAxisLength, gridXOffset, gridYOffset, xAxis, yAxis);

		// x and y axis labels
        addAxisLabels(generatedDocument, graphicAttrs, svgRoot, xAxisLength, yAxisLength, gridXOffset, gridYOffset, xAxis, yAxis);
                
        // axis titles
        addTitles(generatedDocument, graphicAttrs, svgRoot, xAxisLength, yAxisLength, gridXOffset, gridYOffset, xAxis, yAxis, xLabelRowCount, rangeMaxValue);

        // data points
		Element gTextBoxElement = generatedDocument.createElement("g");
        addDataSets(generatedDocument, graphicAttrs, svgRoot, gTextBoxElement, xAxisLength, yAxisLength, gridXOffset, gridYOffset, xAxis, yAxis, dataSets); 

       // legend
        if (!graphicAttrs.isLegendSuppressed()) {      		
	  		try {
	       		String [] legendLabels = dataRetriever.getLegendLabels(newDataDocument);		       
				addLegend(generatedDocument, graphicAttrs, gTextBoxElement, legendLabels, legendFlyOverLabels, (int)maxLengthOfYmarkerWidth + xAxisLength + (halfGridXOffset), gridYOffset); 								

	  		} catch (DataInputProcessingException e) {
	  			e.setWidgetType(graphicType);
	  			//System.out.println(e.getWidgetType() + " " + e.getElement() + " " + e.getMessage());	  			
	  			throw e;
	  		} 		
    	}
		svgRoot.appendChild(gTextBoxElement);
    	return generatedDocument;
	}
	
	/**
     * @see SVGDocumentGenerator#addLegendItemGraphic()
     */ 
	protected Element addLegendItemGraphic(Document generatedDocument, Element parent, int index, int fontSize, int x, int y) {
		Element gColourElement;

		final String[] flyOverShapeType = {"flyoverCircleshape", "flyoverSquareshape", "flyoverHorizontalrectshape", "flyoverVerticalrectshape", "flyoverTriangleshape", "flyoverDiamondshape"};				
		// add the standard shape to the legend
		gColourElement = super.addLegendItemGraphic(generatedDocument, parent, index, fontSize, x, y);
		
		gColourElement.setAttribute("onmouseover", "setVisibility('" + flyOverShapeType[index % (flyOverShapeType.length)] + "' , 'visible');");
		gColourElement.setAttribute("onmouseout", "setVisibility('" + flyOverShapeType[index % (flyOverShapeType.length)] + "' , 'hidden');");			
		return gColourElement;
	}

	// specific to area chart 
	private void addDataSets(Document generatedDocument, GraphicAttributes attrs, Element parent, Element gTextBoxElement, int xAxisLength, int yAxisLength, int gridXOffset, int gridYOffset, DataRange xAxis, DataRange yAxis, DataSet[] dataSets)
		throws DataInputProcessingException {

		// add points and area
		double xMin = xAxis.getMinValue();
		double xMax = xAxis.getMaxValue();
		double yMin = yAxis.getMinValue();
		double yMax = yAxis.getMaxValue();

		if ((xMin == xMax) || (yMin == yMax)) return;

		for (int i = 0; i < dataSets.length; i++) {
			DataSet dataset = dataSets[i];
			DataPoint[] dps = dataset.getDataPoints();
			double[] yPositions = new double[dps.length];
			double[] xPositions = new double[dps.length];
			
			for(int j=0; j < dps.length; j++) {
				xPositions[j] = xAxisLength * (dps[j].getValue1() - xMin) / (xMax - xMin);
				yPositions[j] = yAxisLength * (yMax - dps[j].getValue2()) / (yMax - yMin);
			}
			
		 	// add the points and area	
	  	 	addPoints(generatedDocument, parent, dps, i, xPositions, yPositions, xAxisLength, yAxisLength, gridXOffset, gridYOffset);

	  	 	// add the texbox
			String stylesheetClass = "dataValues anchorAtMiddle";		
			Element gToggleElement = generatedDocument.createElement("g");
			gToggleElement.setAttribute("id","textbox" + i);
			gToggleElement.setAttribute("class", stylesheetClass);
			gToggleElement.setAttribute("transform", "translate(" + gridXOffset + "," + gridYOffset + ")");
			//gToggleElement.setAttribute("font-family", font);
			gToggleElement.setAttribute("visibility", "hidden");
			gToggleElement.setAttribute("onclick", "toggleVisibility('textbox" + i + "');");
			gTextBoxElement.appendChild(gToggleElement);	
			addTextBoxes(generatedDocument, attrs, gToggleElement, i, xPositions, yPositions, dps, xAxisLength, yAxisLength);
		}			
	}


	// Add a set of polyline and polygon elements for each dataset
	private void addPoints(Document generatedDocument, Element parent, DataPoint[] dps, int linesetNumber, double[] xPositions, double[] yPositions, int xAxisLength, int yAxisLength, int gridXOffset, int gridYOffset) {

		final String[] shapeType = {"#circleshape", "#squareshape", "#horizontalrectshape", "#verticalrectshape", "#triangleshape", "#diamondshape"};
		final String[] flyoverShapeType = {"#flyoverCircleshape", "#flyoverSquareshape", "#flyoverHorizontalrectshape", "#flyoverVerticalrectshape", "#flyoverTriangleshape", "#flyoverDiamondshape"};
		final double[] shapeWidth = {5.6, 5.6, 5.6, 2.8, 5.6, 5.6};
		final double[] shapeHeight = {5.6, 5.6, 2.8, 5.6, 4.85, 5.6};
		final double BOX_HEIGHT = 18.0;

		int length = xPositions.length;
		double xPrevious = 0;
		double yPrevious = 0;
		double xNext;
		double yNext;
		
		// group for data colour changes and toggling exact values visibility
		Element gColourElement = generatedDocument.createElement("g");
		String pointAttribute = xPositions[0] + " " + yAxisLength;
		for (int i = 0; i < length; i++) 
		{
			xNext = xPositions[i];
			yNext = yPositions[i];
			pointAttribute += ", " + xNext + " " + yNext;
		}
		pointAttribute += ", " + xPositions[length-1] + " " + yAxisLength;
		// Draw area for the scatter chart
		Element polygonElement = generatedDocument.createElement("polygon");
		polygonElement.setAttribute("id", "poly" + linesetNumber);
		polygonElement.setAttribute("points", pointAttribute);
		polygonElement.setAttribute("style", "fill-opacity: 0.0; stroke: black; stroke-width: 0;");
		polygonElement.setAttribute("transform", "translate(" + gridXOffset + "," + gridYOffset + ")");
		gColourElement.appendChild(polygonElement);

		gColourElement.setAttribute("id", "dataColour" + linesetNumber);
		gColourElement.setAttribute("class", "dataset" + linesetNumber);
		gColourElement.setAttribute("visibility", "visible");
		gColourElement.setAttribute("onclick", "toggleVisibility('textbox" + linesetNumber + "');");
		
		parent.appendChild(gColourElement);
		
		// draw a polyline with data points
		int shapeNumber = linesetNumber % shapeType.length;
		for (int i = 0; i < length; i++) {
			xNext = xPositions[i];
			yNext = yPositions[i];
			
			// Add shapes				
			double shapeX = xNext - shapeWidth[shapeNumber] / 2.0;
			if (shapeX <= 0) { // adjust shape x position if it crosses the left boundary
				shapeX = 0;	
			} else if (shapeX + shapeWidth[shapeNumber] > xAxisLength) { // adjust shape x position if it crosses the right boundary
				shapeX = xAxisLength - shapeWidth[shapeNumber];
			}
			
			double shapeY = yNext - shapeHeight[shapeNumber] / 2.0;
			if (shapeY + shapeHeight[shapeNumber] > yAxisLength) {  // adjust the shape y position if it crosses the x-axis
				shapeY = yAxisLength - shapeHeight[shapeNumber];	
			}
			
			Element shapeElement = generatedDocument.createElement("use");
			Element flyOverShapeElement = generatedDocument.createElement("use");
			if (dps[i].getType() == DataPoint.HOLE)
			{
				shapeElement.setAttribute("xlink:href", "#holeshape");
			}else
			{
				shapeElement.setAttribute("xlink:href", shapeType[shapeNumber]);
			}			
				flyOverShapeElement.setAttribute("xlink:href", flyoverShapeType[shapeNumber]);
				shapeElement.setAttribute("transform", "translate(" + (gridXOffset + shapeX) + "," + (gridYOffset + shapeY) + ")");
	
			if(shapeNumber == 2 ){
				flyOverShapeElement.setAttribute("transform", "translate(" + (gridXOffset + shapeX-4.5) + "," + (gridYOffset + shapeY-3.5) + ")");
			}else if(shapeNumber == 3)
			{
				flyOverShapeElement.setAttribute("transform", "translate(" + (gridXOffset + shapeX-4.5) + "," + (gridYOffset + shapeY-7) + ")");
			}else if(shapeNumber == 4)
			{
				flyOverShapeElement.setAttribute("transform", "translate(" + (gridXOffset + shapeX-7) + "," + (gridYOffset + shapeY-9) + ")");
			}else
			{
				flyOverShapeElement.setAttribute("transform", "translate(" + (gridXOffset + shapeX-7) + "," + (gridYOffset + shapeY-7) + ")");				
			}
	
			shapeElement.setAttribute("visibility", "visible");
			flyOverShapeElement.setAttribute("visibility", "hidden");
			gColourElement.appendChild(shapeElement);
			gColourElement.appendChild(flyOverShapeElement);
						
			// add shapes on the data point
			xPrevious = xNext;
			yPrevious = yNext;	
		}		
	}	
	
	/**
	 * 	// add defs element with scatter chart specific definitions 
	 */
	private void addExtraDefinitions(Document generatedDocument, GraphicAttributes attrs, int numberOfDatasets, Element parent) {

		Element defsElement = generatedDocument.createElement("defs");

		// id: squareshape
		parent.appendChild(defsElement);
		Element gElement_flyoverSquare = generatedDocument.createElement("g");	
		defsElement.appendChild(gElement_flyoverSquare);
		gElement_flyoverSquare.setAttribute("id","flyoverSquareshape");
		gElement_flyoverSquare.setAttribute("visibility", "hidden");
			
	    Element flyoverSquareElement = generatedDocument.createElement("rect");
	    gElement_flyoverSquare.appendChild(flyoverSquareElement);
		flyoverSquareElement.setAttribute("width", "20.0");
		flyoverSquareElement.setAttribute("height", "20.0");
		flyoverSquareElement.setAttribute("style", "fill-opacity:0.3;stroke-width:0");
		
		// id: circleshape
		Element gElement_flyoverCircle = generatedDocument.createElement("g");
		defsElement.appendChild(gElement_flyoverCircle);
		gElement_flyoverCircle.setAttribute("id", "flyoverCircleshape");
		gElement_flyoverCircle.setAttribute("visibility", "hidden");
					
		Element flyoverCircleElement = generatedDocument.createElement("circle");
		gElement_flyoverCircle.appendChild(flyoverCircleElement);
		flyoverCircleElement.setAttribute("r", "10.0");
		flyoverCircleElement.setAttribute("cx", "10.0");
		flyoverCircleElement.setAttribute("cy", "10.0");
		flyoverCircleElement.setAttribute("style", "fill-opacity:0.3;stroke-width:0");
		
		// id: holeshape
		Element gElement_flyoverHole = generatedDocument.createElement("g");
		defsElement.appendChild(gElement_flyoverHole);
		gElement_flyoverHole.setAttribute("id", "flyoverHoleshape");
		gElement_flyoverHole.setAttribute("visibility", "hidden");
					
		Element flyoverHoleElement = generatedDocument.createElement("circle");
		gElement_flyoverHole.appendChild(flyoverHoleElement);
		flyoverHoleElement.setAttribute("r", "2.8");
		flyoverHoleElement.setAttribute("cx", "2.8");
		flyoverHoleElement.setAttribute("cy", "2.8");
		flyoverHoleElement.setAttribute("style", "fill:none;");
		flyoverHoleElement.setAttribute("style", "fill-opacity:0.3;stroke-width:0");					
		
		// id: diamondshape
		Element gElement_flyoverDiamond = generatedDocument.createElement("g");
		defsElement.appendChild(gElement_flyoverDiamond);
		gElement_flyoverDiamond.setAttribute("id", "flyoverDiamondshape");
		gElement_flyoverDiamond.setAttribute("visibility", "hidden");
				
		Element flyoverDiamondElement = generatedDocument.createElement("polygon");
		gElement_flyoverDiamond.appendChild(flyoverDiamondElement);
		flyoverDiamondElement.setAttribute("points", "0,10 10,0 20,10 10,20");
		flyoverDiamondElement.setAttribute("style", "fill-opacity:0.3;stroke-width:0");						

		// id: horizontalrectshape
		Element gElement_flyoverHorizontalrectshape = generatedDocument.createElement("g");
		defsElement.appendChild(gElement_flyoverHorizontalrectshape);
		gElement_flyoverHorizontalrectshape.setAttribute("id", "flyoverHorizontalrectshape");
		gElement_flyoverHorizontalrectshape.setAttribute("visibility", "hidden");
					
		Element flyoverHorizontalrectElement = generatedDocument.createElement("rect");
		gElement_flyoverHorizontalrectshape.appendChild(flyoverHorizontalrectElement);
		flyoverHorizontalrectElement.setAttribute("width", "15");
		flyoverHorizontalrectElement.setAttribute("height", "10");
		flyoverHorizontalrectElement.setAttribute("style", "fill-opacity:0.3;stroke-width:0");
		
		// id: verticalrectshape
		Element gElement_flyoverVerticalrectshape = generatedDocument.createElement("g");
		defsElement.appendChild(gElement_flyoverVerticalrectshape);
		gElement_flyoverVerticalrectshape.setAttribute("id", "flyoverVerticalrectshape");
		gElement_flyoverVerticalrectshape.setAttribute("visibility", "hidden");	
				
		Element flyoverVerticalrectElement = generatedDocument.createElement("rect");
		flyoverVerticalrectElement.setAttribute("width", "12");
		flyoverVerticalrectElement.setAttribute("height", "20");		
		gElement_flyoverVerticalrectshape.appendChild(flyoverVerticalrectElement);
		gElement_flyoverVerticalrectshape.setAttribute("style", "fill-opacity:0.3;stroke-width:0");
		
		// id: triangleshape 
		Element gElement_flyoverTriangleshape = generatedDocument.createElement("g");
		defsElement.appendChild(gElement_flyoverTriangleshape);
		gElement_flyoverTriangleshape.setAttribute("id", "flyoverTriangleshape");
		gElement_flyoverTriangleshape.setAttribute("visibility", "hidden");
					
		Element flyoverTriangleElement = generatedDocument.createElement("polygon");
		flyoverTriangleElement.setAttribute("points", "10 0, 0 20, 20 20");
		gElement_flyoverTriangleshape.appendChild(flyoverTriangleElement);
		gElement_flyoverTriangleshape.setAttribute("style", "fill-opacity:0.3;stroke-width:0");
				
	}

	/**
	 * Initialize the Scatter chart attributes
	 */
	protected GraphicAttributes createGraphicAttributes(GraphicDocumentProperties input) throws DataInputProcessingException {
		GraphicAttributes graphicAttrs = new GraphicAttributes();
			
		// set up the defaults
		graphicAttrs.setGraphicWidth(SCATTER_DEFAULT_WIDTH);
		graphicAttrs.setPreferencesPage(IUserPreferencesConstants.SCATTER_PREFS);
		graphicAttrs.setPreferencesPageWidth(IUserPreferencesConstants.PREFS_WIDTH);
		graphicAttrs.setPreferencesPageHeight(IUserPreferencesConstants.SCATTER_PREFS_HEIGHT);
		
		// get the input documents
		graphicAttrs.setConfigDocument(input.getConfigDocument());
		graphicAttrs.setDataDocument(input.getDataDocument());
		
		// (re)set any properties from the input documents		
		getConfigAttributes(graphicAttrs);
		
		// (re)set any properties from the input bean
		graphicAttrs.setAttributesFrom(input);		
				
		return graphicAttrs;
	}	
	
}
