/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.analysis.engine;

import java.util.Arrays;
 
/**
 * Instructive solution element in a symptom database resulting from a successful 
 * <code>Incident</code> match.
 * <p>
 * Each <code>Solution</code> contains a unique identifier, English 
 * description and an associated array of <code>Directive</code>s.
 * <p>
 * <code>Solution</code> maps to the symptom database <code>solution</code>
 * tag:
 * <pre>
 * < solution id="" description="" directives="" />
 * </pre>
 */
 
public class Solution {

	/**
	 * Unique identifier.
	 */
	private String id = null;

	/**
	 * English description.
	 */
	private String description = null;

	/**
	 * Associated array of Directives.
	 */
	private Directive[] directives = null;

	/**
	 * Convenience constructor for creating a <code>Solution</code>
	 * with the parameter unique identifier but no English description 
	 * and no associated array of Directives.
	 * 
	 * @param id the unique identifier
	 * @see   #Solution(String,String,Directive[])
	 */
	public Solution(String id) {
		this(id, null, null);
	}

	/**
	 * Convenience constructor for creating a <code>Solution</code>
	 * with the parameter unique identifier and English description 
	 * but no associated array of Directives.
	 * 
	 * @param id the unique identifier
	 * @param description the English description
	 * @see   #Solution(String,String,Directive[])
	 */
	public Solution(String id, String description) {
		this(id, description, null);
	}

	/**
	 * Constructor for creating a <code>Solution</code>
	 * with the parameter unique identifier, English description 
	 * and associated array of Directives.
	 * 
	 * @param id the unique identifier
	 * @param description the English description
	 * @param directives the associated array of Directives
	 */
	public Solution(String id, String description, Directive[] directives) {
		this.id = id;
		this.description = description;
		this.directives = directives;
	}

	/**
	 * Sets the unique identifier.
	 * 
	 * @param id the unique identifier
	 * @see   #getId()
	 */
	public void setId(String id) {
		this.id = id;
	}

	/**
	 * Gets the unique identifier.
	 * 
	 * @return the unique identifier
	 * @see    #setId(String)
	 */
	public String getId() {
		return id;
	}

	/**
	 * Sets the English description.
	 * 
	 * @param description the English description
	 * @see   #getDescription()
	 */
	public void setDescription(String description) {
		this.description = description;
	}

	/**
	 * Gets the English description.
	 * 
	 * @return the English description
	 * @see    #setDescription(String)
	 */
	public String getDescription() {
		return description;
	}

	/**
	 * Sets the associated array of Directives.
	 * 
	 * @param directives the associated array of Directives
	 * @see   #getDirectives()
	 */
	public void setDirectives(Directive[] directives) {
		this.directives = directives;
	}

	/**
	 * Gets the associated array of Directives.
	 * 
	 * @return the associated array of Directives
	 * @see    #setDirectives(Directive[])
	 */
	public Directive[] getDirectives() {
		return directives;
	}

	/**
	 * Overrides java.lang.Object.equals() to indicate 
	 * whether some other object is "equal to" this Solution
	 * object.
	 * 
	 * @param  object  the reference object with which to compare.
	 * @return true if this object is the same as the object argument otherwise false.
	 */
	public boolean equals(Object object) {

		if ((object == null) || (!(object instanceof Solution)))
			return false;

		Solution testSolution = ((Solution) (object));
		String variable;

		//Test id:
		variable = testSolution.getId();

		if (((id != null) && (!id.equals(variable))) || ((id == null) && (variable != null)))
			return false;

		//Test description:
		variable = testSolution.getDescription();

		if (((description != null) && (!description.equals(variable))) || ((description == null) && (variable != null)))
			return false;

		//Test directives:
		if (!Arrays.equals(directives, testSolution.getDirectives()))
			return false;

		return true;
	}

	/**
	 * Overrides java.lang.Object.toString() to return 
	 * a string representation of this Solution object.
	 * 
	 * @return the string representation of this object.
	 */
	public String toString() {
		StringBuffer toStringBuffer = new StringBuffer("com.ibm.etools.analysis.engine.Solution [");

		toStringBuffer.append("id = ");
		toStringBuffer.append(id);
		toStringBuffer.append(", description = ");
		toStringBuffer.append(description);
		toStringBuffer.append(", directives = ");

		if (directives == null)
			toStringBuffer.append("null");
		else
			toStringBuffer.append(Arrays.asList(directives).toString());

		toStringBuffer.append("]");

		return (toStringBuffer.toString());
	}
}