/**********************************************************************
 * Copyright (c) 2003, 2004 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM Corporation - initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.http.runner;

import java.io.InputStream;
import java.io.OutputStream;
import java.net.Socket;

import org.eclipse.hyades.test.http.runner.internal.exec.HttpRequestHandler;
import org.eclipse.hyades.test.http.runner.internal.exec.SSLHttpExecutor;

//import org.eclipse.hyades.test.common.util.BaseString;

/**
 * @author marcelop
 * @since 1.0.2
 */
public class HttpExecutor
{
	private static int iSSLClassesAvailable = -1;
	private String strLastHost = null;
	private int iLastPort = 0;
	private Socket socket = null;
	private InputStream from_server = null;
	private OutputStream to_server = null;
	private Object sslExecutor = null;	
	private HttpRequestHandler httpRequestHandler = null;
	private int socketBufSize = 0;
	
	public int currentPageNumber;
	public int currentPageOrder;
	public String currentPageName;
	public long pageStart;
	public long pageEnd;
	public long pageResponseTime;
	
	
	public HttpExecutor(){
		iLastPort = 0;
	}
		
	public HttpResponse execute(HttpRequest request)
	throws Exception
	{
		HttpResponse response = new HttpResponse(request);
		// do we support SSL?  A good test is to see if SSLSocket class is available
		// In JVM 1.3.x, the SSL classes are not available within the standard install
		// this allows running a non-ssl playback on 1.3.x  
		if (iSSLClassesAvailable == -1){
			try{
				Class cl = Class.forName("javax.net.ssl.SSLSocket");
				iSSLClassesAvailable = 1;
			}
			catch (ClassNotFoundException cnf){
				iSSLClassesAvailable = 0;			
			}
		}

		try{
			request.getClass().getMethod( "getThinkTime", null );
			long thinkTime = request.getThinkTime();
			if( thinkTime > 0 )
				Thread.sleep( thinkTime );
		} catch (Exception e){}
		
		if (httpRequestHandler == null){
			httpRequestHandler = new HttpRequestHandler();
		}
			
		currentPageNumber = request.getPageNumber();
		currentPageOrder = request.getPageOrder();
		
		long start = System.currentTimeMillis();
		if ((currentPageOrder == HttpRequest.PAGE_START) || (currentPageOrder == HttpRequest.PAGE_ONLY)) 
		{
			pageStart = start;
			currentPageName = request.getURL();
		}
		
		
		if (iSSLClassesAvailable == 1 && request.getSecure() == true){
			SSLHttpExecutor ssl = null;
			if (sslExecutor == null)
				sslExecutor = new SSLHttpExecutor(httpRequestHandler);
			ssl = (SSLHttpExecutor)sslExecutor;	
			ssl.execute(request, response);
			setResponseEndingData(request, response, start);			
			return response; 
		}

		else if (iSSLClassesAvailable == 0 && request.getSecure() == true)
		{
			System.out.println(HttpResourceBundle.getInstance().getString("SSL_NOTSUPPORTED"));
			//TODO: we should probably just bail out here
		}
		String strHost = request.getHost();
		int port = request.getPort();
		
		if (port != iLastPort || strLastHost == null ||
			 strHost.regionMatches(0, strLastHost, 0, strLastHost.length()) != true){
			 	if ((connectToServer(response, strHost, port)) == false){
			 		response.setCode(-1);
			 		return response;
			 	}	
		}
		/* In the following code, if the first attempt to send data to the server fails,
		 * connect to the server again.  If the connection fails, return, otherwise
		 * send the request on the new connection.  If that fails, give up, otherwise,
		 * proceed to get the server response.
		 */
	
		 
		if (httpRequestHandler.sendRequest(request, to_server) == false){
			if (connectToServer(response, strHost, port) == false){
				response.setCode(-1);
				return response; 
			}
			else
			{
				if (httpRequestHandler.sendRequest(request, to_server) == false){
					response.setCode(-1);
					return response;
				}	
			}
		}
		httpRequestHandler.getServerResponse(request, response, from_server, socketBufSize);
		if (response.getCode() == 0){
			if (connectToServer(response, strHost, port) == true){
				if (httpRequestHandler.sendRequest(request, to_server) == true){
					httpRequestHandler.getServerResponse(request, response, from_server, socketBufSize);
				}
			}
		}
		if (response.getShouldCloseSocket() == true)
			strLastHost = null; // make sure the next request opens a new socket
		setResponseEndingData(request, response, start);		
		return response;
	}
	
	private boolean setResponseEndingData(HttpRequest request, HttpResponse response, long start){
		long end = System.currentTimeMillis();
	
		//If PAGE_END or PAGE_ONLY, we need the end timestamp; otherwise ignore.
		if (currentPageOrder == HttpRequest.PAGE_END || currentPageOrder == HttpRequest.PAGE_ONLY)
		{
			pageEnd = end;
			//System.out.println("Page " + currentPageNumber + " End: " + pageEnd);
		}
		
		if (request.getPageOrder() == HttpRequest.PAGE_ONLY || request.getPageOrder() == HttpRequest.PAGE_END)
		{
			pageResponseTime = (pageEnd - pageStart);
			System.out.println(HttpResourceBundle.getInstance().getString("PAGERESPONSE_URL") + " " + currentPageName 
				+ " " + HttpResourceBundle.getInstance().getString("PAGERESPONSE_TIME") + " " + pageResponseTime);
		}
		else
			pageResponseTime = -1;
			
		response.setElapsedTime(end-start);
		response.setPageResponseTime(pageResponseTime);
		
		return true;
	}
	private boolean connectToServer(HttpResponse response, String strHost, int port){	
	
		try {
			if (socket != null) // close the existing socket, if any, before creating a new one
				socket.close();
			socket = new Socket(strHost, port);
			from_server = socket.getInputStream();
			to_server = socket.getOutputStream();
			socketBufSize = socket.getReceiveBufferSize();
			iLastPort = port;
			strLastHost = strHost;
		}
		catch (Exception e){
			response.setCode(-1);
			response.setDetail(e.getLocalizedMessage());
			return false;
		}
		return true;
	}
}
