/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.log.ui.internal.views;

import org.eclipse.hyades.analysis.engine.*;

/**
 * Implementation of the IAnalysisEngine interface.  
 * <p>
 * This class provides a static API to retrieve a new instance 
 * of an AnalysisEngine.
 * <p>
 * An Analysis Engine instance uses one symptom database to 
 * search for zero or more <code>Directive</code>(s) based 
 * on zero or more <code>Incident</code> matches.
 * <p>
 * The symptom database is loaded into an EMF model  
 * structure from a predefined formatted XML file.  
 * The symptom database may be reloaded/merged/removed/replaced.
 * <p>
 * <b>
 * Usage:
 * </b>  
 * <pre>
 * AnalysisEngine analysisEngine = AnalysisEngine.getInstance(symptomDBPath)
 * Object[] analysis = analysisEngine.analyze(incident);
 * </pre>
 */
public final class AnalysisEngine implements IAnalysisEngine {

	/**
	 * Current symptom database.
	 */
	private XMISymptomDatabase symptomDatabase = null;
	
	/**
	 * Private constructor to force use of getInstance().
	 */
	private AnalysisEngine() {
	}

	/**
	 * Private constructor to force use of getInstance(symptomDbPath).
	 * <p>
	 * A new XMI symptom database is created
	 * and the symptom database XMI file denoted by the parameter path
	 * is attempted to be loaded.
	 * 
	 * @param symptomDbPath the path to the symptom database XMI file
	 */

	private AnalysisEngine(String symptomDbPath) {

		symptomDatabase = new XMISymptomDatabase();
		
		replaceSymptomDatabase(symptomDbPath);
	}

	/**
	 * Static API to retrieve a new instance of an AnalysisEngine 
	 * without an associated symptom database.
	 * <p>
	 * <b>NOTE:</b> A base symptom database may be added using 
	 * <code>replaceSymptomDatabase(symptomDbPath)</code>.
	 * 
	 * @return the newly created AnalysisEngine instance
	 * @see    #replaceSymptomDatabase(String)
	 */
	public static AnalysisEngine getInstance() {
		return (new AnalysisEngine(null));
	}

	/**
	 * Static API to retrieve a new instance of an AnalysisEngine.
	 * <p>
	 * 
	 * @param  symptomDbPath the path to the symptom database XMI file
	 * @return the newly created AnalysisEngine instance
	 */
	public static AnalysisEngine getInstance(String symptomDbPath) {
		return (new AnalysisEngine(symptomDbPath));
	}

	/**
	 * Reloads the base symptom database 
	 * from the base symptom database XMI file.
	 * 
	 * @return true if the symptom database has been successfully reloaded
	 */
	public synchronized boolean reloadSymptomDatabase() {
		return (symptomDatabase.load());
	}

	/**
	 * Loads a new symptom database XMI file denoted by the parameter path
	 * and merges any unique records with the base symptom database  
	 * 
	 * @param  symptomDbPath the path to the merged symptom database XMI file
	 * @return true if the new symptom database has been successfully merged 
	 */
	public synchronized boolean mergeSymptomDatabase(String symptomDbPath) {
		return (symptomDatabase.merge(symptomDbPath));
	}

	/**
	 * Removes the base symptom database.
	 * <p>
	 * <b>NOTE:</b> A base symptom database may be added using 
	 * <code>replaceSymptomDatabase(symptomDbPath)</code>.
	 * 
	 * @return true if the symptom database has been successfully removed
	 * @see    #replaceSymptomDatabase(String)
	 */
	public synchronized boolean removeSymptomDatabase() {
		return (symptomDatabase.replace(null));
	}

	/**
	 * A new XMISymptomDB symptom database is created.
	 * and the symptom database XMI file denoted by the parameter path 
	 * is attempted to be loaded.
	 * <p>
	 * If the new symptom database XMI file was successfully loaded, it
	 * replaces the base symptom database. 
	 * 
	 * @param  symptomDbPath the path to the symptom database XMI file
	 * @return true if the new symptom database has been successfully replaced 
	 */
	public synchronized boolean replaceSymptomDatabase(String symptomDbPath) {

		if ((symptomDbPath != null) && (!symptomDbPath.trim().equals(""))) {

			return symptomDatabase.replace(symptomDbPath);

		}

		return false;
	}

    /**
     * Returns the XMI file path of the base symptom database.
     * <p>
     * If no base symptom database is loaded, null is returned.
     * 
     * @return the file path of the base symptom database, otherwise null
     */
    public synchronized String getSymptomDatabasePath(){
    
        if(symptomDatabase != null)
            return symptomDatabase.getPath();
    
        return null;
    }

	/**
	 * The current symptom database is
	 * searched for <code>Incident</code> matches.
	 * <p>
	 * All <code>Directive</code>(s) from successful <code>Incident</code> 
	 * matches are returned in an array structure.
	 * <p>
	 * <code>Incident</code> matches are based on a lexicographical String
	 * comparison between the <code>messageId</code> in the parameter
	 * <code>Incident</code> and the <code>value</code> attribute of the
	 * <code>matchPattern</code> Node in the base symptom database.
	 * 
	 * @param  incident the search criteria used for possible matches
	 * @return an array of <code>Directive</code> from successful <code>Incident</code> matches 
	 * @see    Directive
	 * @see    Incident
	 */
	public synchronized Object[] analyze(Incident incident) {
		return (symptomDatabase.getDirectives(incident));
	}

	/**
	 * The current symptom database is searched for <code>Incident</code> 
	 * matches.
	 * <p>
	 * All <code>Solution</code>(s) from successful <code>Incident</code> 
	 * matches are returned in an array structure.
	 * <p>
	 * <code>Incident</code> matches are based on a lexicographical String
	 * comparison between the <code>messageId</code> and <code>rawData</code>
	 * array in the parameter <code>Incident</code> and the <code>value</code> 
	 * attribute of the <code>matchPattern</code> Node in the base symptom 
	 * database.
	 * 
	 * @param  incident the search criteria used for possible matches
	 * @return an array of <code>Solution</code> from successful <code>Incident</code> matches 
	 * @see    Solution
	 * @see    Incident
	 */
	public Solution[] analyzeForSolutions(Incident incident) {
		return (symptomDatabase.getSolutions(incident));
	}
}