package org.eclipse.hyades.logging.adapter;

/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

import org.eclipse.hyades.logging.adapter.impl.Controller;
import org.eclipse.hyades.logging.adapter.util.Messages;

/**
 * Adapter is a simple main entry point to run an adapter controller.
 */
public class Adapter {

    private Controller rootController = null;
    private String contextFile = null;
    private String componentFile = null;
    

    public void setContextConfigPath(String configFilePath) {
        contextFile = configFilePath;
    }

    public void setComponentConfigPath(String configFilePath) {
        componentFile = configFilePath;
    }
    
    /**
     * Setup the adapter and validate the configuration file.  This method
     * will not run the adapter and stops after validation.
     * @throws AdapterException if the configuration is invalid
     */
    public void validate() throws AdapterException {
    	rootController = new Controller();

        if (componentFile == null) {
            componentFile = Messages.getString("HyadesGADefaultComponentConfigurationsFile"); //$NON-NLS-1$
        }

        if (contextFile == null) {
            contextFile = Messages.getString("HyadesGADefaultContextConfigurationFile"); //$NON-NLS-1$
        }

        rootController.setContextConfigPath(contextFile);
        rootController.setComponentConfigPath(componentFile);
        
        /* try and prepare the adapter for running.  This may fail */
        rootController.prepareAdapter();
    	
    }

    public void start(boolean separateThread, boolean daemon) throws AdapterException {

        rootController = new Controller();

        if (componentFile == null) {
            componentFile = Messages.getString("HyadesGADefaultComponentConfigurationsFile"); //$NON-NLS-1$
        }

        if (contextFile == null) {
            contextFile = Messages.getString("HyadesGADefaultContextConfigurationFile"); //$NON-NLS-1$
        }

        rootController.setContextConfigPath(contextFile);
        rootController.setComponentConfigPath(componentFile);
        
        /* try and prepare the adapter for running.  This may fail */
        rootController.prepareAdapter();

        if (!separateThread) {
            rootController.run();

        }
        else {
            Thread serverThread = new Thread(rootController);
            serverThread.setDaemon(daemon);
            serverThread.start();
        }
    }

    public void stop() {
        if (rootController != null) {
            rootController.stop();
        }
    }
	/**
	 * This stop method will still cause the contexts to be flushed but will
	 * not call the context listener methods.
	 */
	public void hardStop() {
		if (rootController != null) {
			rootController.hardStop();
		}
	}

    public static void main(String[] args) {

        Adapter adapter = new Adapter();

        if (adapter.parseCommandLine(args)) {
            try {
                adapter.start(false, false);
            }
            catch (AdapterException e) {
            	/* RKD:  If we get an exception at this point we likely do not have a valid configuration and hense
            	 * the log file is likely not flushed.  We will need to write this to stderr.
            	 */
            	System.err.println(Messages.getString("HyadesGAAdapterFatal_ERROR_"));
            	System.err.println(e.getMessage());
                adapter.stop();
            }
        }
        else {
            System.out.println(Messages.getString("HyadesGACommand_line_usage"));
        }
    }

    private boolean parseCommandLine(String[] args) {

        if ((args != null) && (args.length > 0)) {

            String ccArgName = Messages.getString("HyadesGA-cc");
            String acArgName = Messages.getString("HyadesGA-ac");

            //Process all arguements:
            for (int i = 0; i < args.length; i++) {

                //If this is the -cc arguement, derive the context configuration:
                if (ccArgName.equalsIgnoreCase(args[i])) {

                    if ((++i < args.length) && (args[i] != null)) {
                    	setContextConfigPath(args[i]);
                    }
					else{
						return false;
					}
                }

                //If this is the -ac arguement, derive the the component configuration:
                else if (acArgName.equalsIgnoreCase(args[i])) {

                    if ((++i < args.length) && (args[i] != null)) {
                    	setComponentConfigPath(args[i]);   
                    }
                    else{
                    	return false;
                    }
                }

                //If there is any other argument, fail:
                else {
                    return false;
                }
            }
        }

        return true;
    }
}