/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.logging.adapter.parsers;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.ListIterator;
import java.util.StringTokenizer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;


import org.eclipse.hyades.logging.adapter.AdapterException;
import org.eclipse.hyades.logging.adapter.IDirectedGraph;
import org.eclipse.hyades.logging.adapter.internal.util.SubstitutionExtensionLoaderUtil;
import org.eclipse.hyades.logging.adapter.util.Messages;
import org.eclipse.hyades.logging.adapter.util.TimeParser;
import org.w3c.dom.Element;

/**
 * @author rduggan
 *
 * This class represents a substitution rule.  It uses the Jakarta-ORO library
 * for processing regular expressions.
 */
public class SubstituteStatement implements IStatement {
	
	/* The fully qualified name for this substitution rule */
	protected String name;
	
	/* The positions string as specified in the rule */
	protected String positions;
	
	/* The match string as specified in the rule */
	protected String match;
	
	/* The substitute string as specified in the rule */
	protected String substitute;
	
	/* The compiled pattern for this SubstitutionStatement */
	protected Pattern pattern;

	/* The list of positions that will be built based upon the positions string */
	protected List positionsList;
	
	/* The process time indicator */
	protected boolean processTime;
	
	/* Time parser class */
	protected TimeParser timeParser;
		
	/* Members to implement the java callout for processing the parsed data */	
	/* The user class to be used to process the parsed data to generate a value */
	protected String substitutionExtensionClass;
	
	/* The statefulSubstitutionExtension object */
	protected IStatefulSubstitutionExtension statefulExtension = null;
	
	/* The indicator for callout processing */
	protected boolean callout = false;
	
	/* The indicator to always create a new instance of the callout class */
	protected boolean calloutNewInstance = true;
	
	/* The substititution extension class object */
	protected Class substitutionExtensionClassClass = null;
			
	/* The builtin function indicator */
	protected boolean useBuiltin;
	
	/* The parsed path for this rule.  This is loaded into the IDirectedGraph to identify 
	 * this rule uniquely to the next component in the chain.
	 */
	protected List path;
	
	protected IDirectedGraph resultGraph;
	
	/**
	 * @see org.eclipse.hyades.logging.adapter.IStatement#prepare(org.w3c.dom.Element)
	 */
	public void prepare(Element node, String path) throws PreparationException {
		positions = node.getAttribute(Messages.getString("HyadesGAPositionsAttributeName"));
		match=node.getAttribute(Messages.getString("HyadesGAMatchAttributeName"));
		substitute=node.getAttribute(Messages.getString("HyadesGASubstituteAttributeName"));
		substitutionExtensionClass=node.getAttribute(Messages.getString("HyadesGAsubstitutionExtensionClassAttributeName"));
		/* User can specify a Java time format that the substitution value will be in.
		 * This format will be used to convert the time string to a long value in milliseconds  
		 */
		String timeFormat=node.getAttribute(Messages.getString("HyadesGAtimeFormatAttributeName"));
		String builtin=node.getAttribute(Messages.getString("HyadesGAuseBuiltInFunctionAttributeName"));
		if(builtin!=null && !builtin.equals("")) {
			if(builtin.equals("true")) {
				useBuiltin=true;
			}
			else {
				useBuiltin=false;
			}
		}
		else {
			useBuiltin=false;
		}

		/* Save the name. */
		this.name=path;

		/* If a time format was specified and use builtin was not then set up the time
		 * parser.
		 */
		if (timeFormat!=null && !timeFormat.equals("") && !useBuiltin) {
			processTime = true;
			try {
				timeParser = new TimeParser(timeFormat);
			}
			catch (AdapterException e) {
				PreparationException details = new PreparationException(Messages.getString("HyadesGA_CBE_Parser_Preparation_Invalid_Time_Format_ERROR_",timeFormat), e);
				details.setDigraph(this.name);
				throw details;
			}
		}
								
		/* If a substition extension class was specified and use builtin was not then
		 * check if it is a stateful or stateless class.  
		 * If stateful, get an instance of the class to use when processing all records.
		 */
		if (substitutionExtensionClass!=null && !substitutionExtensionClass.equals("") && !useBuiltin && !processTime) {
			callout = true;
			ISubstitutionExtension extension = null;

			try {
				/* Try to get the substitution extension class class in case we have to instantiate
				 * the class at run time (for stateless classes)
				 */
				substitutionExtensionClassClass = Class.forName(substitutionExtensionClass);
			}
			catch (Throwable e) {
				// We don't need to do anything at this point.
			}
				
			try {
				// load the class and instantiate an object
				extension = SubstitutionExtensionLoaderUtil.instantiate(substitutionExtensionClassClass, substitutionExtensionClass);
			}
			catch (ClassCastException e) {
				PreparationException details = new PreparationException(Messages.getString("HyadesGA_CBE_Parser_Run_SubstitutionExtensionClass_Invalid_ERROR_",substitutionExtensionClass), e);
				details.setDigraph(this.name);
				throw details;
			}			
			catch (ClassNotFoundException e) {
				PreparationException details = new PreparationException(Messages.getString("HyadesGA_CBE_Parser_Preparation_SubstitutionExtensionClass_NotFound_ERROR_",substitutionExtensionClass), e);
				details.setDigraph(this.name);
				throw details;
			}
			catch(Exception e){
				PreparationException details = new PreparationException(Messages.getString("HyadesGA_CBE_Parser_Preparation_SubstitutionExtensionClass_Exception_ERROR_",substitutionExtensionClass), e);
				details.setDigraph(this.name);
				throw details;
			}
			
			try {
				// Check which type of class we have (stateless or stateful
				statefulExtension = (IStatefulSubstitutionExtension)extension;
				calloutNewInstance = false;
			}
			catch (ClassCastException e) {
				calloutNewInstance = true;
			}
			
		}

		/* Clear the result graph */
		resultGraph=null;
		
		/* Store the path as a list. The path may contain indexes for elements
		 * which are multiples.
		 */
		this.path=new ArrayList();
		StringTokenizer s = new StringTokenizer(path, ".");
		while (s.hasMoreTokens()) {
			String subPath=s.nextToken();
			int indexStart=subPath.indexOf('[');
			if(indexStart>0) {
				String index=subPath.substring(indexStart+1, subPath.length()-1);
				subPath=subPath.substring(0, indexStart);
				this.path.add(subPath);
				this.path.add(index);
			}
			else {
				this.path.add(subPath);
			}
		}
		
		/* Parse the positions attribute value into a tokenized list */
		if(positions!=null && !positions.equals("")) {
			positionsList=PositionParser.getPositionedString(positions, true);
		}
		
	
		/* Compile the match statement */
		if (match!=null && !match.equals("")) {
			compileMatchString(match);
		}

	}

	/*
	 * Compile the match string and set the match pattern
	 */
	protected void compileMatchString(String match) throws PreparationException {
		try {
			pattern=Pattern.compile(match);
		}
		catch(PatternSyntaxException e) {
			/* This is a badly formed regular expression.  Throw it as an error */
			PreparationException details=new PreparationException(Messages.getString("HyadesGAMalformedParserExpression_ERROR_"), e);
			details.setDetails("\""+match+"\"");
			details.setDigraph(this.name);
			throw details;
		}
	}
	
	/**
	 * @see org.eclipse.hyades.logging.adapter.IStatement#run(java.lang.String)
	 */
	public boolean run(String line, HashMap inputs, List outputs) throws AdapterException {
		String tmpLine = line;
		String newLine;
		boolean result=false;
		
		
		/* If we are running a builtin function then ignore the reset of the content */
		if(useBuiltin) {
			resultGraph=new DirectedGraphImpl();
			resultGraph.setPath(this.path);
			resultGraph.setValue("##BUILTIN");
			outputs.add(resultGraph);
			return true;
		}
		
		/* If we have a positions string modify our inputs accordingly */
		if(positions!=null && !positions.equals("")) {
			tmpLine=positionsToString(line, inputs);
		}
		
		/* Continue only if the specified position string results in valid data */
		if (tmpLine != null) {
		
			/* If we have a match pattern to look for then lets get the results.  Otherwise
			 * we just place the substitution string in the value.
			 */
			if (match!=null && !match.equals("")) {
				/* There must be a a substitute value in order to do the substitution */
				if(substitute!= null && !substitute.equals("")) {
					newLine=matchAndSubstitute(tmpLine);
				}
				else {
					newLine = null;
				}
				
				result=(newLine!=null);	
			}
			else {
				if (substitute!=null && !substitute.equals("")) {
					newLine=substitute;	
				}
				else {
					newLine=tmpLine;	
				}
				result=true;
			}
			
			
			/* If we have a match add this match to the result set.  This
			 * is quite a bit of object creation.  Perhaps we could clean this
			 * up some more.
			 */
			if(result) {
				/* If there is a time to parse then do it */
				if (processTime) {
					long convertedTime = timeParser.parse(newLine);
					
					if (convertedTime == 0) {
						resultGraph = null;
						return false;						
					}
					else {
						outputs.add(createResultGraph(new Long(convertedTime)));
						return true;
					}
				}
				
				/* If user wants to use a callout method, do it now */
				if (callout) {
	
					if (calloutNewInstance) {
						ISubstitutionExtension statelessExtension; 
						try {
							if (substitutionExtensionClassClass == null) {
								// If we couldn't find the class in prepare, then load it using the plugin extension
								statelessExtension = SubstitutionExtensionLoaderUtil.instantiate(substitutionExtensionClass);
							}
							else {
								// Instantiate an object
								statelessExtension = (ISubstitutionExtension)substitutionExtensionClassClass.newInstance();
							}
						}
						catch (Throwable e) {
							throw new AdapterException(Messages.getString("HyadesGA_CBE_Parser_Preparation_SubstitutionExtensionClass_Exception_ERROR_", substitutionExtensionClass) , e);
						}
						
						try {
							/* Call the appropriate substitution extension class method */	
							if (newLine == line) {
								newLine = statelessExtension.processRecord(newLine);
							}
							else {
	
								newLine = statelessExtension.processMatchResult(newLine);
							}
						}
						catch(Exception e){
							throw new AdapterException(Messages.getString("HyadesGA_CBE_Parser_Run_StatelessSubstitutionExtensionClass_Exception_ERROR_", substitutionExtensionClass) , e);
						}
					}
					else {
						try {
							/* Call the appropriate substitution extension class method */
							if (newLine == line) {
								newLine = statefulExtension.processRecord(newLine);
							}
							else {			
								newLine = statefulExtension.processMatchResult(newLine);
							}
						}
						catch (Exception e) {
							throw new AdapterException(Messages.getString("HyadesGA_CBE_Parser_Run_StatefulSubstitutionExtensionClass_Exception_ERROR_", substitutionExtensionClass) , e);				
						}
					}
					if (newLine == null) {
						resultGraph = null;
						return false;
					}
				}
				
				/* RKD:  We are instantiating a new instance here each pass.  This isn't
				 * necessary providing the contract between the Parser and the next element
				 * in the chain is on the same thread and the contract states they will
				 * not be changed.  I am leaving the instantiation because I believe we should
				 * support each component running in its own thread. 
				 */

				outputs.add(createResultGraph(newLine));
				return true;
			}

		}
		
		resultGraph=null;
		
		return false;
	}
	
	/**
	 * Check if we have a match and if so do the substitution
	 * @param line to check for pattern match and to do substitution on
	 * @return String resulting string from matching and substitution 
	 */

	protected String matchAndSubstitute (String line) {
		/* We must always have a substitute string */
		if(substitute!= null && !substitute.equals("")) {
			/* Return null if we don't have a match */
			Matcher matcher = pattern.matcher(line);
			if(!matcher.find()) {
				return null;
			} 
			/* Perform the substitution */
		
			String result = matcher.replaceAll(substitute);

			return result.trim(); 
		}
		return null;
		
	}
	
	/**
	 * This method iterates over the positions string and builds a new string
	 * based upon the environment entires.  For each position that is a key into
	 * the enviornment we extract the value and append it to the previous.  All 
	 * of the values are separated by the postion separator.
	 * 
	 * @param line
	 * @param inputs
	 * @return
	 */
	protected String positionsToString(String line, HashMap inputs) {
		if (positionsList!=null)	{
			
			/* Iterate over the tokenized postion information and build
			 * a new line
			 */
			 StringBuffer result=new StringBuffer();
			 ListIterator iterator=positionsList.listIterator();
			 boolean first=true;
			 boolean emptyString=false;
			 while(iterator.hasNext()) {
			 	/* Grab the next token.  It may be a String or a Long */
			 	Object current=iterator.next();
			 	if(first) {
					first=!first;	
			 	}
			 	else {
			 		result.append(PositionParser.POSITION_SEPARATOR);
			 	}
			 	if(current instanceof String) {
					String val=(String)inputs.get(current);
					if (val!=null && val.length()>0) {
						result.append(val);
					}
					else {
						return null;
					}
			 	}
			 	else if(current instanceof Long) {
					Long item = (Long)current;
					String val=null;
					int index = item.intValue()-1;
					val = (String)inputs.get(new Long(index));
					if (val!=null && val.length()>0) {
						result.append(val);
					}
					else {
						return null;
					}
				}
			}
			
			return result.toString();			
		}
		return line;	
	}

	/**
	 * Retrieve the IDirectedGraph created as a result of this rule being run.
	 * @return
	 */
	public IDirectedGraph getResultGraph() {
		return resultGraph;
	}

	/**
	 * Create the IDirectedGraph for this rule with a specific value.
	 * @return
	 */
	public IDirectedGraph createResultGraph(Object value) {
		resultGraph=new DirectedGraphImpl();
		resultGraph.setPath(this.path);
		resultGraph.setValue(value);
		return resultGraph;
	}
}
