/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.logging.adapter.sensors;
import java.io.BufferedReader;
import java.io.EOFException;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.RandomAccessFile;
import java.util.Hashtable;

import org.eclipse.hyades.logging.adapter.AdapterInvalidConfig;
import org.eclipse.hyades.logging.adapter.impl.Sensor;
import org.eclipse.hyades.logging.adapter.util.Messages;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
/**
 * Sample Implementation of an OS File Sensor.
 *
 *
 * @version 0.1
 * @since 0.1
 * @see org.eclipse.hyades.logging.adapter.impl.Sensor
 * @see org.eclipse.hyades.logging.adapter.ISensor
 */
public class SingleOSFileSensor extends Sensor
{
	/* The relative or absolute directory containing the log file. */
	private String directory = null;
	
	/* The name including extension of the log file.
	 *
	 * Note: This value may contain a regular expression.
	 */
	private String fileName = null;
	
	/* The full (e.g. absolute paths) command string of the converter application. */
	private String converterCommand = null;
	
	/* The shell to run the converter in */
	private String converterShell = null;
	
	/* The logfile we are processing */
	private String rawLogFileName = null;
	
	/* bThe mechanism for reading the file */
	private BufferedPeriodicReader input = null;
	
	
	/* allocate the max blocking size  */
	private String[] lastLine = new String[maximumBlocking];
	
	/* Current thread lock for spawning child threads. */
	private Object currentThreadLock = new Object();
	
	/* Counter for tracking completed child threads. */
	private int childThreadsDone = 0;
	
	/* If the directory does not end with the proper file
	 * separator then we need to add one.  RKD:  We probably
	 * need to make sure we convert incorrect path separators
	 * to the proper platform.
	 */
	private String pathSeparator=System.getProperty("file.separator");
	
	
	/**
	 * No-arguement constructor to create a SingleOSFileSensor.
	 */
	public SingleOSFileSensor()
	{
		super();
	}
	/**
	 * Stops the sensor from reading the log file.  All object
	 * references are cleaned, file handles are closed and temporary
	 * files deleted.
	 *
	 * @see org.eclipse.hyades.logging.adapter.IComponent#stop()
	 */
	public void stop()
	{
		super.stop();
		clean();
	}
	/**
	 * Update the configuration based on the configuration Element.
	 *
	 * @see org.eclipse.hyades.logging.adapter.IComponent#update()
	 */
	public void update() throws AdapterInvalidConfig
	{
		// first get the basic configuration set
		super.update();
		// maximumBlocking is set by the sensor config
		lastLine = new String[maximumBlocking];
		String directory = null;
		String fileName = null;
		String converterCmdAttribute = null;

		Element sensorTypeInstance = null;
		Element sensorNode;
		Element element = getConfiguration();

		// Get the sensor parameters from the sensor properties
		Hashtable sensorProperties = getProperties();		
		if (sensorProperties != null && !sensorProperties.isEmpty()) {
			directory = (String)sensorProperties.get(Messages.getString("HyadesGAdirectoryAttributeName"));

			fileName = (String)sensorProperties.get(Messages.getString("HyadesGAfileNameAttributeName"));
			
			converterCmdAttribute = (String)sensorProperties.get(Messages.getString("HyadesGAconverterCmdAttributeName"));
			
			converterShell = (String)sensorProperties.get(Messages.getString("HyadesGAconverterShellAttributeName"));										
		}
		else {
			// If there are no properties then get sensor parameters from the old SingelFileSensor type instance
			NodeList sensorNodes = element.getChildNodes();
			for (int i = 0; i < sensorNodes.getLength(); i++) {
				if (sensorNodes.item(i).getNodeType() == Node.ELEMENT_NODE) {
					sensorNode = (Element) sensorNodes.item(i);
	
					/* Else the configuration is the old style SingleFileSensor type element */
					if (sensorNode.getTagName().equals(Messages.getString("HyadesGASingleFileSensorTagName"))) {
						// Get the sensor parameters from the sensor type instance attributes
						sensorTypeInstance = sensorNode;
						if (sensorTypeInstance.hasAttribute(Messages.getString("HyadesGAdirectoryAttributeName"))) {
							directory = sensorTypeInstance.getAttribute(Messages.getString("HyadesGAdirectoryAttributeName"));
						}
						if (sensorTypeInstance.hasAttribute(Messages.getString("HyadesGAfileNameAttributeName"))) {
							fileName = sensorTypeInstance.getAttribute(Messages.getString("HyadesGAfileNameAttributeName"));
						}
	
						if (sensorTypeInstance.hasAttribute(Messages.getString("HyadesGAconverterCmdAttributeName"))) {
							converterCmdAttribute = sensorTypeInstance.getAttribute(Messages.getString("HyadesGAconverterCmdAttributeName"));
						}
	
						if (sensorTypeInstance.hasAttribute(Messages.getString("HyadesGAconverterShellAttributeName"))) {
							converterShell = sensorTypeInstance.getAttribute(Messages.getString("HyadesGAconverterShellAttributeName"));
						}					
					}
				}
			}
		}
		
		if (converterCmdAttribute != null) {
			converterCmdAttribute = converterCmdAttribute.trim();
			if (converterCmdAttribute.length() == 0) {
				converterCmdAttribute = null;
			}
		}
		
		if (converterShell != null) {
			converterShell = converterShell.trim();
			if (converterShell.length() == 0) {
				converterShell = null;
			}
		}
		//We must have a vaild directory and fileName that exists on the local file system:
		if (directory == null || directory.trim().length() == 0 ||
			fileName == null || fileName.trim().length() == 0 ||
			(converterCmdAttribute == null && !(new File(directory,fileName).canRead()))) {
			throw new AdapterInvalidConfig(Messages.getString("HyadesGA_CBE_SingleFileSensor_Invalid_Config_File_ERROR_"));
		}
		
		setDirectory(directory.trim());
		setFileName(fileName.trim());
		setConverterCommand(converterCmdAttribute);
	}
	/**
	 * 	simulates a getNext
	 * @see com.ibm.acad.general.sensor.ISensor#testGetNext()
	 */
	public Object[] testGetNext()
	{
		return testGetNextLine();
	}
	/**
	 * implements the testGetNext behaviour by returning a test string
	 * @return String[]
	 */
	public String[] testGetNextLine()
	{
		String[] lineArray = new String[2];
		lineArray[0] = "test string";
		return lineArray;
	}
	
	/**
	 * returns last read line
	 * @see com.ibm.acad.general.sensor.ISensor#getNext()
	 */
	public Object[] getNext() {
		
		
		/* On our first pass through getNext() and subsequent calls after we have reached the end of the file
		 * we will need to do any pre-processing before reading the data.  If we are flushing do not run the converter again
		 */
		if (rawLogFileName == null  && !flushingMode) {
			if(getDirectory().endsWith(pathSeparator)) {
				rawLogFileName = getDirectory() + getFileName();
			}
			else {
				rawLogFileName = getDirectory() + pathSeparator + getFileName();
			}
			
			if(rawLogFileName != null) {
				try {
					if(input==null) {
						input = new BufferedPeriodicReader(rawLogFileName,converterCommand, converterShell);
					}
					input.prepare();
				}
				catch (Exception e) {

				    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();
				    event.setMsg(e.toString());
				    event.setSeverity(CommonBaseEvent.SEVERITY_CRITICAL);

				    log(event);
				}    			
			}
		}
		
		/* Retrieve the contents of the file up to maximumBlocking lines */
		if (input != null) {	
			for (int i = 0; i < maximumBlocking; i++) {
				try {
					lastLine[i] = input.readLine();
				}
				catch (Exception e) {

				    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();
				    event.setMsg(e.toString());
				    event.setSeverity(CommonBaseEvent.SEVERITY_WARNING);

				    log(event);
				}
				if (lastLine[i] == null) {
					i = maximumBlocking;
					rawLogFileName=null;
				}
			}
		}
		
		/* Was there any data that we retrieved?  if so, return it, otherwise indicate there is no more data. */
		if (lastLine[0] != null) {
			return lastLine;
		}
		else {
			return null;
		}
	}
	
	/**
	 * do all cleaning here
	 */
	public void clean() {
		try	{
			if (input != null)
			{
				input.close();
			}
		}
		catch (Exception e) {
		
		    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();
		    event.setMsg(e.toString());
		    event.setSeverity(CommonBaseEvent.SEVERITY_HARMLESS);

		    log(event);
		}
	}
	
	/**
	 * Returns the directory.
	 * @return String
	 */
	final public String getDirectory()
	{
		return directory;
	}
	
	/**
	 * Returns the fileName.
	 * @return String
	 */
	final public String getFileName()
	{
		return fileName;
	}
	
	/**
	 * Returns the converter command.
	 * @return String
	 */
	final public String getConverterCommand()
	{
		return converterCommand;
	}
	/**
	 * Sets the directory.
	 * @param directory The directory to set
	 */
	final public void setDirectory(String directory)
	{
		this.directory = directory;
	}
	/**
	 * Sets the fileName.
	 * @param fileName The fileName to set
	 */
	final public void setFileName(String fileName)
	{
		this.fileName = fileName;
	}
	/**
	 * Sets the converter command.
	 * @param converterCmd The converter command to set
	 */
	final public void setConverterCommand(String converterCmd)
	{
		this.converterCommand = converterCmd;
	}
	/**
	 * BufferedPeriodicReader reads a file line by line and populates the returned
	 * string array. It holds a pointer offset to the last read byte in order to resume.
	 */
	private class BufferedPeriodicReader
	{
		private String filename = null;
		private String converter = null;
		
		/* The shell to run the convertor in. */
		private String shell = null;
		
		/* The reader for the current file. */
		private RandomAccessFile currentFile = null;
		
		/* The stat info for the file on disk */
		private File currentFileStat=null;
		
		/*
		 * bugzilla 78521 
		 * Initial size of buffer to read data from log file into. 
		 */ 
		private int initialBufferSize = 2048;
		
		/* The last pointer into the file where we finished reading at. */
		long lastBookmark = 0;
		
		/* The last time this file was changed */
		long lastModified=0;
		
		/* The last size of this file when it was opened (bugzilla 78224) */
		long lastSize=0;
		
		/* A buffer of historic info that we store whenever we hit the end of a file
		 * This historic info is used to locate the previous end of file.  Note:  We
		 * store the information in this buffer in reverse order.  This is because we
		 * traverse the new file in reverse order as well.
		 */
		byte[] historicInfo=null;

		/* line separator characters */
		private String EOL_CHARS = System.getProperty("line.separator");
		
		/* Last line separator character
		 * bugzilla 70772 - on z/OS the line separator character is x'15' but the System.getProperty returns x'0A'
		 * so we explicitly set the last line separator character to x'15' or decimal 21 on z/OS.
		 */
		private char EOL_LAST_CHAR= (System.getProperty("os.name", "Windows").equals("z/OS") || System.getProperty("os.name", "Windows").equals("OS/390")) ? 21 : EOL_CHARS.charAt(EOL_CHARS.length()-1);
		
		/**
		 *
		 * @param newFilename
		 */
		public BufferedPeriodicReader(String newFilename) {
			this(newFilename,null,null);
		}
		
		/**
		 *
		 * @param newFilename
		 * @param converter
		 */
		public BufferedPeriodicReader(String newFilename, String converter)	{
			this(newFilename,converter,null);
		}

		/**
		 *
		 * @param newFilename
		 * @param converter
		 * @param shell
		 */
		public BufferedPeriodicReader(String newFilename, String converter, String shell) {
			this.filename = newFilename;
			this.converter = converter;
			this.shell = shell;
		}	
		
		/**
		 * prepare file for the reading
		 */
		public void prepare() {
			/* If the path to the converter is set, convert the file first. */
			if (converter != null && converter.length() > 0) {
				try {
					//Execute the converter command as a runtime process and monitor any stdout and stderr output:
					Process converterProcess;
					
					//PROBLEM: Because some native platforms only provide limited buffer size for standard input streams, failure to read the input stream of the process may cause the process to block, and even deadlock.
					//SOLUTION: Process input streams must be read on seperate threads.
					synchronized (currentThreadLock) {
						//Reset the counter for tracking completed child threads:
						childThreadsDone = 0;
						
						// Execute the converter command:
							
						String platform = System.getProperty("os.name", null);
					
						// Use the converter command alone on Windows
						if (platform.startsWith("Windows")) {
							converterProcess = Runtime.getRuntime().exec(converter);
						}
						// Run the converter in a shell on all other platforms
						else if (platform.startsWith("AIX")) {
							// For AIX we need to specify the command as an array of strings
							
							// If no shell is specified, use sh
							if (shell == null) {
								shell = "sh";
							}
							
							String [] cmdarr = {shell, converter};
							//Execute the converter command string array
							converterProcess = Runtime.getRuntime().exec(cmdarr);
						}
						else {
							// For all other non-Windows platforms we can run the command as one string
							
							// If no shell is specified, use sh
							if (shell == null) {
								shell = "sh";
							}
							// Add the shell to the converter command
							String converterCmd = shell + " " + converter;
							
							//Execute the converter command string
							converterProcess = Runtime.getRuntime().exec(converterCmd);
						}
						
						//Monitor any stdout and stderr output from the execution of the converter command:
						new ConverterOutputReader(converterProcess.getErrorStream()).start();
						new ConverterOutputReader(converterProcess.getInputStream()).start();

						//ASSUMPTION: The process is only terminated after all its streams (i.e. stdout and stderr) have terminated.
						//Wait for all of the stdout and stderr output to be read before continuing:
						try {
							/* Wait no more than 5 minutes (300000 ms):
							 * RKD:  We need to log a message if this time expires.
							 */
							currentThreadLock.wait(300000);
						}
						catch (InterruptedException e) {
						}
					}

					/* Capture the exit value (by convention, the value 0 indicates normal termination) from the terminated converter process: */
					int exitValue = -1;
					
					try {
						exitValue = converterProcess.waitFor();
					}
					catch (InterruptedException i) {
						/* Destroy the converter process if it has not yet exited: */
						converterProcess.destroy();
					}

					/* Throw an exception is the process has not exited or exits abnormally (by convention, the value 0 indicates normal termination):
					 * RKD:  I think we should just log the exception occured.  Conventions are fine as long as everyone conforms to them.  Experience
					 * has proven this is not always the case.
					 */
					
					if (exitValue != 0) {
						throw new Exception();
					}
				}
				catch (Exception e) {

				    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();
				    event.setMsg(e.toString());
				    event.setSeverity(CommonBaseEvent.SEVERITY_CRITICAL);

				    log(event);
				}
			}
			
			/* RKD:  There are six conditions we need to be aware of and handle at this point.
			 *       1.  File has had content appended to the end.
			 *       2.  File has been replaced but otherwise content is appended to the end
			 *       3.  File has been replaced with data shifted up/left (circular log)
			 *       4.  File has been replaced with all new data.
			 *       5.  New file has been created on size or time boundaries
			 *       6.  Rotating logs with all of the 5 above conditions.
			 *
			 *  The solution below addresses the top four scenarios.  The algorithm is as follows:
			 *     1.  Always copy the last n bytes of the file from the last time we were at the end of
			 *         the file (n is configurable)
			 *     2.  Always save the filepointer from the last time we read the file.
			 *     3.  When the file is reopened, and it has been changed since the last time we read it
			 *         and has grown in size, the content of the file before
			 *         the current filepointer compared to determine if the filepointer is still
			 *         valid.  If it is, then compare the last n bytes of the previous read (where n
			 *         is a configurable value) if these n bytes are identical then conditions 1 and
			 *         2 are satisfied.   Set the new filepointer and remove our historic info saved in 1.
			 *         Else goto 4.
			 *     3.  Start searching from the end of the file until we find a match for
			 *         the historic buffer.  When found compare n bytes to determine this
			 *         is the old end of file. If this occurs condition 3 is satisfied.  Set the new
			 *         filepointer and close the old file.  Else condition 4 is satisfied, close the old
			 *         file and start from the beginning of the new file.
			 */
			
			try	{
				currentFileStat=new File(filename);
				/* If the file has been modified since we opened it OR
				 *    (bugzilla 78224) its size has changed since we opened it
				 * Then open the file again and save the modified time and size
				 */
				if(currentFileStat.lastModified()>lastModified || currentFileStat.length() != lastSize) {
					currentFile = new RandomAccessFile(filename, "r");
					/* RKD:  We likely need to lock the file read only and then set the lastmodified date before reading
					 * the file.  That way there it does not get changed underneigth us.  I have left this out for now.
					 */
					lastModified=currentFileStat.lastModified();
					lastSize = currentFile.length();
				}
			}
			catch (Exception e) {
				currentFile = null;
			}
			
			
			
			/* Is there an old file.  if so we need to satisfy the conditions above.  Otherwise just open the new file. */
			if(lastBookmark!=0 && currentFile!=null) {
				try {
					/* determine how many characters we need to read to validate the file offsets */
					long previousOffset=lastBookmark;
					long currentOffset=currentFile.length();		
					long charactersToCompare=Math.min(Math.min(confidenceBufferSize, previousOffset), currentOffset);
					
					/* Walk the end of the file and try and determine if this is an append */
					if(currentOffset>=previousOffset) {	
						boolean append=true; 				
						for(int i=fileFooterSize; i<charactersToCompare; i++) {
							currentFile.seek(previousOffset-1-i);
							char value=(char)currentFile.read();
							if(historicInfo[i]!=(byte)value) {
								append=false;
								break;
							}
						}	
						/* If this is an append then the footer needs to be accouted for */
						if(append) {
							currentFile.seek(previousOffset-fileFooterSize);
							for(int i=(int)fileFooterSize-1; i>=0; i--) {	
								int value=currentFile.read();
								if(historicInfo[i]!=(byte)value) {
									break;
								}
							}
							historicInfo=null;
							lastBookmark=currentFile.getFilePointer();
							return;
						}
					}
					
					/* We are testing condition 3 here */
					long matchedChars=0;
					int historicInfoOffset=fileFooterSize;
					currentOffset-=fileFooterSize;
					long currentCurrentFileOffset=currentOffset;
					do {
						currentFile.seek(--currentCurrentFileOffset);
						if(historicInfo[historicInfoOffset++]==currentFile.read()) {
							matchedChars++;
						}
						else {
							currentOffset--;
							currentCurrentFileOffset=currentOffset;
							historicInfoOffset=fileFooterSize;
							matchedChars=0;
						}
					}while(matchedChars<charactersToCompare-fileFooterSize && currentOffset>0);
					
					/* if we have satified our file offset comparison we have condition 3, otherwise condition 4. */
					if(matchedChars==charactersToCompare-fileFooterSize) {
						/* The footer information is gone so we need to determine where to start from.  This
						 * can be done by traversing the historic info until we get a difference.
						 */
						currentFile.seek(currentOffset);
						for(int i=fileFooterSize-1; i>=0; i--) {	
							int value=currentFile.read();
							if(historicInfo[i]!=(byte)value) {
								historicInfo=null;
								lastBookmark=currentFile.getFilePointer();
								return;
							}
						}
						historicInfo=null;
						lastBookmark=currentOffset+fileFooterSize;
						return;
					}
					else {
						historicInfo=null;
						lastBookmark=0;
						return;
					}
				}
				catch (IOException e) {

				    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();
				    event.setMsg(e.toString());
				    event.setSeverity(CommonBaseEvent.SEVERITY_WARNING);

				    log(event);
				}
			}
		}
		
		/**
		 * close file
		 */
		public void close()	{
			try {
				if (currentFile != null) {
					currentFile.close();
				}
			}
			catch (Exception e) {
			}
			currentFile = null;
		}
		/**
		 *
		 * @return
		 * @throws IOException
		 */
		public String readLine() throws IOException
		{
			boolean endOfFile=false;
			
			if (currentFile != null) {
				String last = null;
				currentFile.seek(lastBookmark);
				
				/* read a line */
				if (currentFile != null) {
					try {
						/* RKD:  START of workaround for RandomAccessFile.readLine()
						 * There are issues with using the RandomAccessFile.readLine() method
						 * as it takes each byte in a line and maps that to a character.  When dealing
						 * with DBCS character sets it fails.  Additionally, the readLine() method
						 * does not recognize end of line charaters on EBCIDIC machines.
						 */
						
						/* bugzilla 78521 - use the same code here as in 
						 *	org.eclipse.hyades.logging.parsers.Parser.readLine()
		                 */
						
			            //Create a byte buffer with a default size of 2048 bytes (e.g. upper-bound for expected line length) which may need to be resized:
			            byte[] buffer = new byte[initialBufferSize];
			            
			            //The current size or number of bytes in the byte buffer:
			            int bufferSize = 0;

			            //A single byte from the file:
			            int singleByte = 0;
			            
			            //Iterate the remaining bytes in the file until the EOF (e.g. -1) or EOL character(s) (e.g. '\n', '\r' or '\r\n'):
			            while((singleByte = currentFile.read()) != -1) {

							// Check to see if we have hit the end of line character
							if (singleByte == EOL_LAST_CHAR) {
								if(EOL_CHARS.length()>1 && bufferSize > 0) {
									/* Handle case of line separator greater then 1 character in length */
									for(int k=EOL_CHARS.length()-2; k>=0;k--) {
										if((byte)EOL_CHARS.charAt(k) == buffer[bufferSize-1]) {
											/* Remove the line separator characters from the saved line */
											buffer[--bufferSize] = 0;
										}
										else {
											/* Stop procesing end of line characters if there are no more */
											break;
										}
									}
								}
								/* bugzilla 70772 - sometimes on UNIX machines lines end with \r\n
								 * so both characters need to be stripped
								 */
								else if (bufferSize > 0 && EOL_LAST_CHAR == '\n' && buffer[bufferSize-1] == '\r'){
									buffer[--bufferSize] = 0;
								}
								break;
							}

			                //If the byte buffer has overflowed, resize by another initial buffer size:
			                if(bufferSize == buffer.length){
			                    
			                    byte[] tempBuffer = new byte[buffer.length + initialBufferSize];
			                    
			                    System.arraycopy(buffer,0,tempBuffer,0,bufferSize);
			                    
			                    buffer = tempBuffer;                                       
			                }
			                
			                //Valid downcast (int --> byte) since the RandomAccessFile.read() API returns an integer in the range 0 to 255 (0x00 - 0x0ff). 
			                buffer[bufferSize++] = ((byte)(singleByte));
			            }
			            
			            if (singleByte == -1) {
			            	endOfFile = true;
			            }
			            
			            // If we've reached the end of file then return null
			            if (bufferSize == 0 && endOfFile) {
			            	last = null;
			            }
			            else {
			            	// Else If a line of data was read or a blank line was read Then return the line
			                // Convert the byte buffer to a string based on the default charset:
			                last = new String(buffer, 0, bufferSize);
			            }
			            
			            /* End of bugzilla 78521 */
						/* RKD:  END of workaround for RandomAccessFile.readLine() */
			            
					}
					catch (EOFException e) {
						last=null;
					}
					catch (NullPointerException e) {
						last = null;
					}
					
					if(endOfFile) {
						/* Load the end of the file into a buffer so we can compare later.  We actually load the
						 * data in the buffer in reverse order
						 */
						int bytesToSave=(int)Math.min(currentFile.length(), confidenceBufferSize);
						currentFile.seek(currentFile.length()-bytesToSave);
						historicInfo=new byte[bytesToSave];
						for(int i=bytesToSave-1; i>=0; i--) {
							historicInfo[i]=(byte)currentFile.read();
						}
						
						/* We are done with the file.  Close it */
						close();
					}
					else {
						lastBookmark = currentFile.getFilePointer();
					}
					
					return last;
				}
			}
			return null;
		}
	}
	//PROBLEM: Because some native platforms only provide limited buffer size for standard input streams, failure to read the input stream of the process may cause the process to block, and even deadlock.
	//SOLUTION: Process input streams must be read on seperate threads.
	//Class used to read input streams from the converter process on seperate threads.
	class ConverterOutputReader extends Thread
	{
		private BufferedReader reader;
		
		public ConverterOutputReader(InputStream inputStream)
		{
			reader = new BufferedReader(new InputStreamReader(inputStream));
		}
		public void run()
		{
			
			try
			{
				/* Read the input stream until the stream is closed (i.e. null termination): */
				while(reader.read()!=-1);
			}
			catch (IOException e)
			{
			}
			
			/* If both input streams (i.e. stdout and stderr) are closed, notify the waiting parent thread: */
			synchronized (currentThreadLock)
			{
				if (++childThreadsDone == 2)
					currentThreadLock.notify();
			}
		}
	}
}
