/*******************************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.model.ui;

import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.models.common.util.ICommonConstants;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.internal.model.EMFUtil;
import org.eclipse.jface.viewers.ILabelProvider;

/**
 * @author jgout
 *
 */
public class VolatileTypes {
	
	private static VolatileTypes instance;
	private Map volatileTypes;

	public static class VolatileElementInfo {
		private URI uri;
		private String type;
		
		public VolatileElementInfo(URI uri, String type) {
			this.uri = uri;
			this.type = type;
		}
		
		public String getType() {
			return type;
		}
		
		public URI getURI() {
			return uri;
		}
	}
	
	public static VolatileTypes getInstance() 
	{
		if (instance == null) 
			instance = new VolatileTypes();
		return instance;
	}

	private VolatileTypes () {
		volatileTypes = new HashMap();
		IExtensionPoint extPoint = Platform.getExtensionRegistry().getExtensionPoint(TestUIPlugin.getID() + ".testNavigatorVolatileType"); //$NON-NLS-1$
		if (extPoint != null) {
			IConfigurationElement[] members = extPoint.getConfigurationElements();
			for (int i = 0; i < members.length; i++) {
				if (members[i].getName() != null) {
					//- get extension of files we want to make visible
					IConfigurationElement element = members[i];
					String typeName = element.getAttribute("targetType"); //$NON-NLS-1$
					if(typeName != null && typeName.length() != 0) {
						addVolatileType(typeName, element);
					}
				}
			}
		}		
	}

	/**
	 * Returns whether the given type has been registered as a volatile type using the the extension point <code>org.eclipse.hyades.test.ui.testNavigatorVolatileType</code>
	 * @param type the type to check
	 * @return true if the parameter is a volatile type and false otherwise
	 */
	public boolean isVolatile(String type) {
		return volatileTypes.containsKey(type);
	}
	
	/** Load the model stored inside the given EMF file and test whether this file is volatile or not. 
	 *  The returned value is composed by the array of root EMF objects of the file and the volatility boolean set  
	 * @param file a .testsuite or .execution file
	 * @return an EMFFileContent instance
	 * @see EMFFileContent
	 */
	public VolatileElementInfo getVolatileElementInfo(IFile file) {
		try {
			IMarker[] markers = file.findMarkers("org.eclipse.hyades.test.ui.volatileMarker", false, IResource.DEPTH_ZERO);//$NON-NLS-1$
			if(markers.length > 0) {
				//- marker exists retrieve info
				String uri = markers[0].getAttribute("uri", null);//$NON-NLS-1$
				String type = markers[0].getAttribute("type", null);//$NON-NLS-1$
				return new VolatileElementInfo(uri == null ? null : URI.createURI(uri), type);
			} else {
				//- marker does not exist, load model, check the volatility, build marker and return 
				EObject[] eObjects = EMFUtil.load(null, (IFile) file);
				String type = "";//$NON-NLS-1$
				
				String extension = file.getFileExtension();
				if (extension.equals(ICommonConstants.EXECUTION_FILE_EXTENSION)) {
					if(eObjects.length > 0 && eObjects[0] instanceof TPFExecutionResult) {
						type = ((TPFExecutionResult)eObjects[0]).getType();
					}
				} else if (extension.equals(ICommonConstants.TEST_SUITE_FILE_EXTENSION)) {
					if(eObjects.length > 0 && eObjects[0] instanceof TPFTestSuite) {
						type = ((TPFTestSuite)eObjects[0]).getType();
					}
				}
				boolean isVolatile = isVolatile(type);
				if(isVolatile) {
					class UpdateMarkerJob extends Job {
						private IFile _file;
						private EObject eObject = null;
						private String type = null;
						
						public UpdateMarkerJob(IFile _file, EObject eObject, String type) {
							super("update marker for file proxy"); //$NON-NLS-1$
							this._file = _file;
							this.eObject = eObject;
							this.type = type;
						}
						
						protected IStatus run(IProgressMonitor monitor) {
							//- before creating the marker remove old one if exists
							try {
								IMarker marker = _file.createMarker("org.eclipse.hyades.test.ui.volatileMarker");//$NON-NLS-1$
								URI uri = EcoreUtil.getURI(eObject);
								//- store the URI and the type of the EMF object 
								marker.setAttribute("uri", uri.toString());//$NON-NLS-1$
								marker.setAttribute("type", type);//$NON-NLS-1$
								return new Status(IStatus.OK, TestUIPlugin.getID(), 0,  "ok", null); //$NON-NLS-1$
							} catch (CoreException e) {
								TestUIPlugin.logError(e);
								return new Status(IStatus.ERROR, TestUIPlugin.getID(), 0,"unable to update marker for file: "+_file.getName() , e); //$NON-NLS-1$
							}
						}
					}
					Job job = new UpdateMarkerJob(file, eObjects[0], type);
					job.setRule(file);
					job.schedule();

					//- unload the loaded model
					eObjects[0].eResource().unload();
					URI uri = EcoreUtil.getURI(eObjects[0]);
					return new VolatileElementInfo(uri, type);
				} else {
					return new VolatileElementInfo(null, "");//$NON-NLS-1$
				}
			}
		} catch (CoreException e) {
			TestUIPlugin.logError(e);
			return new VolatileElementInfo(null, "");//$NON-NLS-1$
		}
	}
	
	/**
	 * @param typeName
	 */
	public void addVolatileType(String typeName, IConfigurationElement element) {
		if(!volatileTypes.containsKey(typeName)) {
			//- if the given type is not yet in the internal list, add it
			volatileTypes.put(typeName, element);
		}
	}

	/** Returns the associated label provider for the given volatile type of element found in the HTN
	 * 
	 * @param type the type of elements (this type shoudl been declared through the extension point <code>org.eclipse.hyades.test.ui.testNavigatorVolatileType</code>
	 * @return a provider for the given type <code>type</code> or <code>null</code> if the type does not exist (or problem has been encountered)
	 */
	public ILabelProvider getProvider(String type) {
		ILabelProvider provider = null;
		if(volatileTypes.containsKey(type)) {
			Object value = volatileTypes.get(type);
			if (value instanceof IConfigurationElement) {
				//- if the provider has not yet been loaded
				IConfigurationElement element = (IConfigurationElement) value;
				try {
					provider = (ILabelProvider)element.createExecutableExtension("provider"); //$NON-NLS-1$
					volatileTypes.put(type, provider);
				} catch (CoreException e) {
					provider = null;
				}
			} else {
				//- the provider has been loaded yet
				if (value instanceof ILabelProvider) {
					provider = (ILabelProvider) value;
				}
			}
		}
		return provider;
	}

	
}
