/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.ui.internal.wizard;

import java.io.InputStream;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;

import org.eclipse.hyades.test.ui.TestUI;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.ui.internal.util.UIUtil;
import org.eclipse.hyades.ui.internal.wizard.HyadesWizard;
import org.eclipse.hyades.ui.internal.wizard.LocationPage;
import org.eclipse.hyades.ui.report.IReportGenerator;

public class ReportWizard 
extends HyadesWizard
{
	protected static final String SET_LAST_REPORT = "LAST_REPORT"; //$NON-NLS-1$
	protected static final String SET_LAST_SELECTION = "LAST_FOLDER_SELECTION"; //$NON-NLS-1$
	
	private ReportWizardPage reportPage;
	private LocationPage locationPage;

	/**
	 * Constructor for ReportWizard
	 */
	public ReportWizard()
	{
		setWindowTitle(TestUIPlugin.getString("NEW_REPORT_TITLE")); //$NON-NLS-1$

		IDialogSettings dialogSettings = TestUIPlugin.getInstance().getDialogSettings();
		IDialogSettings settings = dialogSettings.getSection("ReportWizard"); //$NON-NLS-1$
		if(settings == null)
			settings = dialogSettings.addNewSection("ReportWizard"); //$NON-NLS-1$
		setDialogSettings(settings);
	}

	protected void initPages()
	{
		reportPage = new ReportWizardPage("report", getSelection()); //$NON-NLS-1$
		reportPage.setTitle(TestUIPlugin.getString("WIZ_TST_REPP_PG_TTL")); //$NON-NLS-1$
		reportPage.setDescription(TestUIPlugin.getString("WIZ_TST_REPP_PG_LOC_DSC")); //$NON-NLS-1$
		
		locationPage = new LocationPage("location", adjustLocation()); //$NON-NLS-1$
		locationPage.setTitle(TestUIPlugin.getString("WIZ_TST_REP_PG_TTL")); //$NON-NLS-1$
		locationPage.setDescription(TestUIPlugin.getString("WIZ_TST_REP_PG_LOC_DSC")); //$NON-NLS-1$
		String fileExtension = getFileExtension();
		if (fileExtension != null)
		{
			locationPage.setFileExtension(fileExtension);
			locationPage.setFileName("*." + fileExtension); //$NON-NLS-1$
		}
	}
	
	protected String getFileExtension()
	{
		return null;
	}

	protected IStructuredSelection adjustLocation()
	{
		IStructuredSelection structuredSelection = getSelection();
		if(getDialogSettings() != null)
		{
			String lastSelection = getDialogSettings().get(SET_LAST_SELECTION);
			if(lastSelection != null)
			{
				IResource resource = ResourcesPlugin.getWorkspace().getRoot().findMember(lastSelection);
				if((resource != null) && resource.exists())
					structuredSelection = new StructuredSelection(resource);
			}
		}
		
		return structuredSelection;
	}

	public void addPages()
	{
		addPage(reportPage);
		addPage(locationPage);
	}

	public boolean performFinish()
	{
		IFile file = null;
		IPath path = locationPage.getContainerFullPath();
		IResource res = createContainer(path);

		String fileExtension = TestUI.REPORT_FILE_EXTENSION;
		IReportGenerator generator = reportPage.getReportGenerator();
		try
		{
			if (generator != null && generator.getReportFileExtension(getSelection()) != null)
				fileExtension = generator.getReportFileExtension(getSelection());
		}
		catch (Exception e)
		{
			TestUIPlugin.logError(e);

			String error = e.getLocalizedMessage();
			if ((error == null) || (error.length() == 0))
				error = e.getMessage();
			if ((error == null) || (error.length() == 0))
				error = e.getClass().getName();

			MessageDialog.openError(getShell(), 
			        TestUIPlugin.getString("W_ERROR"),  //$NON-NLS-1$
			        TestUIPlugin.getString("GENERATE_ERROR_", error)); //$NON-NLS-1$
			return false;
		}

		path = new Path(locationPage.getFileName() + "." + fileExtension); //$NON-NLS-1$
		file = ((IContainer)res).getFile(path);
		if (file == null)
		{
			return false;
		}

		if (file.exists())
		{
			if (!MessageDialog.openQuestion(getShell(), 
			        TestUIPlugin.getString("W_QUESTION"),  //$NON-NLS-1$
			        TestUIPlugin.getString("FILE_EXISTS"))) //$NON-NLS-1$
				return false;
		}

		InputStream inputStream = null;
		try
		{
			if (generator != null)
			{
				inputStream = generator.generate(file, getSelection());
			}
		}
		catch (Exception e)
		{
			TestUIPlugin.logError(e);

			String error = e.getLocalizedMessage();
			if ((error == null) || (error.length() == 0))
				error = e.getMessage();
			if ((error == null) || (error.length() == 0))
				error = e.getClass().getName();

			MessageDialog.openError(getShell(), 
			        TestUIPlugin.getString("W_ERROR"),  //$NON-NLS-1$
			        TestUIPlugin.getString("GENERATE_ERROR_", error)); //$NON-NLS-1$
			return false;
		}

		if (inputStream == null)
		{
			MessageDialog.openError(getShell(), 
			        TestUIPlugin.getString("W_ERROR"),  //$NON-NLS-1$
			        TestUIPlugin.getString("REOPRT_NO_CONTENT_ERROR_")); //$NON-NLS-1$
			return false;
		}

		try
		{
			if (file.exists())
			{
				file.delete(true, true, null);
			}

			file.create(inputStream, true, null);
		}
		catch (CoreException e)
		{
			TestUIPlugin.logError(e);
			UIUtil.openSaveFileErrorDialog(getShell(), file.getFullPath().toString(), e);
			return false;
		}

		IPreferenceStore store = TestUIPlugin.getInstance().getPreferenceStore();
		store.setValue(TestUI.REPORT_OPEN_EDITOR, reportPage.openEditor());
		if (reportPage.openEditor())
			UIUtil.openEditor(file, null, false);

		getDialogSettings().put(SET_LAST_REPORT, reportPage.getReportDescription().getName());
		getDialogSettings().put(SET_LAST_SELECTION, res.getFullPath().toString());
		return true;
	}

	private IContainer createContainer(IPath path)
	{
		IContainer container = null;
		IWorkspace workbench = ResourcesPlugin.getWorkspace();
		int segCount = path.segmentCount();

		for (int idx = 0; idx < segCount; idx++)
		{
			if (idx == 0)
			{
				//project
				IProject project = workbench.getRoot().getProject(path.uptoSegment(idx + 1).toString());
				if (project == null || !project.exists())
				{
					//create the project

					try
					{
						project.create(null);
					}
					catch (Exception e)
					{
						TestUIPlugin.logError(e);
						e.printStackTrace();
					}
				}

				try
				{
					project.open(null);
				}
				catch (Exception e)
				{
				}
				container = project;
			}
			else // (idx > 1)
				{ //folder

				IFolder folder = workbench.getRoot().getFolder(path.uptoSegment(idx + 1));
				if (folder == null || !folder.exists())
				{
					//create the folder

					try
					{
						folder.create(false, true, null);
					}
					catch (Exception e)
					{
						TestUIPlugin.logError(e);
						e.printStackTrace();
					}
				}
				container = folder;
			}

		}

		try
		{

			container.getProject().refreshLocal(IResource.DEPTH_INFINITE, null);
		}
		catch (CoreException exc)
		{
			TestUIPlugin.logError(exc);
			exc.printStackTrace();
		}

		return container;
	}

}
