/**********************************************************************
 * Copyright (c) 2003-2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.views.util.internal;

import java.util.List;

import org.eclipse.hyades.models.hierarchy.TRCConfiguration;
import org.eclipse.hyades.models.hierarchy.TRCFilter;
import org.eclipse.hyades.models.trace.TRCClass;
import org.eclipse.hyades.models.trace.TRCMethod;
import org.eclipse.hyades.models.trace.TRCPackage;
import org.eclipse.hyades.models.trace.TRCSourceInfo;

/**
 * Here are some utilities related to Coverage feature
 */
public class CoverageUtil {

	/**
	 * @param e
	 * @return source location signature if any
	 */
	private static String getSourceSig(TRCSourceInfo e) {
		return (e != null ? e.getLocation()+":" : "");
	}
	
	/**
	 * @param e
	 * @return
	 */
	public static String getFullSig(TRCMethod e) {
		return  getSourceSig(e.getSourceInfo())+
				e.getLineNo()+":"+
				getSig(e.getDefiningClass())+"."+
				e.getName()+
				e.getSignature();
	}

	/**
	 * @param e
	 * @return
	 */
	public static String getFullSig(TRCClass e) {
		return  getSourceSig(e.getSourceInfo())+
				e.getLineNo()+":"+
				getSig(e);
	}

	/**
	 * @param e
	 * @return
	 */
	public static String getFullSig(TRCPackage e) {
		return e.getName();
	}
	
	/**
	 * @param e
	 * @return qualified name of the class
	 */
	private static String getSig(TRCClass e) {
		return (e.getPackage().getName() == null ? "" : (e.getPackage().getName()+"."))+
				e.getName();
	}
	
	private static class LocalBoolean
	{
		public boolean bool;
		public LocalBoolean(boolean b)
		{
			bool=b;
		}
	}

	/**
	 * This is for post-filtering needed because of the boundary classes collected even if excluded
	 * by the filter set.<br>
	 * The empty packages are filtered out and
	 * those listed in the active filters for this agent configuration
	 * @param pac
	 * @return
	 */
	public static boolean isFiltered(TRCPackage pac) {
		if (pac.getClasses() == null ||
			pac.getClasses().size() == 0) {
			//if empty package
			return true;
		}

		List cllist = pac.getClasses();
		int remaining = cllist.size();
		for(int idx=0;idx<remaining;idx++){
			Object obj = cllist.get(idx);
			if(obj instanceof TRCClass){
				if(CoverageUtil.isFiltered((TRCClass)obj)){
					remaining--;
				}
			}
		}
		if (remaining != 0) {
			return false;
		}
		
		Object[] conflist = pac.getProcess().getAgent().getAgentProxy().getConfigurations().toArray();
		int idxmax = conflist.length;
		for(int idx = 0; idx < idxmax; idx++){
			TRCConfiguration c = (TRCConfiguration) conflist[idx];
			if(!c.isActive())continue;
			
			Object[] filtlist = c.getFilters().toArray();
			for (int k = 0; k < filtlist.length; ++k) {
				TRCFilter f = (TRCFilter)filtlist[k];
				if(!f.getActive().booleanValue())
					continue;
				LocalBoolean exclude = new LocalBoolean(true);
				if(isFilteredPackage(f,pac,exclude)){
					if(exclude.bool)
						return true;
					else
						return false;
				}
			}
		}
		return false;
	}
	
	/**
	 * The empty classes are filtered out too
	 * @param c
	 * @return
	 */
	public static boolean isFiltered(TRCClass cl) {
		if (cl.getMethods() == null ||
			cl.getMethods().size() == 0) {
			//if empty class
			return true;
		}
		Object[] conflist = cl.getPackage().getProcess().getAgent().getAgentProxy().getConfigurations().toArray();
		int idxmax = conflist.length;
		for(int idx = 0; idx < idxmax; idx++){
			TRCConfiguration c = (TRCConfiguration) conflist[idx];
			if(!c.isActive())continue;
			
			Object[] filtlist = c.getFilters().toArray();
			for (int k = 0; k < filtlist.length; ++k) {
				TRCFilter f = (TRCFilter)filtlist[k];
				if(!f.getActive().booleanValue())
					continue;
				LocalBoolean exclude = new LocalBoolean(true);
				if(isFilteredClass(f,cl,exclude)){
					if(exclude.bool)
						return true;
					else
						return false;
				}
			}
		}
		return false;
	}

	/**
	 * @param filter
	 * @param pack
	 * @return 
	 */
	private static boolean isFilteredPackage(TRCFilter filter, TRCPackage pack, LocalBoolean exclude)
	{
		String pattern = filter.getPattern();
		String mode = filter.getMode();
		String type = filter.getType();
		if(type.equals(mode)){
			type = parseFilterPattern(pattern);
			pattern = externFilterPattern;
		}
		String methodPattern = filter.getOperation();
		if(methodPattern!=null && (methodPattern.length()>0 && !methodPattern.equals("*"))){
			return false;
		}
		String name = pack.getName();
		if(mode.equals("INCLUDE")){//$NON-NLS-1$
			exclude.bool = false;
		}else{
			exclude.bool = true;
		}
		if(type.equals("NONE")){
			if (name.equals(pattern)) {
				return true;
			}
		}else if(type.equals("PREFIX")){
			if (name.endsWith(pattern)) {
				return true;
			}
		}else if(type.equals("SUFFIX")){
			if (name.startsWith(pattern)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * @param filter
	 * @param cl
	 * @return 
	 */
	private static final String Path_Separator = ".";
	private static boolean isFilteredClass(TRCFilter filter, TRCClass cl, LocalBoolean exclude)
	{
		String pattern = filter.getPattern();
		String mode = filter.getMode();
		String type = filter.getType();
		if(type.equals(mode)){
			type = parseFilterPattern(pattern);
			pattern = externFilterPattern;
		}
		String methodPattern = filter.getOperation();
		if(methodPattern!=null && methodPattern.length()>0 && !methodPattern.equals("*")){
			return false;
		}
		String name = cl.getName();
		if(pattern.indexOf(Path_Separator)!=-1){
			name = cl.getPackage().getName() + Path_Separator + name;
		}
		if(mode.equals("INCLUDE")){//$NON-NLS-1$
			exclude.bool = false;
		}else{
			exclude.bool = true;
		}
		if(type.equals("NONE")){
			if (name.equals(pattern)) {
				return true;
			}
		}else if(type.equals("PREFIX")){
			if (name.endsWith(pattern)) {
				return true;
			}
		}else if(type.equals("SUFFIX")){
			if (name.startsWith(pattern)) {
				return true;
			}
		}
		return false;
	}
	
	private static String externFilterPattern=null;
	private static String parseFilterPattern(String filterPattern)
	{
	 String genericPattern = "NONE";
	 
	 if(filterPattern.length()==0){
	 	externFilterPattern = filterPattern;
	 	return genericPattern;
	 }

	 int p = filterPattern.indexOf('*');
	 if (p == 0)
	 {
	  genericPattern = "PREFIX";
	  externFilterPattern = filterPattern.substring(1);
	 }
	 else if (p == (filterPattern.length()-1))
	 {
	  genericPattern = "SUFFIX";
	  externFilterPattern = filterPattern.substring(0,(filterPattern.length()-1));
	 }else{
	 	externFilterPattern = filterPattern;
	 }
	 return genericPattern;
	}

}
