/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

import java.io.*;
import java.util.*;

import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Plugin;
import org.eclipse.hyades.probekit.internal.*;

/* 
 * Note related to NLS processing: this class runs as a stand-alone
 * Java application, so it can't use the usual resource bundle system.
 * 
 * TODO: improve error reporting, especially when the error occurs
 * deep in a directory hierarchy (which they all do, with Java classes and jars...).
 */ 

/**
 * This class exports an API that you can use to instrument class files and
 * jar files on disk. You pass a File object which refers to the saved BCI engine
 * script from the probe compiler, and more File objects referring to files
 * that you want to process. If the File object refers to a directory, all 
 * *.class and *.jar files in that directory and its subdirectories, 
 * recursively, will be processed. "Processing" a file means rewriting the 
 * class or jar file in place with an instrumented verison of itself.
 * <P>
 * This class also has a "main" so you can use it as a command-line program.
 * The first argument should be an engine script file to use, and the subsequent
 * arguments are things to process: class files, jar files, and directories.
 * When you use it this way, the command "probebciengine" should be in your PATH
 * or equivalent, so Runtime.exec() will find it without a path prefix.
 * <P>
 * The actual byte-code instrumentation is done with a native executable.
 * If you use this as a "main" program from the command line, the native
 * executable "probebciengine" should appear on your PATH, such that Runtime.exec
 * can find it. If you use this as a plug-in, the logic in SetExePathFromPlugin
 * will find the native executable relative to the plugin directory,
 * in os/$os$/$arch$/probebciengine (with a .exe suffix on Windows).
 * <P>
 * How to use this as a command-line tool:
 * <P>
 * The first argument must be an engine script file name.
 * Subsequent arguments are class files, jar files, and directories
 * to process.
 * <P>
 * If an argument is a directory name, begin recursive instrumentation
 * of everything in that directory.
 * <P>
 * If an argument is a class file, instrument it.
 * <P>
 * If an argument is a jar file, instrument its contents by extracting
 * the contents to a temporary location, instrumenting the class files,
 * and rebuilding the jar file. Before the jar file is rebuilt, the
 * MANIFEST.MF file is edited to remove any "Digest" lines, because the
 * hashes will have changed.
 * <P>
 * Classes and jar files are rewritten even if instrumentation resulted
 * in no change whatsoever. 
 */
public class ProbeInstrumenter {
	/**
	 * Main: accept command-line arguments and process them.
	 * 
	 * TODO: we don't check the already-instrumented state and leave instrumented classes alone.
	 * TODO: consider detecting instrumented files and reinstrumenting from *.bak files.
	 * 
	 * @param args the arguments: an engine script, and a list of class file,
	 * jar file, and directory names.
	 */
	public static void main(String[] args) {
		String scriptFileName = "";
		ProbeInstrumenter m = new ProbeInstrumenter();
		int arg_counter = 0;
		try {
			// Usage errors throw a string so we display usage.
			for ( ; arg_counter < args.length; arg_counter++) {
				if (args[arg_counter].equals("-v")) {
					m.setVerbose(true);
				}
				else if (args[arg_counter].equals("-h")) {
					throw new Exception("");
				}
				else if (args[arg_counter].startsWith("-")) {
					throw new Exception("Unrecognized option " + args[arg_counter]);
				}
				else break;
			}
			if (args.length - arg_counter < 2) {
				throw new Exception("Not enough arguments.");
			}
			
			scriptFileName = args[arg_counter];
			if (!new File(scriptFileName).exists()) {
				throw new Exception("First argument (script file) does not exist.");
			}
			arg_counter++;
		}
		catch (Exception s) {
			System.out.println("Usage error: " + s);
			System.out.println("Usage: ProbeInstrumenter engine_script item [item ...]");
			System.out.println("Where \"items\" may be class files, jar files, or directories.");
			System.out.println("Directories are processed recursively, instrumenting all jar and");
			System.out.println("class files found within.");
			return;
		}

		String[] items = new String[args.length - arg_counter];
	
		for (int i = 0; i < (args.length - arg_counter); i++) {
			items[i] = args[arg_counter + i];
		}

		try {
			// Set the exe path assuming we'll find it in the user's PATH
			m.SetExePath(INSERTION_EXECUTABLE_BASENAME);
			m.instrumentItems(new File(scriptFileName), items, true);
		}	
		catch (StaticProbeInstrumenterException e) {		
			System.err.println(e);
		}
	}

	boolean verbose = false;
	public void setVerbose(boolean f) {
		verbose = f;
	}
	
	String exePath = null;
	public void SetExePath(String s) {
		exePath = s;	
	}

	String newline = System.getProperty("line.separator");

	static final String COMPILER_PLUGIN_NAME = "org.eclipse.hyades.probekit";
	static final String INSERTION_EXECUTABLE_BASENAME = "probeinstrumenter";

	/**
	 * This function calls SetExePath using an absolute path name
	 * derived from the org.eclipse.hyades.probekit plugin location
	 * and a platform-specific subdirectory, plus a platform-specific
	 * suffix on the default executable name "probeinstrumenter."
	 * 
	 * @throws StaticProbeInstrumenterException if the target executable is not found.
	 */
	void SetExePathFromPlugin() throws StaticProbeInstrumenterException {
		SetExePath(ProbeInstrumenterInner.getExeName());
	}
	
	/**
	 * Inner class that computes the executable name relative to the plugin.
	 * This is an inner class because that way we can successfully 
	 * use this Java program outside Eclipse - Classes like Plugin and other
	 * references won't need to be resolvable.
	 */
	static class ProbeInstrumenterInner {
		static String getExeName() throws StaticProbeInstrumenterException {
			String engine_executable_name = "$os$/" + INSERTION_EXECUTABLE_BASENAME;
	
			// Platform-specific suffix processing goes here.
			if (org.eclipse.core.boot.BootLoader.getOS().equals("win32")) {
				engine_executable_name += ".exe";
			}
			Plugin compilerPlugin = Platform.getPlugin(COMPILER_PLUGIN_NAME);
			IPath relativePath = new Path(engine_executable_name);
			java.net.URL nativeInstrumnterURL = compilerPlugin.find(relativePath, null);
			if (nativeInstrumnterURL == null) {
				throw new StaticProbeInstrumenterException(
					"Native executable for instrumentation not found (tried " + engine_executable_name + " relative to " + compilerPlugin.getDescriptor().getUniqueIdentifier());
			}
			String exe_path = nativeInstrumnterURL.getFile();
			if (!(new File(exe_path).exists())) {
				throw new StaticProbeInstrumenterException(
					"Native executable for instrumentation not found (tried " + exe_path + ")");
			}
			return exe_path;
		}
	}
	
	static private final String[] JAR_FILENAME_EXTENSIONS = { ".jar", ".war", ".ear" };
	/**
	 * This is the main API method for this subsystem: use it to instrument
	 * class files and jar files, including recursive operations on directories.
	 * 
	 * Errors in one item don't stop processing
	 * of other items. After using this method, use getErrorStatus() and getErrorString()
	 * to find out how it went. 
	 * 
	 * @param scriptFile a File that refers to the engine script file to use
	 * @param items each string should be a class file name, a jar file name, 
	 * or a directory name.
	 * @param saveBackups true if you want this code to rename original class and jar files
	 * to *.bak - otherwise they'll be lost completely.
	 * @throws StaticProbeInstrumenterException if there are errors.
	 */
	public void instrumentItems(File i_scriptFile, String[] items, boolean saveBackups) throws StaticProbeInstrumenterException {
		scriptFile = i_scriptFile;
		String errorString = "";
		
		for (int i = 0; i < items.length; i++) {
			try {
				File f = new File(items[i]);
				if (!f.exists()) {
					throw new StaticProbeInstrumenterException("no such file or directory");
				}
				else if (endsWithIgnoreCase(items[i], ".class")) {
					processClassFile(f, saveBackups);
				}
				else if (endsWithIgnoreCase(items[i], JAR_FILENAME_EXTENSIONS)) {
					processJarFile(f, saveBackups);
				}
				else if (f.isDirectory()) {
					processDirectory(f, saveBackups);
				}
				else {
					// Neither a class file, nor a jar file, nor a directory.
					// Ignore it.
				}
			}
			catch (StaticProbeInstrumenterException e) {
				// Store a string version of the per-argument problem. 
				// We still go on to process other args.
				errorString = errorString + "Item \"" + items[i] + "\": " + e + newline;
			}
		}
		if (errorString.length() > 0) {
			throw new StaticProbeInstrumenterException(errorString);
		}
		
		if (verbose) System.out.println("Finished.");
	}
	
	static private boolean endsWithIgnoreCase(String s, String suffix) {
		int len = s.length();
		int suffixLen = suffix.length();
		if (len < suffixLen) 
			return false;
		
		if(s.substring(len - suffixLen).equalsIgnoreCase(suffix))
			return true;
		else 
			return false;
	}
	
	static private boolean endsWithIgnoreCase(String s, String[] suffixList) {
		for(int i = 0; i < suffixList.length; i++) {
			if (endsWithIgnoreCase(s, suffixList[i])) 
				return true;
		}
		return false;
	}

	/**
	 * The stored "engine script" File object. Its canonical name is passed to the
	 * native instrumentation engine.
	 */
	private File scriptFile;
	
	/**
	 * The exception type this subsystem can throw.
	 */
	public static class StaticProbeInstrumenterException extends Exception {
		public StaticProbeInstrumenterException(String m) {
			super(m);
		}
	}

	/**
	 * accumulates strings representing errors and warnings as items are processed.
	 * Will be the empty string if there's nothing to say.
	 */
	String errorString = "";
	
	/**
	 * The number of class files to batch up from a directory before running the instrumenter process.
	 */
	int classFilesBatchSize = 10;

	/**
	 * Process a batch of class files - this invokes the instrumentation engine just once,
	 * passing the whole batch.
	 * 
	 * @param files
	 * @param shouldSaveBakFile
	 */
	void processClassFiles(List files, boolean shouldSaveBakFile) throws StaticProbeInstrumenterException {
		if (verbose) System.out.println("(Class file list)");
		try {
			String[] args = new String[2 + files.size()];
	
			// If nobody's set exePath yet, set it from the plugin location.
			if (exePath == null) SetExePathFromPlugin();
	
			args[0] = exePath;
			args[1] = scriptFile.getCanonicalPath(); 
			for (int i = 0; i < files.size(); i++) {
				Object obj = files.get(i);
				if (obj instanceof File) {
					args[i+2] = ((File)obj).getAbsolutePath();
				}
				else {
					throw new StaticProbeInstrumenterException("Internal error: processClassFiles with a non-File list element");
				}
			}
	
			// The instrumenter creates a file with the input name plus .bci
			String[] outputStrings;
			try {
				outputStrings = executeCommandAndWait(args);
			} 
			catch (IOException e) {
				throw new StaticProbeInstrumenterException("Error executing instrumenter - not on path? " + newline + "Tried \"" + args[0] + "\"" + newline);
			}

			// Rename the original class file to *.bak
			// unless we've been told not to bother.
			// (We don't rename if we are in a jar file.)
			
			for (int j = 0; j < files.size(); j++) {
				File f = (File)(files.get(j));
				File bciFile = new File(f.getAbsolutePath() + ".bci");
				if (!bciFile.exists()) {
					throw new StaticProbeInstrumenterException(
						"Instrumenting " + f.getAbsolutePath() + " did not result in a *.bci file." + newline +
						"Instrumenter error output follows:" + newline +
						outputStrings[1]);
				}
				if (shouldSaveBakFile) {
					File bakFile = new File(f.getAbsolutePath() + ".bak");
					bakFile.delete();	// don't care if this fails
					if (!f.renameTo(bakFile)) {
						throw new StaticProbeInstrumenterException("Failed to rename " + f.getAbsolutePath() + " to *.bak");
					}
				}
				else {
					// Don't save a backup file, delete the original instead.
					f.delete();
				}
				if (!bciFile.renameTo(f)) {
					throw new StaticProbeInstrumenterException("Failed to rename " + bciFile.getAbsolutePath() + " to *.class");
				}
			}
		}
		catch(IOException e) {
			throw new StaticProbeInstrumenterException("I/O error in class file batch: " + e.getMessage() + newline);
		}
	}

	/**
	 * Instrument the contents of a class file. This is the only method in this class 
	 * that actually does any instrumentation work - all others eventually
	 * call this one to operate on real class files.
	 * 
	 * TODO: Don't instrument a class file if it's already instrumented, or throw an exception.
	 * Today, the command-line tool does not produce a .bci file for them, and that
	 * causes us to throw the "did not result in a *.bci file" exception.
	 * 
	 * @param f the file to instrument
	 * @throws ProbeException if anything goes wrong
	 */
	void processClassFile(File f, boolean shouldSaveBakFile) throws StaticProbeInstrumenterException {
		if (verbose) System.out.println("Class file " + f.getName());
		String canonicalPath = "";
		try {
			try {
				canonicalPath = f.getAbsolutePath();
				String[] args = new String[3];
	
				// If nobody's set exePath yet, set it from the plugin location.
				if (exePath == null) SetExePathFromPlugin();
	
				args[0] = exePath;
				args[1] = scriptFile.getAbsolutePath(); 
				args[2] = canonicalPath;
	
				// Probetest creates a file with the input name plus .bci
				String[] outputStrings = executeCommandAndWait(args);
	
				// Rename the original class file to *.bak
				// unless we've been told not to bother.
				// (We don't rename if we are in a jar file.)
				
				File bciFile = new File(f.getAbsolutePath() + ".bci");
				if (!bciFile.exists()) {
					throw new StaticProbeInstrumenterException(
						"Instrumenting " + f.getAbsolutePath() + " did not result in a *.bci file." +
						"Instrumenter error output follows:" + newline +
						outputStrings[1]);
				}
				if (shouldSaveBakFile) {
					File bakFile = new File(f.getAbsolutePath() + ".bak");
					bakFile.delete();	// don't care if this fails
					if (!f.renameTo(bakFile)) {
						throw new StaticProbeInstrumenterException("Failed to rename " + f.getAbsolutePath() + " to *.bak");
					}
				}
				else {
					// Don't save a backup file, delete the original instead.
					f.delete();
				}
				if (!bciFile.renameTo(f)) {
					throw new StaticProbeInstrumenterException("Failed to rename " + bciFile.getAbsolutePath() + " to *.class");
				}
			}
			catch (IOException e) {
				throw new StaticProbeInstrumenterException(e.getMessage());
			}
		}
		catch (StaticProbeInstrumenterException e) {
			throw new StaticProbeInstrumenterException("Error processing class file " + canonicalPath + ": " + e.getMessage() + newline);
		}
	}

	/**
	 * Instrument the contents of a jar file.
	 * 
	 * Jar files are processed by unpacking them into a temporary directory,
	 * processing the *.class files found inside, and packing them up again.
	 * The manifest file (if any) will be rewritten to remove any lines that
	 * contain "-Digest:" because we've (potentially)
	 * destroyed the checksum of every class file.
	 * 
	 * TODO: only remove Digest entries for class files that actually change.
	 *
	 * @param f the file to process
	 */
	void processJarFile(File f, boolean shouldSaveBakFile) throws StaticProbeInstrumenterException {
		if (verbose) System.out.println("Jar file " + f.getName());
		File tmpdir = null;

		try {
			// Create a temporary file, then remove it and use its name 
			// for a temporary directory.
			tmpdir = File.createTempFile("probetemp-", "");
			if (!tmpdir.delete() || !tmpdir.mkdir()) {
				throw new StaticProbeInstrumenterException("Some problem managing temporary/batch files" + newline);
			}

			// Unpack the jar file to the temporary directory.
			// Get its manifest, too.
			JarReader jr = new JarReader(f);
			java.util.jar.Manifest m = jr.getManifest();
			jr.extractAll(tmpdir);
			jr.close();
			
			// Process the temporary directory to instrument the class files
			processDirectory(tmpdir, false);

			// Strip the manifest of Digest entries
			if (m != null) {
				stripManifest(m);
			} 

			if (verbose) System.out.println("Repack " + f.getName());

			// Rename the original jar file to a backup name if we were asked to.
			// Otherwise delete the original so we can create a new one.
			if (shouldSaveBakFile) {
				File bakFile = new File(f.getCanonicalPath() + ".bak");
				bakFile.delete();
				if (bakFile.exists()) {
					throw new StaticProbeInstrumenterException("Unable to delete backup jar file: " + bakFile.getCanonicalPath() + newline);
				}
				if (!f.renameTo(bakFile)) {
					throw new StaticProbeInstrumenterException("Unable to rename jar file to *.bak: " + f.getCanonicalPath() + newline);
				}
			}
			else {
				f.delete();
				if (f.exists()) {
					throw new StaticProbeInstrumenterException("Unable to delete/overwrite jar file: " + f.getCanonicalPath() + newline);
				}
			}
			// Now jar everything up again.
			JarWriter jw = new JarWriter(f, m, null);
			jw.write(tmpdir, "", true);
			jw.close();

			return;
		}
		catch (IOException e) {
			throw new StaticProbeInstrumenterException("I/O Exception processing jar file " + f.getAbsolutePath() + ": " + e.getClass() + e.getMessage() + newline);
		}
		finally {
			// Delete the temporary directory
			recursiveDelete(tmpdir);
		}
	}

	/**
	 * Process a whole directory, recursively.
	 * 
	 * For each class file in the directory, call processClassFile.
	 * For each jar file, call processJarFile.
	 * For each directory, call ProcessDirectory.
	 *  
	 * @param f the File representing the directory to process.
	 * @throws ProbeException
	 */
	void processDirectory(File f, boolean shouldSaveBakFile) throws StaticProbeInstrumenterException {
		if (verbose) System.out.println("Directory " + f.getName());
		if (!f.isDirectory()) {
			throw new StaticProbeInstrumenterException("non-directory passed to processDirectory" + newline);
		}

		String accumulatedErrors = "";
		File[] contents = f.listFiles();
		List classFilesList = new Vector();
		for (int i = 0; i < contents.length; i++) {
			// use a try block to store problem reports
			// while continuing to process other files
			try {
				if (contents[i].isDirectory()) {
					processDirectory(contents[i], shouldSaveBakFile);
				}
				else if (contents[i].getName().endsWith(".class")) {
					classFilesList.add(contents[i]);
				}
				else if (contents[i].getName().endsWith(".jar")) {
					processJarFile(contents[i], shouldSaveBakFile);
				}
				
				// If we just handled the last file or the class file list
				// is big enough, process it.
				if ((i == (contents.length - 1)) || 
					(classFilesList.size() == classFilesBatchSize)) 
				{
					processClassFiles(classFilesList, shouldSaveBakFile);
					classFilesList.clear();
				}
			}
			catch (StaticProbeInstrumenterException e) {
				String newline = System.getProperty("line.separator");
				accumulatedErrors += "Error processing directory " + 
					f.getAbsolutePath() + ": " + newline + e + newline;
			}
		}
		if (accumulatedErrors.length() > 0) {
			throw new StaticProbeInstrumenterException(accumulatedErrors);
		}
	}

	/**
	 * Function to recursively delete a file. This means if it's
	 * a directory we'll delete children first, then the directory.
	 * Even in the face of errors, deletes everything that can be deleted.
	 * 
	 * @param f represents the file to delete
	 * @return true on success, false for failure
	 */
	boolean recursiveDelete(File f) {
		if (f.isDirectory()) {
			// We don't need to track success of inner deletes,
			// because if any fail then the top-level delete will fail
			// and cause us to return false.
			File[] contents = f.listFiles();
			for (int i = 0; i < contents.length; i++) {
				recursiveDelete(contents[i]);
			}
		}
		return f.delete();
	}

	/**
	 * Strips a manifest of any "-Digest" attributes. We do this because
	 * we've changed the signatures of classes that we did instrumentation on.
	 * 
	 * @param m the manifest to strip
	 */

	static void stripManifest(java.util.jar.Manifest m) {
		Map entries = m.getEntries();
		if (entries != null) {
			Iterator iter = entries.entrySet().iterator();
			while (iter.hasNext()) {
				Object o = iter.next();
				java.util.Map.Entry e = (java.util.Map.Entry)o;
				java.util.jar.Attributes attr = (java.util.jar.Attributes)e.getValue();
				if (attr != null) {
					Set keys = attr.keySet();
					Iterator keyIter = keys.iterator();
					Set keysToRemove = new HashSet();
					while (keyIter.hasNext()) {
						Object keyObj = keyIter.next();
						String keyName = keyObj.toString();
						if (keyName.endsWith("-Digest")) {
							// Add this to the list of objects to remove from the map
							// (we can't do it now because we'd destroy the key set iterator)
							keysToRemove.add(keyName);
						}
					}

					// Now remove the digest strings we found from the attribute map
					Iterator keysToRemoveIterator = keysToRemove.iterator();
					while (keysToRemoveIterator.hasNext()) {
						String keyName = (String)keysToRemoveIterator.next();
						java.util.jar.Attributes.Name attrName = new java.util.jar.Attributes.Name(keyName); 
						attr.remove(attrName);
					}
				}
			}
		}
	}

	/**
	 * Class to capture the contents of a stream into a String.
	 * Run it as a thread to soak up stdout and stderr of a child process.
	 */	
	static class StreamCapture extends Thread {
		InputStream stream;
		StringBuffer capture = new StringBuffer();
		static final String newline = System.getProperty("line.separator");
			
		public StreamCapture(InputStream s) {
			stream = s;
		}
			
		public void run() {
			BufferedReader breader = new BufferedReader(new InputStreamReader(stream));
			String line = null;
			boolean anyExceptions = false;
			do {
				try {
					line = breader.readLine();
					if (line != null) {
						capture.append(line);
						capture.append(newline);
					}
				}
				catch (IOException e) {
					capture.append("IO Exception in stream reader:");
					capture.append(newline);
					capture.append(e.getMessage());
					capture.append(newline);
					
					// break this loop on the second exception we get
					if (anyExceptions) break;
					anyExceptions = true;
				}
			} while (line != null);
		}
			
		public StringBuffer getCapture() {
			return capture;
		}
	}

	/**
	 * Function that executes a command with Runtime.exec 
	 * and doesn't return until it exits.
	 * Consumes and discards the standard output, standard error, and exit code.
	 * On Windows at least, this can start a batch file as well as native executables.
	 * 
	 * @param args the argument array to execute. First element is the executable name.
	 * @throws IOException when something goes wrong.
	 */
	private String[] executeCommandAndWait(String[] args) throws IOException {
		Runtime rt = Runtime.getRuntime();
		Process proc = rt.exec(args);

		// Set up threads to read the stdout and stderr of the process
		StreamCapture stdoutCapture = new StreamCapture(proc.getInputStream());
		stdoutCapture.start();
		StreamCapture stderrCapture = new StreamCapture(proc.getErrorStream());            
		stderrCapture.start();
 
		// Wait for the process to exit.
		// Loop if the waitFor gets interrupted by something.
		boolean interrupted = false;
		int exitCode;
		do {
			try {
				exitCode = proc.waitFor();
			}
			catch (InterruptedException e) {
				interrupted = true;
			}
		} while (interrupted);	
		
		String out = stdoutCapture.getCapture().toString();
		String err = stderrCapture.getCapture().toString();
		return new String[] { out, err };
	}
}
