package org.eclipse.hyades.logging.core.tests;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLDecoder;
import java.util.Enumeration;
import java.util.Hashtable;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.w3c.dom.DOMException;
import org.w3c.dom.Document;
import org.xml.sax.ErrorHandler;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

import sun.net.www.ParseUtil;

/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * Singleton class representing various testing utilities.
 * <p>
 * 
 * @author Paul E. Slauenwhite
 * @version August 12, 2004
 * @since May 11, 2004
 */
public final class TestingUtilities {

    public final static String LINE_SEPARATOR = System.getProperty("line.separator");

    public final static String FILE_SEPARATOR = System.getProperty("file.separator");

    public final static char FILE_SEPARATOR_CHARACTER = FILE_SEPARATOR.charAt(0);

    public final static String PATH_SEPARATOR = System.getProperty("path.separator");

    private static TestingUtilities instance = null;

    //Current thread lock for spawning child threads:
    private Object currentThreadLock = new Object();

    //Counter for tracking completed child threads:
    private int childThreadsDone = 0;

    private TestingUtilities() {
    }

    public static TestingUtilities getInstance() {

        if (instance == null) {
            instance = new TestingUtilities();
        }

        return instance;
    }

    public synchronized boolean deleteDirectory(String directoryName) {
        return (new File(directoryName).delete());
    }

    public synchronized boolean deleteFile(String fileName) {
        return (new File(fileName).delete());
    }

    public synchronized boolean createDirectory(String directoryName) {
        return (new File(directoryName).mkdir());
    }

    public synchronized void createPropertiesFile(Hashtable properties, String fileName) throws IOException {

        PrintWriter writer = new PrintWriter(new FileOutputStream(new File(fileName)));
        Enumeration keys = properties.keys();
        Object key = null;

        while (keys.hasMoreElements()) {

            key = keys.nextElement();
            writer.print(key);
            writer.print("=");
            writer.println(properties.get(key));
        }

        writer.close();
    }

    public synchronized void createFile(String[] lines, String fileName) throws IOException {

        PrintWriter writer = new PrintWriter(new FileOutputStream(new File(fileName)));

        for (int counter = 0; counter < lines.length; counter++) {
            writer.println(lines[counter]);
        }

        writer.close();
    }

    public synchronized void createFile(String lines, File file) throws IOException {

        PrintWriter writer = new PrintWriter(new FileOutputStream(file));
        writer.println(lines);
        writer.close();
    }

    public synchronized String getFile(File file) throws IOException {

        FileReader reader = new FileReader(file);
        StringBuffer buffer = new StringBuffer();
        
        while(reader.ready()){
            buffer.append(((char)(reader.read())));
        }
        
        reader.close();
        
        return (buffer.toString());
    }

    public synchronized String executeProcess(String[] command) throws IOException {

        try {

            //Execute the command and capture any stdout and stderr output and
            // throw the appropriate exception:
            Process process = null;

            //Buffer to hold stdout output from the process:
            StringBuffer stdout = new StringBuffer();

            //Buffer to hold stderr output from the process:
            StringBuffer stderr = new StringBuffer();

            //PROBLEM: Because some native platforms only provide limited
            // buffer size for standard input streams, failure to read the input
            // stream of the process may cause the process to block, and even
            // deadlock.
            //SOLUTION: Process input streams must be read on separate threads.
            synchronized (currentThreadLock) {

                //Reset the counter for tracking completed child threads:
                childThreadsDone = 0;

                //Execute the command:
                process = Runtime.getRuntime().exec(command);

                //Capture any stdout and stderr output from the execution of
                // the command:
                new ProcessOutputReader(process.getErrorStream(), stderr).start();
                new ProcessOutputReader(process.getInputStream(), stdout).start();

                //ASSUMPTION: The process is only terminated after all its
                // streams (i.e. stdout and stderr) have terminated.
                //Wait for all of the stdout and stderr output to be read
                // before continuing:
                try {

                    //Wait no more than 5 minutes (300000 ms):
                    currentThreadLock.wait(300000);
                } catch (InterruptedException e) {
                }
            }

            //Capture the exit value (by convention, the value 0 indicates
            // normal termination) from the terminated process:
            int exitValue = -1;

            try {
                exitValue = process.waitFor();
            } catch (InterruptedException i) {

                //Destroy the process if it has not yet exited:
                process.destroy();
            }

            //Throw the appropriate exception if the execution of the command
            // has created any stderr output:
            if (stderr.length() > 0) { throw new IOException(stderr.toString().trim()); }

            //Throw an exception is the process has not exited or exits
            // abnormally (by convention, the value 0 indicates normal
            // termination):
            if (exitValue != 0) { throw new IOException("Process terminated abnormally."); }

            //Return any stdout output from the execution of the command:
            if (stdout.length() > 0) { return (stdout.toString().trim()); }
        } catch (Exception e) {
            throw new IOException(e.getMessage());
        }

        return null;
    }

    public synchronized String getClassLocation(Class theClass) {

        String classResource = "/".concat(theClass.getName().replace('.', '/')).concat(".class");

        //Attempt to find the actual local location of the class:
        //NOTE: The resource name must start with a forward slash and only
        // contain forward slashes.
        URL classResourceURL = theClass.getResource(classResource);

        //If the current class' class loader cannot find the actual local
        // location of the class,
        //consult the current thread's class loader to find the actual local
        // location of the class:
        //NOTE: The resource name must start with a forward slash and only
        // contain forward slashes.
        if (classResourceURL == null) {
            classResourceURL = Thread.currentThread().getContextClassLoader().getResource(classResource);
        }

        //If the actual local java.net.URL location of the class was found:
        if (classResourceURL != null) {

            //Retrieve the string representation of the actual local
            // java.net.URL location of the class:
            String classResourceURLString = classResourceURL.toExternalForm();

            //Find the possible index of the last exclamation mark character:
            int exclamationIndex = classResourceURLString.lastIndexOf('!');

            //If the actual local java.net.URL location of the class represents
            // a JAR file (JAR java.net.URL syntax: jar:<url>!/{entry}
            //or WebSphere Portal Server JAR java.net.URL syntax:
            // wsjar:<url>!/{entry}) which contains the class:
            if ((classResourceURLString.startsWith("jar:") || (classResourceURLString.startsWith("wsjar:"))) && (exclamationIndex != -1)) {

                try {
                    //Create a new java.net.URL object with the <url> portion
                    // of the current java.net.URL string:
                    classResourceURL = new URL(classResourceURLString.substring((classResourceURLString.indexOf(":") + 1), exclamationIndex));
                } catch (MalformedURLException e) {
                    //Use the 'best-guess' at where the class resides locally.
                }
            }

            //Retrieve the path string from the java.net.URL object:
            String classResourceURLPath = classResourceURL.getFile();

            try {
                //NOTE: The sun.net.www.ParseUtil.decode() API decodes spaces
                // (e.g. %20) and non-English characters:
                classResourceURLPath = ParseUtil.decode(classResourceURLPath);
            }

            //The sun.net.www.ParseUtil class is not in the CLASSPATH at
            // run-time:
            catch (Throwable t) {

                //NOTE: The java.net.URLDecoder.decode() API only decodes
                // spaces (e.g. %20) and %xy, where xy is the two-digit
                // hexadecimal representation of the lower 8-bits of the
                // character.upper 8-bits of the character:
                classResourceURLPath = URLDecoder.decode(classResourceURLPath);
            }

            //Retrieve the java.io.File object representing the local JAR or
            // class file:
            File classResourceFile = new File(classResourceURLPath);

            //If the local JAR or class file exists:
            if (classResourceFile.exists()) {

                //Retrieve the absolute path of the local JAR or class file:
                String classResourceFilePath = classResourceFile.getAbsolutePath();

                //Retrieve the relative path of the package and class file
                // (e.g. '/com/ibm/etools/logging/was/classResource.class' on
                // UNIX and '\com\ibm\etools\logging\was\classResource.class' on
                // Windows):
                String classResourcePackagePath = classResource.replace('/', FILE_SEPARATOR_CHARACTER);

                //Remove the trailing
                // '/com/ibm/etools/logging/was/classResource.class' on UNIX and
                // '\com\ibm\etools\logging\was\classResource.class' on Windows
                // from the absolute path of the local class file:
                if (classResourceFilePath.endsWith(classResourcePackagePath)) { return (classResourceFilePath.substring(0, classResourceFilePath.lastIndexOf(classResourcePackagePath))); }

                return classResourceFilePath;
            }
        }

        return null;
    }

    public synchronized Document createDocumentObjectModel(String xmlDocument) throws DOMException {

        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();

        try {

            DocumentBuilder documentBuilder = factory.newDocumentBuilder();

            documentBuilder.setErrorHandler(new ErrorHandler() {

                public void warning(SAXParseException exception) throws SAXException {
                    throw exception;
                }

                public void error(SAXParseException exception) throws SAXException {
                    throw exception;
                }

                public void fatalError(SAXParseException exception) throws SAXException {
                    throw exception;
                }
            });

            return (documentBuilder.parse(new ByteArrayInputStream(xmlDocument.getBytes())));
        } catch (Exception e) {
            throw new DOMException(((short) (1)), e.toString());
        }
    }
    
    //PROBLEM: Because some native platforms only provide limited buffer size
    // for standard input streams, failure to read the input stream of the
    // process may cause the process to block, and even deadlock.
    //SOLUTION: Process input streams must be read on separate threads.
    //Class used to read input streams from the process on separate threads.
    class ProcessOutputReader extends Thread {

        private BufferedReader reader;

        private StringBuffer output;

        private String line = "";

        public ProcessOutputReader(InputStream inputStream, StringBuffer output) {
            reader = new BufferedReader(new InputStreamReader(inputStream));
            this.output = output;
        }

        public void run() {

            try {

                //Read the input stream until the stream is closed (i.e. null
                // termination) and append the read string to the storage
                // buffer:
                while ((line = reader.readLine()) != null) {

                    output.append(line.trim());
                    output.append(LINE_SEPARATOR);
                }
            } catch (IOException e) {
            }

            //If both input streams (i.e. stdout and stderr) are closed, notify
            // the waiting parent thread:
            synchronized (currentThreadLock) {

                if (++childThreadsDone == 2) currentThreadLock.notify();
            }
        }
    }
}