/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.resources.database.internal.impl;
import java.sql.PreparedStatement;
import java.sql.Statement;
import java.util.List;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.hyades.models.hierarchy.util.PerfUtil;
import org.eclipse.hyades.resources.database.internal.DBMap;
import org.eclipse.hyades.resources.database.internal.dbmodel.Column;
import org.eclipse.hyades.resources.database.internal.dbmodel.Constraint;
import org.eclipse.hyades.resources.database.internal.dbmodel.Table;
import org.eclipse.hyades.resources.database.internal.extensions.DatabaseType;
import org.eclipse.hyades.resources.database.internal.extensions.JDBCHelper;
/**
 * This class sets a reference for the given source object, target object,
 * reference, and position.
 */
public class SetReferenceCommand extends DBCommand {
	protected DatabaseType type;
	protected Integer sourceId;
	protected Integer targetId;
	protected EReference reference;
	protected int position;
	private List targets;
	private List positions;
	private Column source;
	private Column target;
	public SetReferenceCommand(JDBCHelper helper, DBMap map, DatabaseType dbType, Integer sourceId, EReference reference, Integer targetId, int position) {
		super(helper, map);
		this.sourceId = sourceId;
		this.reference = reference;
		this.targetId = targetId;
		this.position = position;
	}
	/**
	 * @param helper
	 * @param dbMap
	 * @param type2
	 * @param sources
	 * @param reference2
	 * @param targets
	 * @param positions
	 */
	public SetReferenceCommand(JDBCHelper helper, DBMap dbMap, DatabaseType type, Integer source, EReference reference, List targets, List positions) {
		super(helper, dbMap);
		this.sourceId = source;
		this.reference = reference;
		this.targets = targets;
		this.positions = positions;
	}
	public Object execute() throws Exception {
		if (positions != null && !positions.isEmpty()) {
			return doMultiplePositions();
		}
		SQLStatement sql = null;
		if (referenceInClassTable())
			sql = StatementFactory.INSTANCE.createUpdateReferenceStatement(dbType, dbMap, sourceId, reference, targetId, position);
		else
			sql = StatementFactory.INSTANCE.createInsertReferenceStatement(dbType, dbMap, sourceId, reference, targetId, position);
		Statement statement = helper.createStatement();
		String sqlString = sql.getStatement();
		if (debug)
			System.out.println(sqlString);
		helper.execute(statement, sqlString);
		statement.close();
		return null;
	}
	/**
	 * @return
	 */
	private Object doMultiplePositions() throws Exception {
		if (referenceInClassTable()) {
			doUpdates();
		} else {
			doInserts();
		}
		return null;
	}
	/**
	 *  
	 */
	private void doInserts() throws Exception {
		StringBuffer buffer = new StringBuffer();
		DBMap.ReferenceData data = (DBMap.ReferenceData) dbMap.getDBRepresentation(reference);
		setSourceAndTarget(data);
		boolean order = false;
		buffer.append("INSERT INTO ");
		buffer.append(addQuotes(data.getTable().getName()));
		buffer.append(" ");
		buffer.append("(");
		buffer.append(addQuotes(source.getName()));
		buffer.append(", ");
		buffer.append(addQuotes(target.getName()));
		if (data.getOrderColumn() != null) {
			buffer.append(", ");
			buffer.append(addQuotes(data.getOrderColumn().getName()));
		}
		buffer.append(") ");
		buffer.append(" VALUES (");
		buffer.append(sourceId);
		buffer.append(", ");
		buffer.append('?');
		if (data.getOrderColumn() != null) {
			buffer.append(", ");
			buffer.append(position);
			order = true;
		}
		buffer.append(")");
		PreparedStatement ps = helper.createPreparedStatement(buffer.toString());
		PerfUtil p = new PerfUtil("SetReferenceCommand.doInserts() sql="+buffer+", size="+positions.size(),true);
		for (int i = 0; i < positions.size(); i++) {
			ps.setInt(1, ((Integer) targets.get(i)).intValue());
			if (order) {
				ps.setInt(2, ((Integer) positions.get(i)).intValue());
			}
			ps.addBatch();
		}
		try {
			helper.executeBatchedPreparedStatement(ps);
		} catch (Exception e) {
			p.stopAndPrintStatus(e.getLocalizedMessage());
			throw e;
		}
		ps.close();
		p.stopAndPrintStatus();
	}
	/**
	 *  
	 */
	private void doUpdates() throws Exception {
		StringBuffer buffer = new StringBuffer();
		DBMap.ReferenceData data = (DBMap.ReferenceData) dbMap.getDBRepresentation(reference);
		setSourceAndTarget(data);
		boolean order = false;
		buffer.append("UPDATE ");
		buffer.append(addQuotes(data.getTable().getName()));
		buffer.append(" SET ");
		buffer.append(addQuotes(target.getName()));
		buffer.append(" = ");
		buffer.append('?');
		if (data.getOrderColumn() != null) {
			buffer.append(", ");
			buffer.append(addQuotes(data.getOrderColumn().getName()));
			buffer.append(" = ");
			buffer.append('?');
			order = true;
		}
		buffer.append(" WHERE ");
		buffer.append(addQuotes(source.getName()));
		buffer.append(" = ");
		if(sourceId==null)
			buffer.append('?');
		else
			buffer.append(sourceId);
		PreparedStatement ps = helper.createPreparedStatement(buffer.toString());
		PerfUtil p = new PerfUtil("SetReferenceCommand.doUpdates() sql="+buffer+", size="+positions.size(),true);
		for (int i = 0; i < positions.size(); i++) {
			if(sourceId==null)
				ps.setInt(1, ((Integer) targetId).intValue());
			else
				ps.setInt(1, ((Integer) targets.get(i)).intValue());
			if (order) {
				ps.setInt(2, ((Integer) positions.get(i)).intValue());
			}
			if(sourceId==null)
				ps.setInt(order ? 3 : 2, ((Integer) targets.get(i)).intValue());
//			else
//				ps.setInt(order ? 3 : 2, ((Integer) sourceId).intValue());
			ps.addBatch();
		}
		try {
			helper.executeBatchedPreparedStatement(ps);
		} catch (Exception e) {
			p.stopAndPrintStatus(e.getLocalizedMessage());
			throw e;
		}
		ps.close();
		p.stopAndPrintStatus();
	}
	protected void setSourceAndTarget(DBMap.ReferenceData data) {
		source = data.getSourceColumn();
		target = data.getTargetColumn();
		if (isPrimaryKey(target)) {
			Column temp = target;
			target = source;
			source = temp;
			Integer temp2 = targetId;
			targetId = sourceId;
			sourceId = temp2;
		}
	}
	protected boolean isPrimaryKey(Column column) {
		List constraints = column.getConstraints();
		for (int i = 0, l = constraints.size(); i < l; i++) {
			Constraint constraint = (Constraint) constraints.get(i);
			if (RDBHelper.PRIMARY_KEY_TYPE.equals(constraint.getType()))
				return true;
		}
		return false;
	}
	protected boolean referenceInClassTable() {
		DBMap.ReferenceData data = (DBMap.ReferenceData) dbMap.getDBRepresentation(reference);
		Table table = data.getTable();
		return dbMap.getClassTables().contains(table);
	}
} // SetReferenceCommand
