/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.resources.database.internal.impl;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EModelElement;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.hyades.models.hierarchy.extensions.LeftOperand;
import org.eclipse.hyades.models.hierarchy.extensions.OrderByElement;
import org.eclipse.hyades.models.hierarchy.extensions.RelationalOperators;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleBinaryExpression;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleSearchQuery;
import org.eclipse.hyades.models.hierarchy.util.PerfUtil;
import org.eclipse.hyades.resources.database.internal.DBCollectedExceptions;
import org.eclipse.hyades.resources.database.internal.DBMap;
import org.eclipse.hyades.resources.database.internal.extensions.DBCommandFactory;
import org.eclipse.hyades.resources.database.internal.extensions.JDBCHelper;
/**
 * @author slavescu
 */
public class SimpleSearchQueryStatement extends QueryStatement {
	protected JDBCHelper helper;
	protected SimpleSearchQuery query;
	protected Set processedAlready = new HashSet();
	protected boolean first = true;
	protected boolean all = false;
	protected Set fromSet = new HashSet();
	/**
	 * @param helper
	 * @param dbMap
	 * @param query
	 */
	public SimpleSearchQueryStatement(JDBCHelper helper, DBMap dbMap, SimpleSearchQuery query) {
		super(helper.getDatabaseType(), dbMap, null);
		this.helper = helper;
		this.query = query;
	}
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.resources.database.internal.impl.QueryStatement#createOrderClause()
	 */
	protected void createOrderClause() {
		first = true;
		processedAlready.clear();
		processOrderByExpressionsInOrderByClause();
		processedAlready.clear();
	}
	/**
	 *  
	 */
	protected void processOrderByExpressionsInOrderByClause() {
		for (Iterator iter = query.getOrderByExpresions().iterator(); iter.hasNext();) {
			OrderByElement oe = (OrderByElement) iter.next();
			if (oe.getOperand() == null || oe.getOperand().getFeature() == null || oe.getOperator() == null)
				continue;
			if (processedAlready.contains(oe.getOperand().getFeature()))
				continue;
			processedAlready.add(oe.getOperand().getFeature());
			if (first) {
				statement.append(" ORDER BY ");
				first = false;
			} else
				statement.append(", ");
			DBMap.AttributeData data = (DBMap.AttributeData) dbMap.getDBRepresentation(oe.getOperand().getFeature());
			statement.append(addQuotes(data.getValueColumn().getName()));
			statement.append(" ");
			statement.append(oe.getOperator().getName());
		}
	}
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.resources.database.internal.impl.QueryStatement#createSelectFrom()
	 */
	protected void createSelectFrom() {
		first = true;
		all = false;
		fromSet.clear();
		processedAlready.clear();
		processOutputElementsInSelectFrom();
		processOrderByInSelectFrom();
		processWhereClauseInSelectFrom();
		//	  	if (dbType.isOrderRequired()) {
		//	  	  statement.append(", ");
		//	  	  statement.append(addQuotes(order));
		//	    }
		//	  
		processedAlready.clear();
		first = true;
		processFromSet();
		processedAlready.clear();
	}
	/**
	 *  
	 */
	protected void processWhereClauseInSelectFrom() {
		String tableName;
		for (Iterator iter = query.getWhereExpressions().iterator(); iter.hasNext();) {
			SimpleBinaryExpression sbe = (SimpleBinaryExpression) iter.next();
			LeftOperand oe = sbe.getOperand();
			tableName = null;
			if (oe.getFeature() != null) {
				DBMap.AttributeData data = (DBMap.AttributeData) dbMap.getDBRepresentation(oe.getFeature());
				tableName = data.getValueColumn().getTable().getName();
			} else if (oe.getType() != null) {
				DBMap.ClassData data = (DBMap.ClassData) dbMap.getDBRepresentation(oe.getType());
				tableName = data.getTable().getName();
			}
			if (tableName != null)
				fromSet.add(tableName);
			tableName = null;
			if (sbe.getRightOperand() instanceof EModelElement) {
				EModelElement element = (EModelElement) sbe.getRightOperand();
				if (element instanceof EStructuralFeature) {
					if (element instanceof EAttribute) {
						DBMap.AttributeData data = (DBMap.AttributeData) dbMap.getDBRepresentation(element);
						tableName = data.getValueColumn().getTable().getName();
					} else {
						DBMap.ReferenceData data = (DBMap.ReferenceData) dbMap.getDBRepresentation(element);
						tableName = data.getTargetColumn().getTable().getName();
					}
				} else if (element instanceof EClass) {
					DBMap.ClassData data = (DBMap.ClassData) dbMap.getDBRepresentation(element);
					tableName = data.getTable().getName();
				}
			}
			if (tableName != null)
				fromSet.add(tableName);
		}
	}
	/**
	 *  
	 */
	protected void processFromSet() {
		for (Iterator iter = fromSet.iterator(); iter.hasNext();) {
			String tableName = (String) iter.next();
			if (processedAlready.contains(tableName))
				continue;
			if (first) {
				statement.append(" FROM ");
				first = false;
			} else
				statement.append(", ");
			statement.append(addQuotes(tableName));
		}
	}
	protected void processOutputElementsInSelectFrom() {
		for (Iterator iter = query.getOutputElements().iterator(); iter.hasNext();) {
			LeftOperand oe = (LeftOperand) iter.next();
			if (processedAlready.contains(getOperandType(oe))) {
				continue;
			}
			if (first) {
				statement.append("SELECT ");
				if (query.isDistinct()) {
					statement.append("DISTINCT ");
				}
				first = false;
			} else
				statement.append(", ");
			fromSet.add(addOperandName(oe));
		}
		if (first && query.isCount()) {
			statement.append("SELECT ");
			statement.append("COUNT(*) ");
			all = true;
			first = false;
		} else if (first) {
			statement.append("SELECT ");
			if (query.isDistinct()) {
				statement.append("DISTINCT ");
			}
			statement.append("* ");
			all = true;
			first = false;
		}
	}
	protected void processOrderByInSelectFrom() {
		if (!all && query.isDistinct()) {
			for (Iterator iter = query.getOrderByExpresions().iterator(); iter.hasNext();) {
				OrderByElement oe = (OrderByElement) iter.next();
				if (processedAlready.contains(getOperandType(oe.getOperand()))) {
					continue;
				}
				if (first) {
					statement.append("SELECT ");
					if (query.isDistinct()) {
						statement.append("DISTINCT ");
					}
					first = false;
				} else
					statement.append(", ");
				fromSet.add(addOperandName(oe.getOperand()));
			}
		}
	}
	/**
	 * @param columnName
	 * @param fromSet
	 * @param oe
	 * @return
	 */
	protected String addOperandName(LeftOperand oe) {
		String tableName = "";
		String columnName = "";
		if (oe.getFeature() != null) {
			processedAlready.add(oe.getFeature());
			DBMap.AttributeData data = (DBMap.AttributeData) dbMap.getDBRepresentation(oe.getFeature());
			tableName = data.getValueColumn().getTable().getName();
			columnName = data.getValueColumn().getName();
		} else if (oe.getType() != null) {
			processedAlready.add(oe.getType());
			DBMap.ClassData data = (DBMap.ClassData) dbMap.getDBRepresentation(oe.getType());
			tableName = data.getTable().getName();
			columnName = dbMap.getClassMetadata(oe.getType()).getPrimaryKey().getName();
		}
		if (oe.isCount() || query.isCount()) {
			statement.append("COUNT(");
		}
		statement.append(addQuotes(tableName));
		statement.append('.');
		statement.append(addQuotes(columnName));
		if (oe.isCount() || query.isCount()) {
			statement.append(")");
		}
		return tableName;
	}
	protected EObject getOperandType(LeftOperand oe) {
		if (oe.getFeature() != null) {
			return oe.getFeature();
		} else if (oe.getType() != null) {
			return oe.getType();
		}
		return null;
	}
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.resources.database.internal.impl.QueryStatement#createWhereClause(int)
	 */
	protected void createWhereClause(int begin) {
		first = true;
		processedAlready.clear();
		processBinaryExpressions();
		appendWhereClauseForSources();
		processedAlready.clear();
	}
	protected void processBinaryExpressions() {
		for (Iterator iter = query.getWhereExpressions().iterator(); iter.hasNext();) {
			SimpleBinaryExpression sbe = (SimpleBinaryExpression) iter.next();
			if (processedAlready.contains(sbe))
				continue;
			if (first) {
				statement.append(" WHERE ");
				first = false;
			} else
				statement.append(" AND ");
			addOperandName(sbe.getOperand());
			statement.append(" ");
			statement.append(RelationalOperators.getString(sbe.getOperator().getValue()));
			statement.append(" ");
			if (sbe.getRightOperand() instanceof EModelElement) {
				addRightOperand((EModelElement) sbe.getRightOperand());
			} else if (sbe.getRightOperand() instanceof String) {
				statement.append("'");
				statement.append(sbe.getRightOperand().toString());
				statement.append("'");
			} else if (sbe.getRightOperand() instanceof Boolean) {
				statement.append("'");
				statement.append(((Boolean) sbe.getRightOperand()).booleanValue() ? '1' : '0');
				statement.append("'");
			} else
				statement.append(sbe.getRightOperand().toString());
		}
	}
	/**
	 * @param classifier
	 */
	protected void addRightOperand(EModelElement classifier) {
		String tableName = "";
		String columnName = "";
		if (classifier instanceof EStructuralFeature) {
			if (classifier instanceof EAttribute) {
				DBMap.AttributeData data = (DBMap.AttributeData) dbMap.getDBRepresentation(classifier);
				tableName = data.getValueColumn().getTable().getName();
				columnName = data.getValueColumn().getName();
			} else {
				DBMap.ReferenceData data = (DBMap.ReferenceData) dbMap.getDBRepresentation(classifier);
				tableName = data.getTargetColumn().getTable().getName();
				columnName = data.getTargetColumn().getName();
			}
		} else if (classifier instanceof EClass) {
			DBMap.ClassData data = (DBMap.ClassData) dbMap.getDBRepresentation(classifier);
			tableName = data.getTable().getName();
			columnName = dbMap.getClassMetadata((EClass) classifier).getPrimaryKey().getName();
		}
		if (tableName.length() > 0 && columnName.length() > 0) {
			statement.append(addQuotes(tableName));
			statement.append('.');
			statement.append(addQuotes(columnName));
		}
	}
	/**
	 *  
	 */
	protected void appendWhereClauseForSources() {
		boolean firstSource = true;
		boolean andBracket = false;
		for (Iterator iter = query.getSources().iterator(); iter.hasNext();) {
			String source = (String) iter.next();
			if (first) {
				statement.append(" WHERE (");
				first = false;
				firstSource = false;
			} else if (firstSource) {
				statement.append(" AND (");
				if(query.getSources().size()>1){
					statement.append(" (");
					andBracket = true;
				}
				firstSource = false;
			} else
				statement.append(") OR (");
			String containerPath = getObjectCompressedURIFragment(source);
			addContainedIn(containerPath, isRootObject(source));
		}
		if(query.getSources().size()>0)
			statement.append(") ");
			
		if (andBracket)
			statement.append(") ");
	}
	/**
	 * @param source
	 * @return
	 */
	protected boolean isRootObject(String source) {
		int i = source.indexOf('#');
		if (i != -1) {
			if (source.indexOf(i + 1, '/') == -1) {
				return true;
			}
		}
		return false;
	}
	/**
	 * @param containerPath
	 */
	protected void addContainedIn(String containerPath, boolean rootObject) {
		boolean first = true;
		for (Iterator iter = query.getOutputElements().iterator(); iter.hasNext();) {
			LeftOperand output = (LeftOperand) iter.next();
			EClass eClass = null;
			if (output.getFeature() != null)
				eClass = output.getFeature().getEContainingClass();
			else
				eClass = output.getType();
			String tableName = dbMap.getClassMetadata(eClass).getTable().getName();
			if (first) {
				first = false;
			} else
				statement.append(" AND ");
			statement.append(addQuotes(tableName));
			statement.append(".");
			// TODO MS - fix this for any root object, not only for TRCAgent
			if (rootObject && tableName.equals("TRCAgent")) {
				statement.append(addQuotes(dbMap.getClassMetadata(eClass).getPrimaryKey().getName()));
				statement.append(" = ");
				statement.append(containerPath.substring(0, containerPath.indexOf('/')));
			} else {
				statement.append(addQuotes("p_p"));
				statement.append(" LIKE ");
				statement.append("'" + containerPath + "%'");
			}
		}
	}
	/**
	 * @param source
	 * @return
	 */
	protected String getObjectCompressedURIFragment(String source) {
		PerfUtil p = new PerfUtil("SimpleSearchQueryStatement.getObjectCompressedURIFragment()", true);
		DBCommand command = DBCommandFactory.INSTANCE.createGetCommpressedPathByURICommand(helper, dbMap, source);
		String s = "";
		try {
			s = (String) command.execute();
		} catch (Exception e) {
			p.stopAndPrintStatus(e.getLocalizedMessage());
			throw new DBCollectedExceptions(e);
		}
		p.stopAndPrintStatus(s);
		return s;
	}
}